#!/usr/bin/env python3

import argparse
import base64
import configparser
import copy
import json
import logging
import os
import random
import string
import sys
import tempfile
import uuid

from pyasn1.codec.der import decoder as asn1_decoder

from cryptography import x509
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives.serialization import Encoding, NoEncryption, PrivateFormat
from cryptography.hazmat.primitives.asymmetric import rsa
from cryptography.x509.oid import ExtensionOID

import names

import requests

from sseclient import SSEClient

# Get an instance of a logger
logger = logging.getLogger(__name__)

# pylint: disable=wrong-import-position
KEYPME_SERVER_CLIENT_DIR   = os.path.dirname(__file__)
CLIENT_DIR                 = os.path.dirname(KEYPME_SERVER_CLIENT_DIR)
KEYPME_SERVER_ROOT         = os.path.dirname(CLIENT_DIR)
KEYPME_SERVER_DIR          = os.path.join(KEYPME_SERVER_ROOT, "keypme_server")

sys.path.append(KEYPME_SERVER_DIR)

from keypme_certificate.certificate_request import create_DER_certificate_request, DER_certificate_to_str, DER_certificate_to_PEM  #pylint: disable=wrong-import-position
# pylint: enable=wrong-import-position

# 1.3.6.1.4.1.63847.1.3.1 is the OID for KeypMe Subject Alternative Name certificate name (63847 is labapart IANA's PEN)
KEYPME_CERTIFICATE_NAME_OID = x509.ObjectIdentifier("1.3.6.1.4.1.63847.1.3.1")

# Get user configuration path
configpath = os.path.join(
    os.environ.get('APPDATA') or
    os.environ.get('XDG_CONFIG_HOME') or
    os.path.join(os.environ['HOME'], '.config'),
    "keypme-server-client"
)
os.makedirs(configpath, exist_ok=True)

# Read user configuration
config = configparser.ConfigParser()
config.read(os.path.join(configpath, 'config.ini'))

#
# Save command line arguments into user configuration file
#
def save_config(request_session, args):
    if not config.has_section('Server'):
        config.add_section('Server')
    if args.server:
        config.set('Server', 'hostname', args.server)
    if args.ca_bundle_path:
        config.set('Server', 'ca_bundle_path', args.ca_bundle_path)

    if not config.has_section('Client'):
        config.add_section('Client')
    if args.client_certificate:
        config.set('Client', 'certificate', args.client_certificate)
    if args.client_key:
        config.set('Client', 'key', args.client_key)

    # Writing our configuration file to 'config.ini'
    config_path = os.path.join(configpath, 'config.ini')
    with open(config_path, 'w', encoding="utf-8") as configfile:
        config.write(configfile)
        logging.info("Save config in '%s'", config_path)

def generate_password(length):
    return ''.join(random.choice(string.ascii_uppercase + string.ascii_lowercase + string.digits) for _ in range(length))

#
# Create User
#
def create_user(request_session, args):
    if not args.firstname:
        args.firstname = input("Enter firstname:")
    if not args.lastname:
        args.lastname = input("Enter lastname:")

    data = {
        'firstname': args.firstname,
        'lastname': args.lastname,
        'email': args.email,
        'with_no_password_expiration': True,
        'is_smartcard_required': args.with_smartcard_required
    }

    if args.profile_pic:
        content_picture = args.profile_pic.read()
        data['photo'] = base64.b64encode(content_picture).decode("utf-8")

    r = request_session.post(args.server + '/api/v1/people/', data=data, timeout=30)
    if r.status_code != 200:
        logger.error("Failed to create user (status_code: %s)", r.status_code)
        sys.exit(1)

    result = r.json()
    if hasattr(args, 'show_json') and args.show_json:
        print(json.dumps(result, indent=4))
    return result

#
# List card templates
#
def list_card_template(request_session, args):
    r = request_session.get(args.server + '/api/v1/card_configuration/card-template/', timeout=30)
    if r.status_code != 200:
        logger.error("Failed to retrieve card template list (status_code: %s)", r.status_code)
        sys.exit(1)

    result = r.json()
    logger.info(json.dumps(result, indent=4))
    return result

#
# Show card template
#
def retrieve_card_template(request_session, args):
    r = request_session.get(args.server + f"/api/v1/card_configuration/card-template/{args.card_template}/card-provisioning-requirement-json/", timeout=30)
    if r.status_code != 200:
        logger.error("Failed to retrieve card template JSON (status_code: %s)", r.status_code)
        sys.exit(1)

    card_template_provisioning_requirenment_json = r.json()
    logger.info(json.dumps(card_template_provisioning_requirenment_json, indent=4))
    return card_template_provisioning_requirenment_json

#
# Retrieve User
#
def retrieve_user(request_session, args):
    # First, we check if the user already exists
    r = request_session.get(args.server + f"/api/v1/people/{args.username}/", timeout=30)
    if r.status_code != 200:
        logger.error("Failed to retrieve user (status_code: %s)", r.status_code)
        sys.exit(1)

    result = r.json()
    logger.info(json.dumps(result, indent=4))
    return result

def _card_template_get_id_from_name(request_session, args, card_template_name):
    r = request_session.get(args.server + '/api/v1/card_configuration/card-template/', timeout=30)
    if r.status_code != 200:
        logger.error("Failed to create user (status_code: %s)", r.status_code)
        sys.exit(1)

    card_template_ids = [ c for c in r.json() if c['name'] == card_template_name]
    if len(card_template_ids) == 0:
        logger.error("Could not find card template '%s' in %s", card_template_name, [ c['name'] for c in r.json() ])
        sys.exit(1)
    elif len(card_template_ids) > 1:
        logger.error("Too many card template with name '%s' (count=%d)", card_template_name, len(card_template_ids))
        sys.exit(1)

    card_template_id = card_template_ids[0]['id']
    return card_template_id

#
# Create User Certificate Request
#
def create_user_certificate_request(request_session, args, user_certificate_request_post_data=None):
    card_template_id = _card_template_get_id_from_name(request_session, args, args.card_template)

    if user_certificate_request_post_data is None:
        user_certificate_request_post_data = {}

    if hasattr(args, 'user_dn'):
        user_certificate_request_post_data['dn'] = args.user_dn
    user_certificate_request_post_data['card_template'] = card_template_id

    r = request_session.post(args.server + '/api/v1/user_certificate_request/', data = user_certificate_request_post_data,
        timeout=30)
    if r.status_code != 200:
        logger.error("Failed to create user certificate request (status_code: %s)", r.status_code)
        sys.exit(1)

    result = r.json()
    if hasattr(args, 'show_json') and args.show_json:
        print(json.dumps(result, indent=4))
    return result

#
# Delete User
#
def delete_user(request_session, args):
    data = {}
    if args.user_id: data['id'] = args.user_id
    if args.dn:      data['dn'] = args.dn

    r = request_session.post(args.server + '/api/v1/people/dev/delete_user', data = data, timeout=30)
    if r.status_code != 200:
        logger.error("Failed to delete user (status_code: %s)", r.status_code)
        sys.exit(1)

#
# Retrieve server profile
#
def server_profile(request_session, args):
    r = request_session.get(args.server + f"/api/v1/user_certificate_request/{args.user_id}/", timeout=30)
    if r.status_code != 200:
        logger.error("Failed to retrieve server profile (status_code: %s)", r.status_code)
        sys.exit(1)
    result = r.json()
    logger.debug(json.dumps(result, indent=4))
    return result

#
# Send CSR
#
def send_csr(request_session, args):
    response_json = {
        'dn': args.dn,
        "card": { "apps": [] }
    }

    # When card template is passed through argument, we parsed it
    if not hasattr(args, 'card_template') or args.card_template is None:
        server_profile_args = copy.copy(args)
        server_profile_args.user_id = args.user_id
        server_profile_result = server_profile(request_session, server_profile_args)

        args.card_template = server_profile_result['card']

    for app in args.card_template['apps']:
        response_app = {
            "type": app['type'],
            "keys": [],
            "fields": []
        }

        for key in app['keys']:
            if 'pki_info' in key:
                if 'csr' in key:
                    with open(key['csr'], 'rb') as csr_file:
                        ca_cert = x509.load_pem_x509_csr(csr_file.read(), default_backend())
                        req_der = ca_cert.public_bytes(Encoding.DER)
                elif hasattr(args, 'certificate_requests'):
                    csr_file = args.certificate_requests.pop(0)
                    logger.debug("Use certificate requests from command line argument '%s'", csr_file)
                    ca_cert = x509.load_pem_x509_csr(csr_file.read(), default_backend())
                    req_der = ca_cert.public_bytes(Encoding.DER)
                else:
                    req_der = create_DER_certificate_request(args.dn, algo=key['algo'], challenge_password=key['pki_info'].get('password'))

                req_der_base64 = base64.b64encode(req_der).decode("utf-8")

                response_app['keys'].append({
                    "key_id": key['key_id'],
                    "csr": req_der_base64
                })

        for field in app['fields']:
            if field['is_read_only'] == 0:
                value = ""
                if "value" in field:
                    with open(field['value'], 'rb') as value_file:
                        content_picture = value_file.read()
                        value = base64.b64encode(content_picture).decode("utf-8")

                response_app['fields'].append({
                    "type": field['type'],
                    "value": value
                })

        response_json['card']['apps'].append(response_app)

    r = request_session.post(args.server + f"/api/v1/user_certificate_request/{args.user_id}/response/",
                             json = response_json, timeout=30)
    if r.status_code != 200:
        logger.error("Failed to send response (status_code: %s)", r.status_code)
        sys.exit(1)
    result = r.json()
    if hasattr(args, 'show_json') and args.show_json:
        print(json.dumps(result, indent=4))
    return result

#
# Approve CSR
#
def approve_csr(request_session, args):
    r = request_session.post(args.server + f"/api/v1/user_certificate_request/{args.user_id}/approve/", data = { 'dn': args.dn }, timeout=30)
    if r.status_code != 200:
        logger.error("Failed to approve user (status_code: %s)", r.status_code)
        sys.exit(1)

def get_certificate_name(der_certificate):
    cert = x509.load_der_x509_certificate(der_certificate, default_backend())
    extension = cert.extensions.get_extension_for_oid(ExtensionOID.SUBJECT_ALTERNATIVE_NAME)
    OtherNames = extension.value.get_values_for_type(x509.OtherName)
    certificateNameValues = [v.value for v in OtherNames if v.type_id == KEYPME_CERTIFICATE_NAME_OID]
    # If there is no KeypMe certificate name OID in SAN, we search into all certificate extensions
    if not certificateNameValues:
        extension = cert.extensions.get_extension_for_oid(KEYPME_CERTIFICATE_NAME_OID)
        extension_value = extension.value

        # Decode the ASN.1 structure
        decoded_data, _ = asn1_decoder.decode(extension_value.value)
        # If it's nested (like in a context tag), you might need to access the inner content
        # The structure appears to be a context-specific tag [0] containing a PrintableString
        if hasattr(decoded_data, 'getComponent'):
            certificateNameValue = decoded_data.getComponent()
            print(f"Decoded string: {certificateNameValue}")
            raise RuntimeError("Do not know how to decode KeypMe certificate extension")
        else:
            certificateNameValue = decoded_data
    else:
        certificateNameValue = certificateNameValues[0].decode()

    if "PIV Authentication" in certificateNameValue:
        return "PIV Authentication"
    elif "PIV Digital signature" in certificateNameValue:
        return "PIV Digital signature"
    elif "Card Authentication" in certificateNameValue:
        return "Card Authentication"
    else:
        raise RuntimeError(certificateNameValue)

#
# Retrieve CRT
#
def retrieve_crt(request_session, args):
    # First, we check if the user already exists
    r = request_session.get(args.server + f"/api/v1/people/{args.username}/", timeout=30)
    if r.status_code == 200:
        json_response = r.json()
        logger.debug("- Retrieved CRT: %s", json_response)
        certificate_found = False
        if json_response['certificates']:
            for i, certificate in enumerate(json_response['certificates']):
                der_cert = bytes(base64.b64decode(certificate))

                certificate_name = None
                if hasattr(args, 'certificate_name'):
                    try:
                        certificate_name = get_certificate_name(der_cert)
                    except KeyError:
                        # Not a certificate name for this request
                        logger.error("No certificate name in this certificate")
                        continue
                else:
                    try:
                        certificate_name = get_certificate_name(der_cert)
                    except KeyError:
                        certificate_name = f"certificate-{i}"

                if args.crt_filepath_prefix:
                    certificate_filepath = f"{args.crt_filepath_prefix} - {certificate_name}.crt"
                    with open(certificate_filepath, "w", encoding='utf-8') as file:
                        pem_cert = DER_certificate_to_PEM(der_cert)
                        file.write(pem_cert)
                        logger.info("Certificate successfully written in %s", certificate_filepath)
                else:
                    logger.info(DER_certificate_to_str(der_cert))

                certificate_found = True

        # In case no certificate has been found, then we need to wait for certifcate
        if certificate_found:
            return
    else:
        logger.error("Failed to retrieve user '%s': %d", args.username, r.status_code)
        sys.exit(1)

    if args.request_id is None:
        logger.info("No '--args.request-id' has been given. Do not wait for an approval.")
        sys.exit(0)

    #
    # Wait for certificate
    #
    logger.info("Wait for user to be approved...")
    try:
        messages = SSEClient(args.server + f"/sse/user_certificate_request/{args.request_id}/", session=request_session)
        for msg in messages:
            if msg.event == 'certificate_request_approved':
                certificates = json.loads(msg.data)
                for certificate in certificates:
                    der_cert = bytes(base64.b64decode(certificate['certificate']))
                    if args.crt:
                        with open(args.crt, "w", encoding='utf-8') as file:
                            pem_cert = DER_certificate_to_PEM(der_cert)
                            file.write(pem_cert)
                            logger.info("Certificate successfully written in %s", args.crt)
                    else:
                        logger.info(DER_certificate_to_str(der_cert))
            elif msg.event == 'certificate_request_rejected':
                logger.info("Certificate rejected")
            else:
                logger.error("Event '%s' not supported", msg.event)
            break
    except KeyboardInterrupt:
        pass

#
# Modify user attributes
#
def modify_user(request_session, args):
    if args.enable is not None:
        r = request_session.post(args.server + f"/api/v1/people/{args.username}/set_disable/", { 'value': args.enable }, timeout=30)
        if r.status_code != 200:
            if args.enable:
                logger.error("Failed to enable %s", args.username)
            else:
                logger.error("Failed to disable %s", args.username)
            sys.exit(1)

    if args.enable_vpn is not None:
        r = request_session.post(args.server + f"/api/v1/people/{args.username}/set_vpn/", { 'value': args.enable_vpn }, timeout=30)
        if r.status_code != 200:
            if args.enable_vpn:
                logger.error("Failed to enable %s VPN", args.username)
            else:
                logger.error("Failed to disable %s VPN", args.username)
            sys.exit(1)

def _pki_server_get_scep_password(challenge_password_arg, server_name):
    for key_value in challenge_password_arg.split(';'):
        if key_value == '':
            continue
        key, value = key_value.split(':', maxsplit=1)
        if key == server_name:
            return value

    raise KeyError()

def does_certificate_request_need_approval(server_profile_json):
    for app in server_profile_json['card']['apps']:
        for key in app['keys']:
            if 'pki_info' in key:
                # At the moment, only KeypMe PKI server support approval
                if 'backend' not in key['pki_info']:
                    return True
                elif key['pki_info']['backend'] in ['SCEP', 'EJBCA']:
                    # SCEP and EJBCA do not require approval
                    continue

    return False

#
# Create User Certificate Request
#
def create_device_certificate_request(args, device_certificate_request_post_data: dict[str, str]) -> requests.Session:
    # Create certificate request for device
    device_private_key = rsa.generate_private_key(public_exponent=65537, key_size=2048,)
    device_uuid = device_certificate_request_post_data['uuid']
    certificate_request_der = create_DER_certificate_request(f"CN={device_uuid}", private_key = device_private_key)
    device_certificate_request_post_data['device_csr'] = base64.b64encode(certificate_request_der).decode()

    r = requests.post(args.server + '/api/v1/device/provision', data = device_certificate_request_post_data, timeout=30)
    if r.status_code != 200:
        logger.error("Failed to create device certificate request (status_code: %s)", r.status_code)
        sys.exit(1)

    result = r.json()

    #
    # Create Python requests session
    #
    # Write device certificate to temporary file
    device_certificate_der = base64.b64decode(result['certificate'])
    device_certificate = x509.load_der_x509_certificate(device_certificate_der)
    device_certificate_pem = device_certificate.public_bytes(Encoding.PEM)
    #TODO: Delete temporary file
    device_certificate_file = tempfile.NamedTemporaryFile(delete=False)  #pylint: disable=consider-using-with
    device_certificate_file.write(device_certificate_pem)
    device_certificate_file.close()

    device_key_pem = device_private_key.private_bytes(Encoding.PEM, PrivateFormat.TraditionalOpenSSL, NoEncryption())
    #TODO: Delete temporary file
    device_key_file = tempfile.NamedTemporaryFile(delete=False)  #pylint: disable=consider-using-with
    device_key_file.write(device_key_pem)
    device_key_file.close()

    device_request_session = requests.Session()
    device_request_session.cert = (device_certificate_file.name, device_key_file.name)
    if args.ca_bundle_path:
        device_request_session.verify = args.ca_bundle_path
    device_request_session.headers.update({'User-Agent': "KeypMe client"})
    return device_request_session


def test(request_session: requests.Session, args):
    # pylint: disable=too-many-locals

    if args.username:
        user_result = retrieve_user(request_session, args)
    else:
        if args.firstname:
            TEST_FIRSTNAME = args.firstname
        else:
            TEST_FIRSTNAME = names.get_first_name()
        if args.lastname:
            TEST_LASTNAME = args.lastname
        else:
            TEST_LASTNAME = names.get_last_name()

        logger.info("* Create user")
        create_args = copy.copy(args)
        create_args.firstname = TEST_FIRSTNAME
        create_args.lastname = TEST_LASTNAME
        create_args.email = "test-ci@labapart.com"
        create_args.profile_pic = None
        #create_args.email = f"{create_args.firstname}@{create_args.lastname}.com".lower()
        create_args.with_smartcard_required = True
        user_result = create_user(request_session, create_args)

    # If '--card-template' is passed then, we retrieve its ID
    if args.card_template:
        card_template_name = args.card_template
        card_template_id = _card_template_get_id_from_name(request_session, args, args.card_template)
    else:
        # Retrieve list of card templates
        card_template_list = list_card_template(request_session, args)
        if len(card_template_list) == 0:
            logger.error("No available card template")
            sys.exit(1)

        # We take the first card template
        card_template_name = card_template_list[0]['name']
        card_template_id = card_template_list[0]['id']

    # Retrieve card template provisioning requirement
    r = request_session.get(args.server + f"/api/v1/card_configuration/card-template/{card_template_id}/card-provisioning-requirement-json/", timeout=30)
    if r.status_code != 200:
        logger.error("Failed to retrieve card template JSON (status_code: %s)", r.status_code)
        sys.exit(1)

    card_template_provisioning_requirenment_json = r.json()
    logger.debug(json.dumps(card_template_provisioning_requirenment_json, indent=4))

    user_certificate_request_post_data = {}
    for k in card_template_provisioning_requirenment_json['hsms'].keys():
        user_certificate_request_post_data[k + '-so-pin'] = "12345678"
        user_certificate_request_post_data[k + '-user-pin'] = "12345678"

    # Use the same password for all certificate requests
    scep_password = generate_password(length=8)

    for k, v in card_template_provisioning_requirenment_json['keys_with_pki_server'].items():
        if 'pki_server' in v and 'backend' in v['pki_server']:
            pki_server = v['pki_server']
            if pki_server['backend'] == 'SCEP':
                print(f"\nPlease create 'End Entity' in '{v['pki_server']['name']}' https://ejbca.codeur.org:8443/ejbca/adminweb/ra/addendentity.xhtml")
                print(f"\t - username: '{user_result['username']}'")
                print(f"\t - CN: '{user_result['username']}'")
                print(f"\t - password: '{scep_password}'")
                print("\nPress a key when user is created.")
                input()
                user_certificate_request_post_data[k] = scep_password
            elif pki_server['backend'] == 'EJBCA':
                user_certificate_request_post_data[k] = {}
            else:
                raise RuntimeError(f"Unknown backend '{pki_server['backend']}' for key '{k}'")

    #for k in card_template_provisioning_requirenment_json['importable_keys'].keys():
    #    with open(TEST_VECTOR_SYMMETRIC_KEY_FILEPATH, 'rb') as f:
    #        user_certificate_request_post_data[k] = \
    #            SimpleUploadedFile(TEST_VECTOR_SYMMETRIC_KEY_FILEPATH, f.read(), content_type="application/octet-stream")

    logger.info("* Create user certificate request")
    create_user_certificate_request_args = copy.copy(args)
    create_user_certificate_request_args.user_dn = user_result['dn']
    create_user_certificate_request_args.card_template = card_template_name
    # If not specified, the challenge password will be None and will not be sent
    create_user_certificate_request_args.challenge_password = args.challenge_password
    user_certificate_request = create_user_certificate_request(request_session, create_user_certificate_request_args,
                                                               user_certificate_request_post_data)

    logger.info("* Create device certificate request")

    device_certificate_request_post_data = {
        'uuid': uuid.uuid4(),
        'user_request': user_certificate_request['id'],
        'type': 'smartphone',
        'manufacturer': 'Test Manufacturer',
        'model': 'Test Model',
        'device_name': 'Test Device',
        'tpm': 'Secure Element',
        'phone_imsi': '310260123456789',
        'operating_system': 'Android',
        'operating_system_version': '14'
    }
    device_session = create_device_certificate_request(args, device_certificate_request_post_data)

    logger.info("* Retrieve server profile")
    server_profile_args = copy.copy(args)
    server_profile_args.user_id = user_certificate_request['id']
    server_profile_result = server_profile(device_session, server_profile_args)
    profile_card_template = server_profile_result['card']

    logger.debug(json.dumps(server_profile_result, indent=4))

    logger.info("* Send card request")
    send_card_request_args = copy.copy(args)
    send_card_request_args.dn = user_result['dn']
    send_card_request_args.user_id = user_certificate_request['id']
    send_card_request_args.card_template = profile_card_template
    #send_card_request_args.csr_file = None
    #send_card_request_args.profile_pic = None
    #send_card_request_args.profile_pic = open("/home/olivier/Pictures/Windows-10-password-reset-min.jpg", 'rb')
    #send_card_request_args.base64_piv_chuid = None
    #send_card_request_args.pkcs11_serial_number = 'HELL0W0RLD'
    #send_card_request_args_certificate_requests = [ f"{hex(k['keyId'])}/{k['algo']}" for k in server_profile_result["card"]["keys"] ]
    #send_card_request_args.certificate_requests = send_card_request_args_certificate_requests
    #send_card_request_args.challenge_password = scep_password
    send_csr_result = send_csr(device_session, send_card_request_args)

    if args.no_approve:
        return

    if does_certificate_request_need_approval(server_profile_result):
        # When using challenge password, it is the SCEP server that approves the CSR
        logger.info("* Approve CSR")
        approve_csr_args = copy.copy(args)
        approve_csr_args.user_id = user_certificate_request['id']
        approve_csr_args.dn = user_result['dn']
        approve_csr(request_session, approve_csr_args)

    if 'id' in send_csr_result:
        # Case the CSR has not been directly approved by (SCEP) server
        logger.info("* Retrieve CRT")
        retrieve_crt_args = copy.copy(args)
        retrieve_crt_args.username = user_result['username']
        crt_filepath_prefix = tempfile.TemporaryDirectory(prefix='tmp-test-crt', dir="/tmp") #pylint: disable=consider-using-with
        retrieve_crt_args.crt_filepath_prefix = crt_filepath_prefix.name
        retrieve_crt_args.request_id = send_csr_result['id']
        retrieve_crt(device_session, retrieve_crt_args)
    elif 'certificates' in send_csr_result:
        for certificate_base64 in send_csr_result['certificates']:
            certificate_der = bytes(base64.b64decode(certificate_base64))
            certificate_name = get_certificate_name(certificate_der)
            logger.info("Received certificate '%s'", certificate_name)

    if args.no_delete:
        return

    logger.info("* Delete user")
    delete_args = copy.copy(args)
    delete_args.user_id = user_certificate_request['id']
    delete_args.dn = user_result['dn']
    delete_user(request_session, delete_args)

def provision(request_session, args):
    user_result = retrieve_user(request_session, args)

    logger.info("* Create user certificate request")
    create_user_certificate_request_args = copy.copy(args)
    create_user_certificate_request_args.user_dn = user_result['dn']
    user_certificate_request = create_user_certificate_request(request_session, create_user_certificate_request_args)

    logger.info("* Retrieve server profile")
    server_profile_args = copy.copy(args)
    server_profile_args.user_id = user_certificate_request['id']
    server_profile_result = server_profile(request_session, server_profile_args)
    logger.info(json.dumps(server_profile_result, indent=4))

    logger.info("* send card request")
    send_card_request_args = copy.copy(args)
    send_card_request_args.dn = user_result['dn']
    send_card_request_args.user_id = user_certificate_request['id']
    send_card_request_args.card_template = server_profile_result['card']
    #send_card_request_args.profile_pic = None
    #send_card_request_args.base64_piv_chuid = None
    #send_card_request_args.pkcs11_serial_number = 'HELL0W0RLD'
    #send_card_request_args.certificate_name = args.certificate_name
    send_card_request_result = send_csr(request_session, send_card_request_args)
    print(send_card_request_result)

    logger.info("* Approve card request")
    approve_card_request_args = copy.copy(args)
    approve_card_request_args.user_id = user_certificate_request['id']
    approve_card_request_args.dn = user_result['dn']
    approve_csr(request_session, approve_card_request_args)

    logger.info("* Retrieve CRT")
    retrieve_crt_args = copy.copy(args)
    retrieve_crt_args.username = user_result['username']
    retrieve_crt_args.crt_filepath_prefix = args.crt_filepath_prefix
    retrieve_crt_args.certificate_name = args.certificate_name
    retrieve_crt_args.request_id = send_card_request_result['id']
    retrieve_crt(request_session, retrieve_crt_args)

parser = argparse.ArgumentParser(description='Register a user to KeypMe server')
subparsers = parser.add_subparsers()

parser.add_argument('--server', type=str, default=config.get('Server', 'hostname', fallback=None), help='KeypMe server hostname')
parser.add_argument('--client-certificate', type=str, default=config.get('Client', 'certificate', fallback=None), help='KeypMe client certificate')
parser.add_argument('--client-key', type=str, default=config.get('Client', 'key', fallback=None), help='KeypMe client key')
parser.add_argument('--ca-bundle-path', type=str, default=config.get('Server', 'ca_bundle_path', fallback=None), help='KeypMe CA Bundle')
parser.add_argument('--verbose', action='store_true', help="Enable verbose logging")

save_config_parser = subparsers.add_parser("save_config", help="Save command line arguments into user configuration")
save_config_parser.set_defaults(func=save_config)

create_user_parser = subparsers.add_parser("create_user", help="Create user on KeypMe server")
create_user_parser.add_argument('firstname', type=str, help='User firstname')
create_user_parser.add_argument('lastname', type=str, help='User lastname')
create_user_parser.add_argument('--email', type=str, help='User email')
create_user_parser.add_argument('--with-smartcard-required', action='store_true', help='Require smartcard for new user')
create_user_parser.add_argument('--profile-pic', type=argparse.FileType('rb'), help='Path to profile picture')
create_user_parser.add_argument('--show-json', action='store_true', help='Print JSON result')
create_user_parser.set_defaults(func=create_user)

list_card_template_parser = subparsers.add_parser("list_card_template", help="List card templates on KeypMe server")
list_card_template_parser.set_defaults(func=list_card_template)

card_template_parser = subparsers.add_parser("card_template", help="Show card template on KeypMe server")
card_template_parser.add_argument('card_template', type=int, help='Card Template ID')
card_template_parser.set_defaults(func=retrieve_card_template)

retrieve_user_parser = subparsers.add_parser("retrieve_user", help="Retrieve user on KeypMe server")
retrieve_user_parser.add_argument('username', type=str, help='User username')
retrieve_user_parser.set_defaults(func=retrieve_user)

create_user_certificate_request_parser = subparsers.add_parser("create_user_certificate_request", help="Create user certificate request on KeypMe server")
create_user_certificate_request_parser.add_argument('--card-template', type=str, required=True, help='Card Template name')
create_user_certificate_request_parser.add_argument('--show-json', action='store_true', help='Print JSON result')
create_user_certificate_request_parser.add_argument('user_dn', type=str, help='User DN')
create_user_certificate_request_parser.set_defaults(func=create_user_certificate_request)

delete_user_parser = subparsers.add_parser("delete_user", help="Delete user on KeypMe server")
delete_user_parser.add_argument('--user-id', type=str, help="ID returned by 'create_user_certificate_request' operation")
delete_user_parser.add_argument('--dn', type=str, help="DN returned by 'create_user' operation")
delete_user_parser.set_defaults(func=delete_user)

server_profile_parser = subparsers.add_parser("server_profile", help="Retrieve server profile")
server_profile_parser.add_argument('--user-id', type=str, required=True, help="ID returned by 'create_user_certificate_request' operation")
server_profile_parser.set_defaults(func=server_profile)

send_csr_parser = subparsers.add_parser("send_csr", help="Send user certificate request back to server")
send_csr_parser.add_argument('--user-id', type=str, required=True, help="ID returned by 'create_user_certificate_request' operation")
send_csr_parser.add_argument('--dn', type=str, required=True, help="DN returned by 'create_user' operation")
send_csr_parser.add_argument('--profile-pic', type=argparse.FileType('rb'), help='Path to profile picture')
send_csr_parser.add_argument('--base64-piv-chuid', type=str, help="Base64 of PIV CHUID")
send_csr_parser.add_argument('--pkcs11-serial-number', type=str, help="PKCS11 Serial Number")
send_csr_parser.add_argument('--show-json', action='store_true', help='Print JSON result')
send_csr_parser.add_argument('certificate_requests', nargs='+', type=argparse.FileType('rb'), help="List of filepath of certificate request to send")
send_csr_parser.set_defaults(func=send_csr)

approve_csr_parser = subparsers.add_parser("approve_csr", help="Approve user certificate request")
approve_csr_parser.add_argument('--user-id', type=str, required=True, help="ID returned by 'create_user_certificate_request' operation")
approve_csr_parser.add_argument('--dn', type=str, required=True, help="DN returned by 'create_user' operation")
approve_csr_parser.set_defaults(func=approve_csr)

retrieve_crt_parser = subparsers.add_parser("retrieve_crt")
retrieve_crt_parser.add_argument('--username', type=str, required=True, help="Username of the user")
retrieve_crt_parser.add_argument('--request-id', type=str, help="ID returned by 'send_csr' operation")
retrieve_crt_parser.add_argument('--crt-filepath-prefix', type=str, help='Prefix for the approved certificates')
retrieve_crt_parser.set_defaults(func=retrieve_crt)

modify_user_parser = subparsers.add_parser("modify_user")
modify_user_parser.add_argument('--username', type=str, required=True, help="Username of the user")
modify_user_parser.add_argument('--enable', action='store_true', help="Enable user")
modify_user_parser.add_argument('--disable', dest='enable', action='store_false', help="Disable user")
modify_user_parser.set_defaults(enable=None)
modify_user_parser.add_argument('--enable-vpn', action='store_true', help="Enable user VPN")
modify_user_parser.add_argument('--disable-vpn', dest='enable_vpn', action='store_false', help="Disable user VPN")
modify_user_parser.set_defaults(enable_vpn=None)
modify_user_parser.set_defaults(func=modify_user)

test_parser = subparsers.add_parser("test", help="Create user + Send CSR + Approve user + Retrieve certificate + Delete user")
test_parser.add_argument('--card-template', type=str, help='Card Template name')
test_parser.add_argument('--username', type=str, help='Existing username which will have a new certificate')
test_parser.add_argument('--firstname', type=str, help='Firstname for the new user')
test_parser.add_argument('--lastname', type=str, help='Lastname for the new user')
test_parser.add_argument('--challenge-password', help='Specified challenge password for CSR. Format --challenge-password "eIDAS server:123456"')
test_parser.add_argument('--no-approve', action='store_true', help="Do not approve user (stop after creating and sending CSR)")
test_parser.add_argument('--no-delete', action='store_true', help="Do not delete user (stop after creating and sending CSR and approval)")
test_parser.set_defaults(func=test)

provision_user_parser = subparsers.add_parser("provision", help="Provision user from his/her certificate request")
provision_user_parser.add_argument('--username', type=str, required=True, help='User username')
provision_user_parser.add_argument('--certificate-name', type=str, required=True, help="Certificate name")
provision_user_parser.add_argument('--crt-filepath-prefix', type=str, required=True, help='Prefix for the approved certificates')
provision_user_parser.add_argument('certificate_requests', nargs='+', type=str, help="List of certificate request to send - format '9a(/RSA2048|:csr_filepath)'")
provision_user_parser.set_defaults(func=provision)

def main():
    #parser.add_argument('--wait-approval', action='store_true', help='Wait until the admin user has approved the account.')
    args = parser.parse_args()

    logging_level = logging.DEBUG if args.verbose else logging.INFO
    logging.basicConfig(level=logging_level, format="%(message)s")

    # Ensure the required arguments are passed
    if not hasattr(args, 'func') or args.func != save_config: #pylint: disable=comparison-with-callable
        if args.server is None:
            logger.error("Missing server parameters")
            parser.print_help()
            sys.exit(0)

    # Remove trailing '/' from server path
    if args.server and args.server[-1] == '/':
        args.server = args.server[:-1]

    #
    # Prepare 'Requests' session
    #
    request_session = requests.Session()
    request_session.cert = (args.client_certificate, args.client_key)
    if args.ca_bundle_path:
        request_session.verify = args.ca_bundle_path
    request_session.headers.update({
        'User-Agent': "KeypMe client",
        'Referer': args.server
    })

    args.func(request_session, args)

    # #
    # # Check arguments
    # #

    # # In case a CSR is passed, ensure the file exists before starting the process
    # if args.csr and not os.path.exists(args.csr):
    #     raise RuntimeError("Certificate Request file '" + args.csr + "' does not exist.")
    # # In case we specify the CRT, we assume we need to wait for approval.
    # if args.crt:
    #     args.wait_approval = True


    # if not args.wait_approval:
    #     logger.info(result)
    #     sys.exit(0)

if __name__ == "__main__":
    main()
