import enum

from cryptography import x509

class Iso7816Algo(enum.Enum):
    DES3 = 0x00
    RSA_1024 = 0x06
    RSA_2048 = 0x07
    RSA_3072 = 0x05
    AES_ECB_128 = 0x08
    AES_ECB_192 = 0x0A
    AES_ECB_256 = 0x0C
    ECC_256 = 0x11
    ECC_384 = 0x14

# Define the Key Usage flags as per RFC 5280
# The order doesn't strictly matter for the bitmask,
# but it's good practice to list them in the order they
# are typically presented or by their bit value (0-8).
KEY_USAGE_FLAGS = (
    ('digital_signature', 'Digital Signature'),
    ('non_repudiation', 'Non Repudiation'),
    ('key_encipherment', 'Key Encipherment'),
    ('data_encipherment', 'Data Encipherment'),
    ('key_agreement', 'Key Agreement'),
    ('key_cert_sign', 'Key Cert Sign'),
    ('crl_sign', 'CRL Sign'),
    ('encipher_only', 'Encipher Only'),
    ('decipher_only', 'Decipher Only'),
)

# Define common Extended Key Usage (EKU) flags (Object Identifiers - OIDs)
# You can add more as needed. These are common ones.
# The string value corresponds to the name you'll use in code.
# The display string is what the user sees.
EXTENDED_KEY_USAGE_FLAGS = (
    ('server_auth', 'TLS Web Server Authentication'),
    ('client_auth', 'TLS Web Client Authentication'),
    ('document_signing', 'Document Signing'),
    ('code_signing', 'Code Signing'),
    ('smartcard_logon', 'Smartcard Logon'),
    ('email_protection', 'S/MIME Email Protection'),
    ('time_stamping', 'Time Stamping'),
    ('ocsp_signing', 'OCSP Signing'),
    #('ipsec_end_system', 'IPsec End System'),
    #('ipsec_tunnel', 'IPsec Tunnel'),
    #('ipsec_user', 'IPsec User'),
    #('any_extended_key_usage', 'Any Extended Key Usage')
)

def _key_usage_to_bitmask(key_usage_definitions, key_usages):
    """
    Returns a bitmask for Extended Key Usage flags.
    Each flag corresponds to a bit in the mask.
    """
    bitmask = 0
    for i, (flag, _) in enumerate(key_usage_definitions):
        if flag in key_usages:
            bitmask |= (1 << i)
    return bitmask

def key_usage_bitmask(key_usages):
    return _key_usage_to_bitmask(KEY_USAGE_FLAGS, key_usages)

def has_key_usage(key_usages_bitset, key_usage_key:str):
    """
    Check if a specific key usage is present in the bitmask.
    :param key_usages_bitset: The bitmask representing key usages.
    :param key_usage_key: The key usage to check for.
    :return: True if the key usage is present, False otherwise.
    """
    if key_usage_key not in dict(KEY_USAGE_FLAGS):
        raise ValueError(f"Invalid key usage: '{key_usage_key}'")

    index = -1
    for i, (key, _) in enumerate(KEY_USAGE_FLAGS):
        if key == key_usage_key:
            index = i
            break

    return int(key_usages_bitset & (1 << index)) != 0

def extended_key_usage_bitmask(extended_key_usages):
    return _key_usage_to_bitmask(EXTENDED_KEY_USAGE_FLAGS, extended_key_usages)

def has_extended_key_usage(key_usages_bitset, key_usage_key:str):
    """
    Check if a specific key usage is present in the bitmask.
    :param key_usages_bitset: The bitmask representing key usages.
    :param key_usage_key: The key usage to check for.
    :return: True if the key usage is present, False otherwise.
    """
    if key_usage_key not in dict(EXTENDED_KEY_USAGE_FLAGS):
        raise ValueError(f"Invalid key usage: '{key_usage_key}'")

    index = -1
    for i, (key, _) in enumerate(EXTENDED_KEY_USAGE_FLAGS):
        if key == key_usage_key:
            index = i
            break

    return int(key_usages_bitset & (1 << index)) != 0

def DER_certificate_expiration_date(der_cert):
    cert = x509.load_der_x509_certificate(der_cert)
    return cert.not_valid_after
