import argparse
import logging
import random

import grpc

from .keypme_pkcs11_bridge_client import *
from . import keypme_pkcs11_bridge_pb2_grpc
from . import keypme_pkcs11_bridge_pb2

def _test_server_from_args(channel, args):
    if args.login_key:
        login_key = args.login_key.read()
    else:
        login_key = None

    info = keypme_pkcs11_client_test_server(channel, login_username = args.login_username, login_key = login_key)
    print(info)

def _algo_support_from_args(channel, args):
    if args.login_key:
        login_key = args.login_key.read()
    else:
        login_key = None

    crypto_support = keypme_pkcs11_client_crypto_support(channel, login_username = args.login_username, login_key = login_key)
    print(crypto_support)

def _create_token_from_args(channel, args):
    if args.login_key:
        login_key = args.login_key.read()
    else:
        login_key = None

    keypme_pkcs11_client_token_create(channel, args.token_name, args.so_pin, args.pin,
                                      login_username = args.login_username, login_key = login_key)

def _unittest_from_args(channel, args):
    token_label = "unittest"
    so_pin = "12345677"
    user_pin = "12345677"

    if args.login_key:
        login_key = args.login_key.read()
    else:
        login_key = None

    # Create or reinit token
    keypme_pkcs11_client_token_create(channel, token_label, so_pin, user_pin,
                                      login_username = args.login_username, login_key = login_key,
                                      reinit_if_exist = True)

    # Open session on token
    session = keypme_pkcs11_client_session_open(channel, token_label, user_pin)

    # #
    # # RSA
    # #
    # key_alias = "rsa_test"
    # rsa_bit_length = 2048
    # rsa_signature_mechanism = keypme_pkcs11_bridge_pb2.RSAPkcs1SignatureMechanism.SHA256_RSA_PKCS
    # data = b'Hello World!'

    # # Generate key
    # public_key = keypme_pkcs11_client_rsa_generate(channel, session, key_alias, rsa_bit_length)
    # if public_key is None:
    #     raise RuntimeError("Public key is not expected to be None")

    # # Sign some data
    # signature = keypme_pkcs11_client_rsa_sign(channel, session, key_alias, rsa_signature_mechanism, data)

    # # Verify signature
    # keypme_pkcs11_client_rsa_verify(channel, session, key_alias, rsa_signature_mechanism, data, signature)

    # try:
    #     # Generate a random signature to make sure it really works!
    #     random_signature_list = [random.randint(0, 255) for _ in range(len(signature))]
    #     random_signature = bytes(random_signature_list)
    #     keypme_pkcs11_client_rsa_verify(channel, session, key_alias, rsa_signature_mechanism, data, random_signature)

    #     raise RuntimeError("Signature verification should have failed")
    # except KeypMePkcs11InvalidSignatureException:
    #     pass

    #
    # ML-DSA
    #
    key_alias = "ml_dsa_test"
    #rsa_bit_length = 2048
    ml_dsa_signature_mechanism = keypme_pkcs11_bridge_pb2.MLDSASignatureMechanism.SIGNATURE_ML_DSA_44
    data = b'Hello World!'

    # Generate key
    public_key = keypme_pkcs11_client_ml_dsa_generate(channel, session, key_alias, MLDsaSecurityLevel.ML_DSA_SECURITY_LEVEL_2_4x4)
    if public_key is None:
        raise RuntimeError("Public key is not expected to be None")
    print(public_key)

    # Sign some data
    signature = keypme_pkcs11_client_ml_dsa_sign(channel, session, key_alias, ml_dsa_signature_mechanism, data)

    # Verify signature
    keypme_pkcs11_client_ml_dsa_verify(channel, session, key_alias, ml_dsa_signature_mechanism, data, signature)

    try:
        # Generate a random signature to make sure it really works!
        random_signature_list = [random.randint(0, 255) for _ in range(len(signature))]
        random_signature = bytes(random_signature_list)
        keypme_pkcs11_client_ml_dsa_verify(channel, session, key_alias, ml_dsa_signature_mechanism, data, random_signature)

        raise RuntimeError("Signature verification should have failed")
    except KeypMePkcs11InvalidSignatureException:
        pass

    print("Test PASS")

def _show_token_from_args(channel, args):
    stub = keypme_pkcs11_bridge_pb2_grpc.Pkcs11Stub(channel)
    try:
        response = stub.open(keypme_pkcs11_bridge_pb2.SessionOpenRequest(token_label=args.token_name, pin=args.pin))
        logging.info(f"Greeter client received: {response}")
    except grpc.RpcError as rpc_error:
        if rpc_error.code() == grpc.StatusCode.NOT_FOUND:
            logging.error(f"Received NOT_FOUND RPC error: message={rpc_error.details()}")
        else:
            logging.error(f"Received unknown RPC error: code={rpc_error.code()} message={rpc_error.details()}")


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--verbose', action='store_true', help='Enable verbose output')
    parser.add_argument('--server', default='localhost:50051', required=True, help='Server host:port')
    parser.add_argument('--root-certificates', type=argparse.FileType('rb'), help='Server root certificates')
    parser.add_argument('--client-certificate', type=argparse.FileType('rb'), help='Client certificate')
    parser.add_argument('--client-key', type=argparse.FileType('rb'), help='Client private key')
    subparsers = parser.add_subparsers(help='subcommand help')

    parser_test_server = subparsers.add_parser('test_server', help='Test server')
    parser_test_server.add_argument('--login-username', help='Login username')
    parser_test_server.add_argument('--login-key', type=argparse.FileType('rb'), help='Login key path')
    parser_test_server.set_defaults(func=_test_server_from_args)

    parser_algo_support = subparsers.add_parser('algo_support', help='Return supported algorithms')
    parser_algo_support.add_argument('--login-username', help='Login username')
    parser_algo_support.add_argument('--login-key', type=argparse.FileType('rb'), help='Login key path')
    parser_algo_support.set_defaults(func=_algo_support_from_args)

    parser_create_token = subparsers.add_parser('create_token', help='Create token')
    parser_create_token.add_argument('token_name', type=str, help='Token name')
    parser_create_token.add_argument('--login-username', help='Login username')
    parser_create_token.add_argument('--login-key', type=argparse.FileType('rb'), help='Login key path')
    parser_create_token.add_argument('--so-pin', required=True, help='SO Pin')
    parser_create_token.add_argument('--pin', required=True, help='Pin')
    parser_create_token.set_defaults(func=_create_token_from_args)

    parser_show_token = subparsers.add_parser('show_token', help='Show token')
    parser_show_token.add_argument('token_name', type=str, help='Token name')
    parser_show_token.add_argument('--pin', required=True, help='Pin')
    parser_show_token.set_defaults(func=_show_token_from_args)

    parser_unittest = subparsers.add_parser('unittest', help='Unittest')
    parser_unittest.add_argument('--login-username', help='Login username')
    parser_unittest.add_argument('--login-key', type=argparse.FileType('rb'), help='Login key path')
    parser_unittest.set_defaults(func=_unittest_from_args)

    args = parser.parse_args()

    if args.verbose:
        logging.basicConfig(level=logging.DEBUG)

    if hasattr(args, 'func'):
        if args.root_certificates is not None and args.client_key is not None and args.client_certificate is not None:
            with keypme_pkcs11_client_open(args.server, args.root_certificates.read(),
                                           args.client_certificate.read(), args.client_key.read()) as channel:
                args.func(channel, args)
        else:
            logging.info("Note: Add --root-certificates, --client-certificate, --client-key for gRPC over HTTPS")
            with keypme_pkcs11_client_open(args.server) as channel:
                args.func(channel, args)
    else:
        logging.error("Missing sub-command")
        parser.print_help()
