// SPDX-License-Identifier: GPL-2.0-only
/*
 * u.trust Anchor CSAR SR-IOV Driver
 *
 * Copyright 2024 Utimaco IS GmbH
 * All Rights Reserved.
 *
 */
#include <linux/time.h>

#include "cryptoserver.h"

/******************************************************************************
 *
 * Globals
 *
 *****************************************************************************/
static const int MonthLength[2][12] =
{
  {31,28,31,30,31,30,31,31,30,31,30,31 }, // normal years
  {31,29,31,30,31,30,31,31,30,31,30,31 }  // leap_years
};


/******************************************************************************
 *
 * Macros
 *
 *****************************************************************************/
#define LEAP_YEAR(a)  (( ((a)%4)==0) && ( (((a)%100)!=0) || (((a)%400)==0) ) )


/******************************************************************************
 *
 * cs_time_to_str
 *
 *****************************************************************************/
static int cs_time_to_str(const u32 epoch, struct tm *ts)
{
  int months      = 1;
  int days        = epoch / 86400;                  // days since 2000
  int seconds     = epoch - days * 86400;           // remaining seconds this day
  int years       = (days<<2)/1461;                 // number of years since 2000
  int leap_year   = LEAP_YEAR(years+2000);          // is this a leap year ?

  days -= years*365 + ((years+3)>>2);               // remaining days this year

  while(days >= MonthLength[leap_year][months-1] )
  {
    days -= MonthLength[leap_year][months-1];       // remaining days
    months++;
  }

  ts->tm_year = years + 70;
  ts->tm_mon  = months - 1;
  ts->tm_mday = days + 1;
  ts->tm_hour = seconds / 3600;
  seconds %= 3600;
  ts->tm_min  = seconds / 60;
  seconds %= 60;
  ts->tm_sec  = seconds;

  return 0;
}

/******************************************************************************
 *
 * cs_time_to_ascii
 *
 *****************************************************************************/
char *cs_time_to_ascii(const u32 epoch)
{
  static char buf[32];
  int len;
  struct tm ts;

  cs_time_to_str(epoch, &ts);

  len = snprintf(buf, sizeof(buf)-1, "%02d.%02d.%04ld %02d:%02d:%02d", ts.tm_mday, ts.tm_mon+1, ts.tm_year + 1900,
                                                                       ts.tm_hour, ts.tm_min, ts.tm_sec);
  buf[len] = 0;
  return buf;
}

/******************************************************************************
 *
 * cs_get_args
 *
 ******************************************************************************/
int cs_get_args(char *p_arg, const char *fmt, ...)
{
  va_list   ap;
  char      *p_x = p_arg;
  int       num_args = 0;

  if (*p_x != '=') return 0;
  p_x++;

  va_start(ap, fmt);

  while (p_x != NULL)
  {
    switch (*fmt++)
    {
      case 'i':
      {
        int *p_int = va_arg(ap, int*);

        if (p_int == NULL) break;

        *p_int = simple_strtol(p_x, NULL, 0);
        break;
      }

      case 'u':
      {
        int *p_uint = va_arg(ap, unsigned int*);

        if (p_uint == NULL) break;

        *p_uint = simple_strtol(p_x, NULL, 0);
        break;
      }

      /* case 'l':
      {
        long long *p_long = va_arg(ap, long long*);

        if (p_long == NULL) break;

        *p_long = simple_strtoll(p_x, NULL, 0);
        break;
      } */

      case 's':
      {
        char **pp_chr = va_arg(ap, char**);

        if (pp_chr == NULL) break;

        *pp_chr = p_x;
        break;
      }

      default:
        goto cleanup;
    }

    num_args++;

    if ((p_x = strchr(p_x, ',')) == NULL) break;
    *p_x++ = 0;
  }

cleanup:
  va_end(ap);
  return num_args;
}

/******************************************************************************
 *
 * cs_backlog_entry
 *
 *****************************************************************************/
void cs_backlog_entry(struct cs_backlog_t *bl, const char *where, int err, char *msg, ...)
{
  struct cs_backlog_entry_t *p_entry = bl->entries + bl->next;
  va_list args;

  p_entry->where = where;
  p_entry->err = err;

  va_start(args, msg);
  vsnprintf(p_entry->msg, sizeof(p_entry->msg), msg, args);
	va_end(args);

  bl->next = (bl->next + 1) & (BACKLOG_SIZE-1);

  if (bl->next == bl->first)
  {
    bl->first = (bl->first + 1) & (BACKLOG_SIZE-1);
  }
}

/******************************************************************************
 *
 * cs_backlog_print
 *
 *****************************************************************************/
void cs_backlog_print(struct cs_backlog_t *bl)
{
  int i = 0;

  while (bl->first != bl->next)
  {
    struct cs_backlog_entry_t *p_entry = bl->entries + bl->first;

    printk("#%4d | %5d | %-28s | %s\n", i++, p_entry->err, p_entry->where, p_entry->msg);

    bl->first = (bl->first + 1) & (BACKLOG_SIZE-1);
  }

  bl->first = bl->next = 0;
}
