/* SPDX-License-Identifier: GPL-2.0-only
 *
 * Copyright 2024 Utimaco IS GmbH
 * All Rights Reserved.
 *
 */
#ifndef __CSAR_DRV_PUB_H__
#define __CSAR_DRV_PUB_H__

/******************************************************************************
 *
 * Definitions
 *
 ******************************************************************************/
typedef unsigned long long  addr_t;

#define PTR_TO_U64(p)   (addr_t)((size_t)(p))

struct pci_request_t
{
  unsigned int  tag;          // sanity checker
  unsigned int  addr;         // recipient address - has become an output value!!!
  unsigned int  handle;       // message handle
  unsigned int  timeout;      // milliseconds

//unsigned int  rfu;
  unsigned char type;
  unsigned char rfu1[3];

  unsigned int  count;        // actual byte count
  addr_t        data;         // user buffer (send/receive)
};

#define REQUEST_TAG             0x52455100
#define REQUEST_TAG_VALID(x)    (((x) & 0xFFFFFF00) == REQUEST_TAG)
#define REQUEST_TAG_VERSION(x)  ((x) & 0xFF)

#define INVALID_REQUEST_HANDLE  (unsigned int)(-1)

#define REQUEST_TYPE_LOG        0x01

#define REQUEST_INIT_VALUE      { REQUEST_TAG, -1, INVALID_REQUEST_HANDLE, 0, 0, 0, 0 }

struct pci_info_t
{
  unsigned int  tag;          // sanity checker
  unsigned int  mask;         // verbosity
  unsigned int  size;         // max. size of buffer
  unsigned int  count;        // actual byte count
  addr_t        data;         // user buffer
};

#define INFO_TAG              0x494E464F

struct pci_reset_t
{
  unsigned int  tag;          // sanity checker
  unsigned int  mask;         // bit field
};

#define RESET_TAG             0x52535400

#define RESET_CHANNEL         1
#define RESET_BLOCK           2
#define RESET_CPU             4

// backward comp.
#define RESET_DEVICE          RESET_CPU
#define RESET_DRIVER          RESET_CHANNEL

struct pci_spm_t
{
  unsigned int  tag;          // sanity checker
  unsigned int  timeout;      // milliseconds
  unsigned int  addr;         // recipient address
  unsigned int  message;      // sponteneous message
};

#define SPM_TAG               0x53504D00


/******************************************************************************
 *
 * Control Codes
 *
 *****************************************************************************/

#if defined(OSYS_win) || defined(WIN32) || defined(_WIN64)

#ifdef _KERNEL_
  #include <devioctl.h>
  #define IOCTL_TRANSFER_TYPE( _iocontrol) (_iocontrol & 0x3)
#else
  #include <winioctl.h>
#endif

#define DEVICE_TYPE_CSAR          0x80C7    // 0x8000 <= custom < 0x10000
#define CSAR_FUNCTION_BASE        0x800     // 0x800  <= custom < 0x1000

#define CSAR_CTL_CODE(function,method) CTL_CODE(DEVICE_TYPE_CSAR,function,method,FILE_ANY_ACCESS)

#define CSAR_CTL_RESET            CSAR_CTL_CODE((CSAR_FUNCTION_BASE +  1),   METHOD_BUFFERED)
#define CSAR_CTL_INFO             CSAR_CTL_CODE((CSAR_FUNCTION_BASE +  2),   METHOD_BUFFERED)
#define CSAR_CTL_GET_REQUEST      CSAR_CTL_CODE((CSAR_FUNCTION_BASE +  3),   METHOD_BUFFERED)
#define CSAR_CTL_CANCEL_REQUEST   CSAR_CTL_CODE((CSAR_FUNCTION_BASE +  4),   METHOD_BUFFERED)
#define CSAR_CTL_RECV             CSAR_CTL_CODE((CSAR_FUNCTION_BASE +  5),   METHOD_BUFFERED)
#define CSAR_CTL_SEND             CSAR_CTL_CODE((CSAR_FUNCTION_BASE +  6),   METHOD_BUFFERED)
#define CSAR_CTL_SPM_SEND         CSAR_CTL_CODE((CSAR_FUNCTION_BASE +  7),   METHOD_BUFFERED)
#define CSAR_CTL_SPM_RECV         CSAR_CTL_CODE((CSAR_FUNCTION_BASE +  8),   METHOD_BUFFERED)
#define CSAR_CTL_LOG_LEVEL        CSAR_CTL_CODE((CSAR_FUNCTION_BASE +  9),   METHOD_BUFFERED)
#define CSAR_CTL_GET_MODEL        CSAR_CTL_CODE((CSAR_FUNCTION_BASE + 10),   METHOD_BUFFERED)
#define CSAR_CTL_SELECT           CSAR_CTL_CODE((CSAR_FUNCTION_BASE + 11),   METHOD_BUFFERED)
#define CSAR_CTL_TEST(nr)         CSAR_CTL_CODE((CSAR_FUNCTION_BASE + (nr)), METHOD_BUFFERED)

#define CTL_TRANSFER_TYPE(code)    (((ULONG)code) & 3)
#define CTL_FUNCTION(code)        ((((ULONG)code) >>  2) & 0x3FF)
#define CTL_DEVICE(code)          ((((ULONG)code) >> 14) & 3)

#else

#define CSAR_IOC_MAGIC            0xC7  // (unused according to 'ioctl-number.txt', see kernel doc.)

#define CSAR_CTL_RESET            _IOW  (CSAR_IOC_MAGIC,  1, struct pci_reset_t)
#define CSAR_CTL_INFO             _IOWR (CSAR_IOC_MAGIC,  2, struct pci_info_t)
#define CSAR_CTL_GET_REQUEST      _IOWR (CSAR_IOC_MAGIC,  3, struct pci_request_t)
#define CSAR_CTL_CANCEL_REQUEST   _IOWR (CSAR_IOC_MAGIC,  4, struct pci_request_t)
#define CSAR_CTL_RECV             _IOWR (CSAR_IOC_MAGIC,  5, struct pci_request_t)
#define CSAR_CTL_SEND             _IOWR (CSAR_IOC_MAGIC,  6, struct pci_request_t)
#define CSAR_CTL_SPM_SEND         _IOWR (CSAR_IOC_MAGIC,  7, struct pci_spm_t)
#define CSAR_CTL_SPM_RECV         _IOWR (CSAR_IOC_MAGIC,  8, struct pci_spm_t)
#define CSAR_CTL_LOG_LEVEL        _IOWR (CSAR_IOC_MAGIC,  9, unsigned int)
#define CSAR_CTL_GET_MODEL        _IOR  (CSAR_IOC_MAGIC, 10, unsigned int)
// new as of 5.31.0
#define CSAR_CTL_LOCK             _IO   (CSAR_IOC_MAGIC, 12)
#define CSAR_CTL_UNLOCK           _IO   (CSAR_IOC_MAGIC, 13)

#define CSAR_CTL_TEST(nr)         _IO   (CSAR_IOC_MAGIC, (nr))

#endif

#endif
