// SPDX-License-Identifier: GPL-2.0-only
/*
 * u.trust Anchor CSAR SR-IOV Driver
 *
 * Copyright 2024 Utimaco IS GmbH
 * All Rights Reserved.
 *
 */
#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#include <linux/version.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/mutex.h>
#include <linux/poll.h>
#include <linux/proc_fs.h>
#include <linux/uaccess.h>
#include <linux/seq_file.h>
#include <linux/sched.h>
#include <linux/time.h>
#include <linux/aer.h>
#include <linux/string.h>
#include <linux/workqueue.h>
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,11,0))
#include <linux/sched/signal.h>
#endif
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/ethtool.h>

#include "cryptoserver.h"
#include "csar_drv.h"
#include "cs2_drv.h"
#include "load_store.h"
#include "gdma.h"
#include "version.h"

// #define USE_STREAMING_DMA  // use streaming DMA instead of coherent memory
// #define PCIE_FLUSH         // flush PCIe after having written queue descripter (submit, fetch)

/******************************************************************************
 *
 * Types
 *
 *****************************************************************************/

//-----------------------------------------------------------------------------
// Linked List
//-----------------------------------------------------------------------------
struct cs8_list_t
{
  struct list_head  head;
  spinlock_t        lock;
  wait_queue_head_t queue;

  unsigned int seq_cnt;
  unsigned int waiter;

  int count;
  int count_min;
  int count_max;
};

//-----------------------------------------------------------------------------
// Packet
//-----------------------------------------------------------------------------
struct cs8_packet_t
{
  void  *base_ptr;
  union
  {
    struct
    {
#ifdef CS_BIG_ENDIAN
      u32 phys_addr_hi;
      u32 phys_addr_lo;
#else
      u32 phys_addr_lo;
      u32 phys_addr_hi;
#endif
    };

    phys_addr_t phys_addr;
  };

  unsigned int size;

  // 'extends' cs_dmabuf_t
  unsigned int idx;

  struct list_head    list;

  union packet_info_t info;
};

#define QUEUE_SIZE            GDMA_QUEUE_SIZE

#define MAX_PACKETS_TX        QUEUE_SIZE - 1
#define MAX_PACKETS_RX        128

#define MAX_PACKET_SIZE       4 * PAGE_SIZE

#define DEF_PACKET_PREFETCH   8

//-----------------------------------------------------------------------------
// DMA channel
//-----------------------------------------------------------------------------
struct cs8_channel_t
{
  unsigned int type;

  unsigned char __iomem *bar;

  // descriptor queue
  unsigned int write_idx;
  unsigned int read_idx;
  unsigned int pending;

  // packets
  enum dma_data_direction dir;
  unsigned int          max_packets;
  struct cs8_packet_t   *packets;
  struct cs8_list_t     free_list;

  spinlock_t            lock;
  struct tasklet_struct task;
  int                   error;

  // statistics
  unsigned int pkt_sched;
  unsigned int pkt_done;
  unsigned int pkt_count[QUEUE_SIZE];
};

#define CHANNEL_TYPE_PCI_AXI  'T'
#define CHANNEL_TYPE_AXI_PCI  'R'


//-----------------------------------------------------------------------------
// Mailbox
//-----------------------------------------------------------------------------
struct cs8_mailbox_t
{
  spinlock_t            lock;
  struct tasklet_struct task;
};

union cs8_spm_t
{
  struct
  {
    u16 message;
    u8  addr;
    u8  type;
  };

  u32 value;
};

/*
#define SPM_TYPE_MASK         0xFF000000
#define SPM_ADDR_MASK         0x00FF0000
#define SPM_MESSAGE_MASK      0x0000FFFF
*/

#define SPM_TYPE_DISABLED     0x10
#define SPM_TYPE_INITIALIZED  0x11
#define SPM_TYPE_ENABLED      0x12
#define SPM_TYPE_VERSION      0x20
#define SPM_TYPE_USER         0x80

//-----------------------------------------------------------------------------
// Message
//-----------------------------------------------------------------------------
struct cs8_message_t
{
  u64                     mask;

  unsigned char           idx;
  volatile unsigned char  state;
  volatile unsigned char  flags;
  unsigned char           type;

  unsigned int            count;

  struct list_head        list;

//struct cs8_list_t       tx_list;  // not yet used
  struct cs8_list_t       rx_list;

  wait_queue_head_t       tx_event;
  wait_queue_head_t       rx_event;

  struct timer_list       timer;
  struct cs8_device_t     *dp;
  struct cs_session_t     *session;
};

#define MSG_STATE_FREE          0
#define MSG_STATE_LOCKED        1
#define MSG_STATE_SEND          2
#define MSG_STATE_SEND_DONE     3
#define MSG_STATE_GETREQ        4
#define MSG_STATE_GETREQ_DONE   5
#define MSG_STATE_RECV          6
#define MSG_STATE_RECV_DONE     7

#define MSG_FLAG_CANCEL         0x01
#define MSG_FLAG_EXPIRED        0x02

#define MSG_TYPE_LOG            0x01

#define MAX_MSG_SIZE            0x100000
#define MAX_MESSAGES            16
#define MSG_IDX_NET             0xFF

#define DEF_MSG_EXPIRY          3*HZ

struct netstats
{
  u64 packets;
  u64 bytes;
  u64 errors;
};

//-----------------------------------------------------------------------------
// Device
//-----------------------------------------------------------------------------
struct cs8_device_t
{
  union
  {
    struct cs_device_t    csdev;
    struct
    {
      struct cs8_device_t *next;
      struct cs8_device_t *parent;
    };
  };

  unsigned int            tag;

  unsigned short          chnid;
  unsigned char           type;
  unsigned char           irq_mode;
  volatile unsigned short state;
  volatile unsigned short flags;

  unsigned int            ddversion;

  unsigned int            relaunch;

  struct delayed_work     background_task;
  struct pci_saved_state  *pss;

  struct cs_iomem_t       iomem[1];
  unsigned char __iomem   *bar;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(4,8,0))
  struct msix_entry       msix[3];
#endif
  int                     irq_vector[3];

  struct cs8_channel_t    tx_channel;
  struct cs8_channel_t    rx_channel;
  struct cs8_mailbox_t    mailbox;

  struct cs8_message_t    messages[MAX_MESSAGES];
  struct cs8_list_t       free_list;

  wait_queue_head_t       spm_event;
  unsigned int            spm_message;

  struct cs8_device_t     **devices;

  struct net_device       *netdev;
  struct netstats         rx_stats;
  struct netstats         tx_stats;

  struct semaphore        sema;

  // statistics
  unsigned int            irq_cnt[3];
  unsigned int            irq_shared[3];
  unsigned int            irq_ghost[3];
};

struct cs8_netdev
{
  struct cs8_device_t *dp;
  unsigned int        use_count;
};

#define DEVICE_TAG    0x44455638

#define DEVICE_STATE_OFFLINE        0
#define DEVICE_STATE_DISABLED       1
#define DEVICE_STATE_INITIALIZED    2
#define DEVICE_STATE_ENABLED        3
#define DEVICE_STATE_RECOVER        0x100

#define DEVICE_TYPE_PARENT          0
#define DEVICE_TYPE_VIRT            1
#define DEVICE_TYPE_CHILD           2

#define DEVICE_FLAG_CRC             0x1
#define DEVICE_FLAG_NET             0x2
#define DEVICE_FLAG_STARTUP         0x4

#define MAX_DEVICES   GDMA_MAX_CHANNELS

#define IRQ_MODE_CHANNEL            0
#define IRQ_MODE_SHARED             1

/******************************************************************************
 *
 * Globals
 *
 *****************************************************************************/
static unsigned int MaxPackets = MAX_PACKETS_RX;
module_param(MaxPackets, int, 0);
MODULE_PARM_DESC(MaxPackets, " max. number of RX packets per message [default: " __stringify(MAX_PACKETS_RX)"]");

static unsigned int Prefetch = DEF_PACKET_PREFETCH;
module_param(Prefetch, int, 0444);
MODULE_PARM_DESC(Prefetch, " number if RX packets to be prefetched [default: " __stringify(DEF_PACKET_PREFETCH)"]");

static unsigned int MaxMessages = MAX_MESSAGES;
//module_param(MaxMessages, int, 0);
//MODULE_PARM_DESC(MaxMessages, " max. number of messages per channel [default: " __stringify(MAX_MESSAGES)"]");

static unsigned int MessageExpiry = DEF_MSG_EXPIRY;
module_param(MessageExpiry, int, 0444);
MODULE_PARM_DESC(MessageExpiry, " message expiration timeout <Jiffies> [default: " __stringify(DEF_MSG_EXPIRY)"]");

static unsigned int DeviceMask = 0;
module_param(DeviceMask, uint, 0444);
MODULE_PARM_DESC(DeviceMask, " bit mask defining child devices in non-SRIOV mode [default: 0]");

static unsigned int DeviceFlags = 0;  // DEVICE_FLAG_CRC:1 | DEVICE_FLAG_NET:2
module_param(DeviceFlags, int, 0444);
MODULE_PARM_DESC(DeviceFlags, " bit mask defining features [2: enable network adapter]");

static unsigned int MaxPayload = 0;
module_param(MaxPayload, int, 0);
MODULE_PARM_DESC(MaxPayload, " max. PCI payload size [default: don't change]");

static unsigned int MaxReadReq = 0;
module_param(MaxReadReq, int, 0);
MODULE_PARM_DESC(MaxReadReq, " max. PCI read request size [default: don't change]");

static unsigned int Loopback = 0;
static unsigned int CoreControl = 0;
#ifdef DEBUG
module_param(Loopback, int, 0);
module_param(CoreControl, int, 0);
#endif

static const char *DeviceTypes[4]  = { "parent", "virt", "child", "?3!" };

static const char *DeviceStates[4] = { "offline", "disabled", "initial", "enabled" };

static const char *IrqModes[2]     = { "channel", "shared" };

static const char *IrqNames[4]     = { "cs8.tx_channel", "cs8.rx_channel", "cs8.mailbox", "?3!" };

static const char *MsgStates[8]    = { "idle", "locked", "send", "send_done",
                                       "getreg", "getreq_done", "recv", "recv_done" };

static const char *PowerStates[8]  = { "D0", "D1", "D2", "D3hot", "D3cold", "unknown", "?6!", "?7!" };

/******************************************************************************
 *
 * Macros
 *
 *****************************************************************************/
#define MASK_QUERY(mask, idx)  ( (mask)[(idx)/32] &   (1 << ((idx) & 31)) )
#define MASK_SET(mask, idx)    ( (mask)[(idx)/32] |=  (1 << ((idx) & 31)) )
#define MASK_CLR(mask, idx)    ( (mask)[(idx)/32] &= ~(1 << ((idx) & 31)) )

#define DMA_WLEN(c) (((c) + 7) / 8)

#define DO_PRAGMA(x) _Pragma(#x)
#define TODO(x) DO_PRAGMA(message ("TODO: " #x))
#define NOTE(x) DO_PRAGMA(message ("NOTE: " #x))

#define SANITY_CHECK(x,t) ( (x) != NULL && (x)->tag == (t) )

static void cs8_device_reset(struct cs8_device_t *dp);
static int  cs8_device_init(struct cs8_device_t *dp);
static int  cs8_messages_cleanup(struct cs8_device_t *dp);

int cs8_probe(struct pci_dev *pci_dev, struct cs8_device_t **pp_dp);

/******************************************************************************
 *
 * cs8_param_info
 *
 *****************************************************************************/
static int cs8_param_info(char *p, int max)
{
  int len = 0;

  len += scnprintf(p+len, max-len, "Module Parameter\n");

  len += scnprintf(p+len, max-len, " LogLevel        %s\n",   LogLevelTxt[LogLevel&7]);
  len += scnprintf(p+len, max-len, " DeviceMask      0x%x\n", DeviceMask);
  len += scnprintf(p+len, max-len, " DeviceFlags     0x%x\n", DeviceFlags);

  len += scnprintf(p+len, max-len, " MaxPackets      %d\n",   MaxPackets);
  len += scnprintf(p+len, max-len, " Prefetch        %d\n",   Prefetch);
  len += scnprintf(p+len, max-len, " MaxMessages     %d\n",   MaxMessages);
  len += scnprintf(p+len, max-len, " MessageExpiry   %d\n",   MessageExpiry);

  len += scnprintf(p+len, max-len, " MaxPayload      %d\n",   MaxPayload);
  len += scnprintf(p+len, max-len, " MaxReadReq      %d\n",   MaxReadReq);
  len += scnprintf(p+len, max-len, " ErrorReporting  %d\n",   ErrorReporting);
#ifdef DEBUG
  len += scnprintf(p+len, max-len, " Loopback        %d\n",   Loopback);
  len += scnprintf(p+len, max-len, " CoreControl     0x%x\n", CoreControl);
#endif

  p[len] = 0;
  return len;
}

/******************************************************************************
 *
 * cs8_set_error_reporting
 *
 *****************************************************************************/
static int cs8_set_error_reporting(struct pci_dev *pci_dev)
{
#ifdef USE_PCI_ERROR_REPORTING
  int (*pci_configure_pcie_error_reporting[2])(struct pci_dev*) =
  {
    pci_disable_pcie_error_reporting,
    pci_enable_pcie_error_reporting
  };

  if (pci_dev->is_physfn)
  {
    int err;
    struct pci_dev *bridge = pci_upstream_bridge(pci_dev);

    if (  (err = pci_configure_pcie_error_reporting[ ErrorReporting    &1](pci_dev)) != 0
       || (err = pci_configure_pcie_error_reporting[(ErrorReporting>>1)&1](bridge)) != 0
       )
    {
      return err;
    }
  }
#endif

  return 0;
}

/******************************************************************************
 *
 * cs8_set_maxpay
 *
 *****************************************************************************/
static int cs8_set_maxpay(struct pci_dev *pci_dev)
{
  int err;
  struct pci_dev *bridge = pci_upstream_bridge(pci_dev);
  int max_mps_bridge = (bridge != NULL) ? pcie_get_mps(bridge) : -1;

  if (MaxPayload != 0)
  {
    if (MaxPayload > max_mps_bridge)
      log_warning("MaxPayload is greater than max. payload size of upstream bridge: %d\n", max_mps_bridge);

    if ((err = pcie_set_mps(pci_dev, MaxPayload)) != 0) return err;

    log_trace("max. payload size was set to: %4d [%4d]\n", pcie_get_mps(pci_dev), max_mps_bridge);
  }

  return 0;
}

/******************************************************************************
 *
 * cs8_set_readrq
 *
 *****************************************************************************/
static int cs8_set_readrq(struct pci_dev *pci_dev)
{
  int err;
  struct pci_dev *bridge = pci_upstream_bridge(pci_dev);
  int max_readrq_bridge = (bridge != NULL) ? pcie_get_readrq(bridge) : -1;

  if (MaxReadReq != 0)
  {
    if ((err = pcie_set_readrq(pci_dev, MaxReadReq)) != 0) return err;
  }
  else
  {
    // automatic evaluation
    int sizes[] = { 1024, 512, 256, 128 };
    int i;

    for (i=0; i<DIM(sizes); i++)
    {
      if ((err = pcie_set_readrq(pci_dev, sizes[i])) == 0) goto end;
    }

    return err;
  }

end:
  log_trace("read request size was set to: %4d [%4d]\n", pcie_get_readrq(pci_dev), max_readrq_bridge);
  return 0;
}

/******************************************************************************
 *
 * cs8_set_core_control
 *
 *****************************************************************************/
static void cs8_set_core_control(struct cs8_device_t *dp, u32 value)
{
  if (value != 0)
  {
    u32 old = readl(dp->bar + GDMA_CORE_CONTROL);
    u32 new = 0;
    u32 mask;

    log_trace("old core control register: %08x\n", old);

    for (mask=0xF; mask!=0; mask<<=4)
    {
      if (value & mask) new |= value & mask;
      else              new |= old & mask;
    }

    writel(new, dp->bar + GDMA_CORE_CONTROL);

    log_trace("new core control register: %08x\n", new);
  }
}

/******************************************************************************
 *
 * cs8_list_init
 *
 *****************************************************************************/
static void cs8_list_init(struct cs8_list_t *list)
{
  INIT_LIST_HEAD(&list->head);

  spin_lock_init(&list->lock);

  init_waitqueue_head(&list->queue);

  list->seq_cnt = 0;
  list->waiter = 0;

  list->count = 0;
  list->count_min = 0;
  list->count_max = 0;
}

/******************************************************************************
 *
 * cs8_io_map
 *
 *****************************************************************************/
static int cs8_io_map(struct pci_dev *pci_dev, int idx, struct cs_iomem_t *bar)
{
  resource_size_t addr;
  resource_size_t size;
  unsigned long flags = pci_resource_flags(pci_dev, idx);

  if ((flags & IORESOURCE_MEM) == 0)
  {
    log_error("invalid resource type: 0x%lx\n", flags);
    return -ENODEV;
  }

  if (  (addr = pci_resource_start(pci_dev, idx)) == 0
     || (size = pci_resource_len  (pci_dev, idx)) == 0
     )
  {
    log_error("no resource BAR%d\n", idx);
    return -ENODEV;
  }

  bar->base_ptr = ioremap(addr, size);
//bar->base_ptr = ioremap_wc(addr, size);
//bar->base_ptr = ioremap_wt(addr, size);

  if (bar->base_ptr == NULL)
  {
    log_error("can't remap io memory for BAR%d\n", idx);
    return -ENODEV;
  }

  bar->phys_addr = addr;
  bar->size = size;

  log_trace("bar#%d: phys.: %08x:%08x -> virt.: %p, size = 0x%08lx\n", idx,
                                                                       bar->phys_addr_hi,
                                                                       bar->phys_addr_lo,
                                                                       bar->base_ptr,
                                                                       bar->size);
  return 0;
}

/******************************************************************************
 *
 * cs8_io_unmap
 *
 *****************************************************************************/
static int cs8_io_unmap(struct cs_iomem_t *bar)
{
  if (bar->base_ptr == NULL) return -ENXIO;

  iounmap(bar->base_ptr);
  bar->base_ptr = NULL;

  return 0;
}

/******************************************************************************
 *
 * cs8_irq_enable
 *
 *****************************************************************************/
static inline void cs8_irq_enable(struct cs8_device_t *dp)
{
  unsigned int mask = GDMA_IRQ_DMA_TX | GDMA_IRQ_DMA_RX | GDMA_IRQ_MAILBOX;

  // clear interrupts
  writeb(mask, dp->bar + GDMA_IRQ_STATUS);

  // enable all interrupts
  writeb(mask, dp->bar + GDMA_IRQ_ENABLE);
  writeb(mask, dp->bar + GDMA_IRQ_MASK);
}

/******************************************************************************
 *
 * cs8_irq_disable
 *
 *****************************************************************************/
static inline void cs8_irq_disable(struct cs8_device_t *dp)
{
  // disable all interrupts
  writeb(0, dp->bar + GDMA_IRQ_ENABLE);
  writeb(0, dp->bar + GDMA_IRQ_MASK);

  // clear interrupts
  writeb(GDMA_IRQ_DMA_TX | GDMA_IRQ_DMA_RX | GDMA_IRQ_MAILBOX, dp->bar + GDMA_IRQ_STATUS);
}

/******************************************************************************
 *
 * cs8_pkt_alloc_ex
 *
 *****************************************************************************/
static int cs8_pkt_alloc_ex(struct cs8_device_t *dp, unsigned int size, enum dma_data_direction dir, struct cs8_packet_t *pkt)
{
  struct pci_dev *pci_dev = dp->csdev.pci_dev;

#ifdef USE_STREAMING_DMA
  int err;
  struct device *dev = &pci_dev->dev;

  pkt->base_ptr = kzalloc(size, GFP_KERNEL);

  if (pkt->base_ptr == NULL)
  {
    log_error("kzalloc(%d) failed\n", size);
    return -ENOMEM;
  }

  pkt->size = size;

  pkt->phys_addr = dma_map_single(dev, pkt->base_ptr, pkt->size, dir);

  if ((err= dma_mapping_error(dev, pkt->phys_addr)) != 0)
  {
    log_error("dma_mapping_error returned: %d\n", err);
    return err;
  }

  if (dir == DMA_TO_DEVICE)
  {
    // TX: CPU should initially own buffer
    dma_sync_single_for_cpu(dev, pkt->phys_addr, pkt->size, DMA_TO_DEVICE);
  }
  else
  {
    // RX: device should initially own buffer
    dma_sync_single_for_device(dev, pkt->phys_addr, pkt->size, DMA_FROM_DEVICE);
  }
#else
  pkt->base_ptr = dma_alloc_coherent(&pci_dev->dev, size, &pkt->phys_addr, GFP_ATOMIC);

  if (pkt->base_ptr == NULL)
  {
    log_error("pci_alloc_consistent failed\n");
    return -ENOMEM;
  }

  pkt->size = size;
#endif

  memset(pkt->base_ptr, 0xCC, size);

  log_debug("virt.addr: %p, phys.addr: %08x:%08x, size = %d\n", pkt->base_ptr,
                                                                pkt->phys_addr_hi,
                                                                pkt->phys_addr_lo,
                                                                pkt->size);
  return 0;
}

/******************************************************************************
 *
 * cs8_pkt_free_ex
 *
 *****************************************************************************/
static void cs8_pkt_free_ex(struct cs8_device_t *dp, enum dma_data_direction dir, struct cs8_packet_t *pkt)
{
  struct pci_dev *pci_dev = dp->csdev.pci_dev;

  if (pkt->base_ptr == NULL) return;

  memset(pkt->base_ptr, 0, sizeof(struct cs8_packet_t));

#ifdef USE_STREAMING_DMA
  if (pkt->phys_addr != 0)
    dma_unmap_single(&pci_dev->dev, pkt->phys_addr, pkt->size, dir);

  kfree(pkt->base_ptr);
#else
  dma_free_coherent(&pci_dev->dev, pkt->size, pkt->base_ptr, pkt->phys_addr);
#endif

  pkt->base_ptr = NULL;
  pkt->phys_addr = 0;
  pkt->size = 0;
}

/******************************************************************************
 *
 * cs8_pkt_get_first
 *
 *****************************************************************************/
static int cs8_pkt_get_first(struct cs8_list_t *list, struct cs8_packet_t **pp_pkt, unsigned int *nr)
{
  struct cs8_packet_t *pkt;
  unsigned long flags;

  spin_lock_irqsave(&list->lock, flags);

  if ((pkt = list_first_entry_or_null(&list->head, struct cs8_packet_t, list)) == NULL)
  {
    if (nr) *nr = ++list->waiter;
    spin_unlock_irqrestore(&list->lock, flags);

    *pp_pkt = NULL;
    return -EBUSY;
  }

  list->waiter = list->seq_cnt;

  list_del_init(&pkt->list);

  list->count--;
  if (list->count < list->count_min) list->count_min = list->count;

  spin_unlock_irqrestore(&list->lock, flags);

  *pp_pkt = pkt;
  return 0;
}

/******************************************************************************
 *
 * cs8_pkt_get_first_timeout
 *
 *****************************************************************************/
static int cs8_pkt_get_first_timeout(struct cs8_list_t *list, long timeout, struct cs8_packet_t **pp_pkt)
{
  int err;
  unsigned int nr;

  while ((err = cs8_pkt_get_first(list, pp_pkt, &nr)) != 0)
  {
    timeout = wait_event_interruptible_timeout(list->queue,
                                               (nr == list->seq_cnt),
                                               timeout);
    if (timeout <= 0)
    {
      log_trace("wait_event_interruptible_timeout returned: %ld\n", timeout);
      *pp_pkt = NULL;
      // return ((timeout < 0) ? timeout : -ETIMEDOUT);
      return err;
    }
  }

  return 0;
}

/******************************************************************************
 *
 * cs8_pkt_put_last
 *
 *****************************************************************************/
static void cs8_pkt_put_last(struct cs8_list_t *list, struct cs8_packet_t *pkt)
{
  unsigned long flags;

  spin_lock_irqsave(&list->lock, flags);

  list_add_tail(&pkt->list, &list->head);

  list->count++;
  if (list->count > list->count_max) list->count_max = list->count;

  spin_unlock_irqrestore(&list->lock, flags);
}

/******************************************************************************
 *
 * cs8_channel_info
 *
 *****************************************************************************/
static int cs8_channel_info(struct cs8_channel_t *channel, char *p, int max)
{
  struct cs8_list_t *free_list = &channel->free_list;
  int len = 0;

  len += scnprintf(p+len, max-len, "%cX channel\n", channel->type);

  // packet pool
  len += scnprintf(p+len, max-len, " packet pool     %2d <%2d..%2d>", free_list->count, free_list->count_min, free_list->count_max);

#ifdef DEBUG
  {
    // dump packets in free list
    unsigned long flags;
    struct cs8_packet_t *pkt;
    int i = 0;

    spin_lock_irqsave(&free_list->lock, flags);

    list_for_each_entry(pkt, &free_list->head, list)
    {
      if (  i < 8
         || i > free_list->count - 8
         )
      {
        len += scnprintf(p+len, max-len, " %02d", pkt->idx);
      }

      if (i++ == 8)
      {
        len += scnprintf(p+len, max-len, " ...");
      }
    }

    spin_unlock_irqrestore(&free_list->lock, flags);
  }
#endif

  p[len++] = '\n';

  // statistics
  len += scnprintf(p+len, max-len, " pkt_sched       %d\n", channel->pkt_sched);
  len += scnprintf(p+len, max-len, " pkt_done        %d", channel->pkt_done);

#ifdef DEBUG
  {
    int cnt;

    for (cnt=DIM(channel->pkt_count); --cnt != 0; )
    {
      if (channel->pkt_count[cnt] != 0) break;
    }

    if (cnt <= 16)
    {
      for (i=0; i<cnt; i++)
      {
        len += scnprintf(p+len, max-len, " %d", channel->pkt_count[i]);
      }
    }
    else
    {
      for (i=0; i<10; i++)
        len += scnprintf(p+len, max-len, " %d", channel->pkt_count[i]);

      len += scnprintf(p+len, max-len, "... ");

      for (i=cnt-6; i<cnt; i++)
        len += scnprintf(p+len, max-len, " %d", channel->pkt_count[i]);
    }

    len += scnprintf(p+len, max-len, " [%d]", cnt);
  }
#endif

  p[len++] = '\n';

  len += scnprintf(p+len, max-len, " pending         %d\n", channel->pending);
  len += scnprintf(p+len, max-len, " last error      %d\n", channel->error);

  p[len] = 0;
  return len;
}

/******************************************************************************
 *
 * cs8_channel_init
 *
 *****************************************************************************/
static int cs8_channel_init(struct cs8_channel_t *channel)
{
  int i;

  // initialize queue
  channel->write_idx = 0;
  channel->read_idx = 0;
  channel->pending = 0;

  // initialize packet list
  cs8_list_init(&channel->free_list);
//cs8_list_init(&channel->pend_list);

  for (i=0; i<channel->max_packets; i++)
  {
    struct cs8_packet_t *pkt = channel->packets + i;

    pkt->idx = i;
    pkt->info.value = 0;

    INIT_LIST_HEAD(&pkt->list);

    cs8_pkt_put_last(&channel->free_list, pkt);
  }

  channel->free_list.count_min = i;
  channel->free_list.count_max = i;

  // stats
  channel->pkt_sched = 0;
  channel->pkt_done = 0;
  memset(channel->pkt_count, 0, sizeof(channel->pkt_count));

  return 0;
}

/******************************************************************************
 *
 * cs8_channel_flush
 *
 *****************************************************************************/
#ifdef PCIE_FLUSH
static inline void cs8_channel_flush(struct cs8_channel_t *channel)
{
  readl(channel->bar + GDMA_VERSION);
}
#else
  #define cs8_channel_flush(...)
#endif

/******************************************************************************
 *
 * cs8_channel_pkt_submit_unlocked
 *
 *****************************************************************************/
static int cs8_channel_pkt_submit_unlocked(struct cs8_device_t *dp, struct cs8_channel_t *channel, struct cs8_packet_t *pkt)
{
  unsigned int ofs;

  if (channel->pending >= QUEUE_SIZE-1)
  {
    log_trace("SRC queue overflow\n");
    return -EBUSY;
  }

#ifdef USE_STREAMING_DMA
  dma_sync_single_for_device(&dp->csdev.pci_dev->dev, pkt->phys_addr, pkt->info.wlen*8, DMA_TO_DEVICE);
  // CPU may no longer access packet buffer
#endif

  // write descriptor
  ofs = GDMA_SRC_Q_ADDR(channel->write_idx);
  writeq(pkt->phys_addr,  channel->bar + ofs);
  writeq(pkt->info.value, channel->bar + ofs + 8);

  cs8_channel_flush(channel);

  // increment write index
  channel->write_idx = (channel->write_idx + 1) & (QUEUE_SIZE - 1);

  // write new index
  writeb(channel->write_idx, channel->bar + GDMA_SRC_Q_WR_IDX);

  cs8_channel_flush(channel);

  channel->pending++;
  channel->pkt_sched++;

  return 0;
}

/******************************************************************************
 *
 * cs8_channel_pkt_submit
 *
 *****************************************************************************/
static inline int cs8_channel_pkt_submit(struct cs8_device_t *dp, struct cs8_channel_t *channel, struct cs8_packet_t *pkt)
{
  int err;
  unsigned long flags;
  spin_lock_irqsave(&channel->lock, flags);
  err = cs8_channel_pkt_submit_unlocked(dp, channel, pkt);
  spin_unlock_irqrestore(&channel->lock, flags);
  return err;
}

/******************************************************************************
 *
 * cs8_channel_pkt_fetch_unlocked
 *
 *****************************************************************************/
static int cs8_channel_pkt_fetch_unlocked(struct cs8_device_t *dp, struct cs8_channel_t *channel, struct cs8_packet_t *pkt)
{
  unsigned int ofs;

  if (channel->pending >= QUEUE_SIZE-1)
  {
    log_trace("DST queue overflow\n");
    return -EBUSY;
  }

  // invalidate data header
  *((u64*)pkt->base_ptr) = -1;

  // invalidate packet header
  pkt->info.value = 0;
  pkt->info.pid = pkt->idx;

#ifdef USE_STREAMING_DMA
  dma_sync_single_for_device(&dp->csdev.pci_dev->dev, pkt->phys_addr, pkt->size, DMA_FROM_DEVICE);
  // CPU may no longer access packet buffer
#endif

  // write descriptor
  ofs = GDMA_DST_Q_ADDR(channel->write_idx);
  writeq(pkt->phys_addr,  channel->bar + ofs);
  writeq(pkt->info.value, channel->bar + ofs + 8);

  cs8_channel_flush(channel);

  // increment write index
  channel->write_idx = (channel->write_idx + 1) & (QUEUE_SIZE - 1);

  // write new index
  writeb(channel->write_idx, channel->bar + GDMA_DST_Q_WR_IDX);

  cs8_channel_flush(channel);

  channel->pending++;
  channel->pkt_sched++;
  return 0;
}

/******************************************************************************
 *
 * cs8_channel_prefetch_unlocked
 *
 *****************************************************************************/
static int cs8_channel_prefetch_unlocked(struct cs8_device_t *dp, struct cs8_channel_t *channel)
{
  int err;

  while(channel->pending < Prefetch)
  {
    struct cs8_list_t *list = &channel->free_list;
    struct cs8_packet_t *pkt;

    if ((err = cs8_pkt_get_first(list, &pkt, NULL)) != 0)
    {
      log_error("channel#%d: cs8_pkt_get_first returned: %d\n", dp->chnid, err);
      return err;
    }

    if ((err = cs8_channel_pkt_fetch_unlocked(dp, channel, pkt)) != 0)
    {
      cs8_pkt_put_last(list, pkt);
      log_error("channel#%d: cs8_channel_pkt_fetch_unlocked(%d) returned: %d\n", dp->chnid, pkt->idx, err);
      return err;
    }
  }

  return 0;
}

/******************************************************************************
 *
 * cs8_channel_prefetch
 *
 *****************************************************************************/
static int cs8_channel_prefetch(struct cs8_device_t *dp)
{
  int err;
  struct cs8_channel_t *channel = &dp->rx_channel;
  unsigned long flags;
  spin_lock_irqsave(&channel->lock, flags);
  err = cs8_channel_prefetch_unlocked(dp, channel);
  spin_unlock_irqrestore(&channel->lock, flags);
  return err;
}

/******************************************************************************
 *
 * cs8_channel_tx_task
 *
 *****************************************************************************/
static void cs8_channel_tx_task(unsigned long param)
{
  struct cs8_device_t *dp = (struct cs8_device_t *)param;
  struct cs8_channel_t *channel = &dp->tx_channel;
  struct cs8_list_t *free_list = &channel->free_list;
  unsigned int  read_idx;
  unsigned int  count = 0;
  unsigned long flags;

  spin_lock_irqsave(&channel->lock, flags);

  if (channel->pending == 0)
  {
    log_debug("no packet pending\n");
    goto cleanup;
  }

  // read actual read index
  read_idx = readb(channel->bar + GDMA_SRC_Q_RD_IDX);

  if (read_idx >= QUEUE_SIZE)
  {
    log_error("channel#%d: invalid read index: %d\n", dp->chnid, read_idx);
    goto cleanup;
  }

  while (channel->read_idx != read_idx)
  {
    unsigned int ofs;
    union packet_info_t info;
    struct cs8_packet_t *pkt;

    ofs = GDMA_SRC_Q_ADDR(channel->read_idx);
    info.value = readq(channel->bar + ofs + 8);

    if (info.pid >= MAX_PACKETS_TX)
    {
      log_error("channel#%d: invalid packet: pid=%d, wlen=%d, user=0x%08x\n", dp->chnid, info.pid, info.wlen, info.user);
      goto skip;
    }

    pkt = channel->packets + info.pid;

#ifdef USE_STREAMING_DMA
    dma_sync_single_for_cpu(&dp->csdev.pci_dev->dev, pkt->phys_addr, pkt->size, DMA_TO_DEVICE);
    // CPU now may access packet buffer
#endif

    // free packet
    cs8_pkt_put_last(free_list, pkt);

    if (info.last && info.msg_idx != MSG_IDX_NET)
    {
      // last packet of message
      struct cs8_message_t *msg = dp->messages + info.msg_idx;

      if (info.msg_idx >= MaxMessages)
      {
        log_error("channel#%d: invalid message index: %d\n", dp->chnid, info.msg_idx);
        goto skip;
      }

      msg->state = MSG_STATE_SEND_DONE;

    //if (waitqueue_active(&msg->tx_event)) ...
      smp_wmb();
      wake_up_interruptible(&msg->tx_event);
    }

skip:
    // next
    channel->read_idx = (channel->read_idx + 1) & (QUEUE_SIZE - 1);
    channel->pending--;

    count++;
  }

  channel->pkt_done += count;

  // statistics
  if (count < QUEUE_SIZE)
  {
    channel->pkt_count[count]++;
  }
  else
  {
    log_warning("packet count: %d\n", count);
  }
cleanup:
  spin_unlock_irqrestore(&channel->lock, flags);

  if (dp->netdev)
  {
    if (netif_queue_stopped(dp->netdev))
    {
      netif_wake_queue(dp->netdev);
    }
  }
}

/******************************************************************************
 *
 * cs8_channel_tx_isr
 *
 ******************************************************************************/
static irqreturn_t cs8_channel_tx_isr(int irq, struct cs8_device_t *dp)
{
  struct cs8_channel_t *channel = &dp->tx_channel;
  tasklet_schedule(&channel->task);
  dp->irq_cnt[0]++;
  return IRQ_HANDLED;
}

/******************************************************************************
 *
 * cs8_channel_tx_isr_shared
 *
 ******************************************************************************/
static irqreturn_t cs8_channel_tx_isr_shared(int irq, struct cs8_device_t *parent)
{
  irqreturn_t ret = IRQ_NONE;
  u32 status;

  while ((status = readl(parent->bar + GDMA_IRQ_QUEUE_SRC)) != 0xFFFFFFFF)
  {
    struct cs8_device_t *dp;

    int chnid = (status & GDMA_IRQ_CHNU_MASK) >> GDMA_IRQ_CHNU_SHIFT;
    int count = (status & GDMA_IRQ_QLEL_MASK) >> GDMA_IRQ_QLEL_SHIFT;

    if ((dp = parent->devices[chnid]) != NULL)
    {
      ret = cs8_channel_tx_isr(irq, dp);
    }

    if (count == 0) break;
  }

  if (ret == IRQ_NONE) parent->irq_ghost[0]++;
  else                 parent->irq_shared[0]++;

  return ret;
}

/******************************************************************************
 *
 * cs8_channel_rx_task
 *
 *****************************************************************************/
static void cs8_channel_rx_task(unsigned long param)
{
  int err;
  struct cs8_device_t *dp = (struct cs8_device_t *)param;
  struct cs8_channel_t *channel = &dp->rx_channel;
  unsigned int  read_idx;
  unsigned int  count = 0;
  unsigned long flags;

  spin_lock_irqsave(&channel->lock, flags);

  if (channel->pending == 0)
  {
    log_trace("no packet pending\n");
    goto cleanup;
  }

  // read actual read index
  read_idx = readb(channel->bar + GDMA_DST_Q_RD_IDX);

  if (read_idx >= QUEUE_SIZE)
  {
    log_error("channel#%d: invalid read index: %d\n", dp->chnid, read_idx);
    goto cleanup;
  }

  while (channel->read_idx != read_idx)
  {
    struct cs8_packet_t *pkt;
    struct cs8_message_t *msg;
    union packet_info_t info;
    unsigned int ofs;

    ofs = GDMA_DST_Q_ADDR(channel->read_idx);
    info.value = readq(channel->bar + ofs + 8);

    if (info.pid >= channel->max_packets)
    {
      log_error("channel#%d: invalid packet: pid=%d, wlen=%d, user=0x%08x\n", dp->chnid, info.pid, info.wlen, info.user);
      goto skip;
    }

    pkt = channel->packets + info.pid;
    pkt->info.value = info.value;

#ifdef USE_STREAMING_DMA
    dma_sync_single_for_cpu(&dp->csdev.pci_dev->dev, pkt->phys_addr, info.wlen*8, DMA_FROM_DEVICE);
    // CPU now may access packet buffer
#endif

    if (  info.msg_idx == MSG_IDX_NET
       && dp->netdev
       )
    {
      int count = info.wlen * 8;
      struct sk_buff *skb = netdev_alloc_skb(dp->netdev, count);
      skb_copy_to_linear_data(skb, pkt->base_ptr, count);
      skb_put(skb, count);
      skb->ip_summed = CHECKSUM_NONE;
      skb->protocol = eth_type_trans(skb, dp->netdev);
      netif_rx(skb);

      dp->rx_stats.packets++;
      dp->rx_stats.bytes += count;

      log_xtrace("%recv", pkt->base_ptr, count);

      cs8_pkt_put_last(&channel->free_list, pkt);
      goto skip;
    }

    // get message
    if (info.msg_idx >= MaxMessages)
    {
      log_error("channel#%d: invalid message index: %d\n", dp->chnid, info.msg_idx);
      cs8_pkt_put_last(&channel->free_list, pkt);
      goto skip;
    }

    msg = dp->messages + info.msg_idx;

    if (  msg->state == MSG_STATE_FREE
       || msg->flags & (MSG_FLAG_CANCEL|MSG_FLAG_EXPIRED)
       )
    {
      // message was cancelled or is expired
      log_trace("channel#%d: got packet belonging to withdrawned message %d [%x]\n", dp->chnid, info.msg_idx, msg->flags);
      cs8_pkt_put_last(&channel->free_list, pkt);
      goto skip;
    }

    // append packet to message
    cs8_pkt_put_last(&msg->rx_list, pkt);

    smp_wmb();

    if (info.seq_cnt == 0)
    {
      // start of message
      wake_up_interruptible(&msg->rx_event);
    }
    else
    {
      wake_up(&msg->rx_event);
    }

    //wake_up_all(&msg->rx_event);

skip:
    // next
    channel->read_idx = (channel->read_idx + 1) & (QUEUE_SIZE - 1);
    channel->pending--;

    count++;

    // fetch next packet
    if ((err = cs8_pkt_get_first(&channel->free_list, &pkt, NULL)) != 0)
    {
      log_error("channel#%d: cs8_pkt_get_first returned: %d\n", dp->chnid, err);
      continue;
    }

    if ((err = cs8_channel_pkt_fetch_unlocked(dp, channel, pkt)) != 0)
    {
      log_error("channel#%d: cs8_channel_pkt_fetch_unlocked returned: %d\n", dp->chnid, err);
      cs8_pkt_put_last(&channel->free_list, pkt);
      continue;
    }
  }

  channel->pkt_done += count;

  // statistics
  if (count < QUEUE_SIZE)
  {
    channel->pkt_count[count]++;
  }
  else
  {
    log_warning("channel#%d: packet count: %d\n", dp->chnid, count);
  }
cleanup:
  spin_unlock_irqrestore(&channel->lock, flags);
}

/******************************************************************************
 *
 * cs8_channel_rx_isr
 *
 ******************************************************************************/
static irqreturn_t cs8_channel_rx_isr(int irq, struct cs8_device_t *dp)
{
  struct cs8_channel_t *channel = &dp->rx_channel;
  tasklet_schedule(&channel->task);
  dp->irq_cnt[1]++;
  return IRQ_HANDLED;
}

/******************************************************************************
 *
 * cs8_channel_rx_isr_shared
 *
 ******************************************************************************/
static irqreturn_t cs8_channel_rx_isr_shared(int irq, struct cs8_device_t *parent)
{
  irqreturn_t ret = IRQ_NONE;
  u32 status;

  while ((status = readl(parent->bar + GDMA_IRQ_QUEUE_DST)) != 0xFFFFFFFF)
  {
    struct cs8_device_t *dp;

    int chnid = (status & GDMA_IRQ_CHNU_MASK) >> GDMA_IRQ_CHNU_SHIFT;
    int count = (status & GDMA_IRQ_QLEL_MASK) >> GDMA_IRQ_QLEL_SHIFT;

    if ((dp = parent->devices[chnid]) != NULL)
    {
      ret = cs8_channel_rx_isr(irq, dp);
    }

    if (count == 0) break;
  }

  if (ret == IRQ_NONE) parent->irq_ghost[1]++;
  else                 parent->irq_shared[1]++;

  return ret;
}

/******************************************************************************
 *
 * cs8_mailbox_send
 *
 *****************************************************************************/
static void cs8_mailbox_send_unlocked(struct cs8_device_t *dp, u8 type, u8 addr, u16 message)
{
  union cs8_spm_t spm = { .message=message, .addr=addr, .type=type };

  writel(spm.value, dp->bar + GDMA_MAILBOX_OUT);
}

static void cs8_mailbox_send(struct cs8_device_t *dp, u8 type, u8 addr, u16 message)
{
  struct cs8_mailbox_t *mailbox = &dp->mailbox;
  unsigned long flags;

  spin_lock_irqsave(&mailbox->lock, flags);
  cs8_mailbox_send_unlocked(dp, type, addr, message);
  spin_unlock_irqrestore(&mailbox->lock, flags);
}

/******************************************************************************
 *
 * cs8_mailbox_task
 *
 *****************************************************************************/
static void cs8_mailbox_task(unsigned long param)
{
  struct cs8_device_t   *dp = (struct cs8_device_t *)param;
  struct cs8_mailbox_t  *mailbox = &dp->mailbox;
  unsigned long         flags;
  union cs8_spm_t       spm;
  int                   err;

  spin_lock_irqsave(&mailbox->lock, flags);

  // read mailbox
  spm.value = readl(dp->bar + GDMA_MAILBOX_IN);

  log_debug("channel#%d: got spm: %08x (addr=%d, message=%04x)\n", dp->chnid, spm.value, spm.addr, spm.message);

  if (spm.addr != dp->chnid)
  {
    log_error("channel#%d: invalid spm address: %d\n", dp->chnid, spm.addr);
    goto cleanup;
  }

  switch (spm.type)
  {
    case SPM_TYPE_DISABLED:
      log_info("channel#%d: was disabled\n", dp->chnid);

      if (dp->netdev)
      {
        netif_carrier_off(dp->netdev);
        log_info("%s is down\n", dp->netdev->name);
      }

      cs8_device_reset(dp);
      cs8_device_init(dp);

      cs8_mailbox_send_unlocked(dp, SPM_TYPE_INITIALIZED, dp->chnid, dp->flags);
      break;

    case SPM_TYPE_INITIALIZED:
      log_trace("channel#%d: was initialized [%x]\n", dp->chnid, spm.message);

      if (dp->state & DEVICE_STATE_RECOVER)
      {
        // device has recovered from reboot
        dp->state &= ~DEVICE_STATE_RECOVER;
        cs8_mailbox_send_unlocked(dp, SPM_TYPE_INITIALIZED, dp->chnid, dp->flags);
        break;
      }

      if (dp->state != DEVICE_STATE_INITIALIZED)
      {
        // device was rebootet (PS-only reset)
        log_warning("channel#%d: unexpected state: %s\n", dp->chnid, DeviceStates[dp->state & 3]);

        if (spm.message & DEVICE_FLAG_STARTUP)
        {
          struct cs8_device_t *parent = dp->parent;

          dp->state = DEVICE_STATE_OFFLINE;
          cs8_messages_cleanup(dp);

          if (parent->state != DEVICE_STATE_OFFLINE)
          {
            if (parent->netdev)
            {
              netif_carrier_off(parent->netdev);
              log_info("%s is down\n", parent->netdev->name);
            }

            parent->state = DEVICE_STATE_OFFLINE;
            cs8_messages_cleanup(parent);
          }
        }
        break;
      }

      if (spm.message & DEVICE_FLAG_STARTUP)
      {
        cs8_mailbox_send_unlocked(dp, SPM_TYPE_INITIALIZED, dp->chnid, dp->flags);
        break;
      }

      dp->flags = spm.message & (DEVICE_FLAG_CRC | DEVICE_FLAG_NET);

      dp->state = DEVICE_STATE_ENABLED;

      cs8_mailbox_send_unlocked(dp, SPM_TYPE_ENABLED, dp->chnid, dp->flags);
      break;

    case SPM_TYPE_ENABLED:
      log_info("channel#%d: was enabled [%x]\n", dp->chnid, spm.message);

      if (dp->state != DEVICE_STATE_ENABLED)
      {
        log_warning("channel#%d: unexpected state: %s\n", dp->chnid, DeviceStates[dp->state & 3]);
        break;
      }

      dp->relaunch++;

      // fetch packets in advance
      if ((err = cs8_channel_prefetch(dp)) != 0)
      {
        log_error("channel#%d: cs8_channel_prefetch returned: %d\n", dp->chnid, err);
      }

      if (waitqueue_active(&dp->spm_event))
      {
        dp->spm_message = spm.value;

        smp_wmb();
        wake_up_interruptible(&dp->spm_event);
      }

      if (  dp->netdev
         && dp->flags & DEVICE_FLAG_NET
         )
      {
        netif_carrier_on(dp->netdev);
        log_info("%s is up\n", dp->netdev->name);
      }

      if (dp->type == DEVICE_TYPE_PARENT)
      {
        u16 version = ((u16)VERSION_MAJOR) << 8 | VERSION_MINOR;
        cs8_mailbox_send_unlocked(dp, SPM_TYPE_VERSION, dp->chnid, version);
      }
      break;

    case SPM_TYPE_VERSION:
      log_trace("channel#%d: got version spm: 0x%04x\n", dp->chnid, spm.message);
      dp->ddversion = spm.message;
      break;

    case SPM_TYPE_USER:
      log_trace("channel#%d: got user spm: 0x%04x\n", dp->chnid, spm.message);

      dp->spm_message = spm.message;

      smp_wmb();
      wake_up_interruptible(&dp->spm_event);
      break;

    default:
      log_error("channel#%d: got unknown spm: 0x%08x\n", dp->chnid, spm.value);
  }

cleanup:
  spin_unlock_irqrestore(&mailbox->lock, flags);
}

/******************************************************************************
 *
 * cs8_mailbox_isr
 *
 ******************************************************************************/
static irqreturn_t cs8_mailbox_isr(int irq, struct cs8_device_t *dp)
{
  struct cs8_mailbox_t *mailbox = &dp->mailbox;
  tasklet_schedule(&mailbox->task);
  dp->irq_cnt[2]++;
  return IRQ_HANDLED;
}

/******************************************************************************
 *
 * cs8_mailbox_shared_isr
 *
 ******************************************************************************/
static irqreturn_t cs8_mailbox_shared_isr(int irq, struct cs8_device_t *parent)
{
  irqreturn_t ret = IRQ_NONE;
  u32 status;

  while ((status = readl(parent->bar + GDMA_IRQ_QUEUE_MAILBOX)) != 0xFFFFFFFF)
  {
    struct cs8_device_t *dp;

    int chnid = (status & GDMA_IRQ_CHNU_MASK) >> GDMA_IRQ_CHNU_SHIFT;
    int count = (status & GDMA_IRQ_QLEL_MASK) >> GDMA_IRQ_QLEL_SHIFT;

    if ((dp = parent->devices[chnid]) != NULL)
    {
      ret = cs8_mailbox_isr(irq, dp);
    }

    if (count == 0) break;
  }

  if (ret == IRQ_NONE) parent->irq_ghost[2]++;
  else                 parent->irq_shared[2]++;

  return ret;
}

static const irq_handler_t irq_handler_channel[3] =
{
  (irq_handler_t)cs8_channel_tx_isr,
  (irq_handler_t)cs8_channel_rx_isr,
  (irq_handler_t)cs8_mailbox_isr
};

static const irq_handler_t irq_handler_shared[3] =
{
  (irq_handler_t)cs8_channel_tx_isr_shared,
  (irq_handler_t)cs8_channel_rx_isr_shared,
  (irq_handler_t)cs8_mailbox_shared_isr
};

/******************************************************************************
 *
 * cs8_irq_request
 *
 *****************************************************************************/
static int cs8_irq_request(struct cs8_device_t *dp)
{
  int             err = 0;
  struct pci_dev  *pci_dev;
  struct device   *dev;
  int             cnt;
  int             i;
  unsigned long   flags = 0; //IRQF_SHARED;

  const irq_handler_t *irq_handler;

  if (  dp == NULL
     || (pci_dev = dp->csdev.pci_dev) == NULL
     || (dev = &pci_dev->dev) == NULL
     )
    return -EINVAL;

  irq_handler = (dp->irq_mode == IRQ_MODE_SHARED) ? irq_handler_shared : irq_handler_channel;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,8,0))
  if ((cnt = pci_alloc_irq_vectors(pci_dev, 3, 3, PCI_IRQ_MSIX | PCI_IRQ_MSI)) != 3)
  {
    log_error("pci_alloc_irq_vectors(3,3,MSI/X) returned: %d\n", cnt);
    return (cnt < 0) ? cnt : -ENOSPC;
  }
#else
  for (i=0; i < 3; i++)
  {
    dp->msix[i].entry = i;
    dp->msix[i].vector = 0;
  }

  if ((cnt = pci_enable_msix_range(pci_dev, dp->msix, 3, 3)) != 3)
  {
    log_error("pci_alloc_irq_vectors(3,3,MSI/X) returned: %d\n", cnt);
    return (cnt < 0) ? cnt : -ENOSPC;
  }
#endif

  for (i=0; i<3; i++)
  {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,8,0))
    int irq = pci_irq_vector(pci_dev, i);
#else
    int irq = dp->msix[i].vector;
#endif

    if (irq <= 0) CLEANUP(-ENXIO);

    if ((err = devm_request_irq(dev, irq, irq_handler[i], flags, IrqNames[i], dp)) != 0)
    {
      log_error("devm_request_irq(%d) for %s returned: %d\n", irq, IrqNames[i], err);
      goto cleanup;
    }

    dp->irq_vector[i] = irq;

    log_trace("successfully initialized %s irq: %d\n", IrqNames[i], irq);
  }

  return 0;

cleanup:
  return err;
}

/******************************************************************************
 *
 * cs8_irq_free
 *
 *****************************************************************************/
static void cs8_irq_free(struct cs8_device_t *dp)
{
  struct pci_dev  *pci_dev;
  struct device   *dev;
  int i;

  if (  dp == NULL
     || (pci_dev = dp->csdev.pci_dev) == NULL
     || (dev = &pci_dev->dev) == NULL
     )
    return;

  for (i=0; i<3; i++)
  {
    int irq = dp->irq_vector[i];

    if (irq <= 0) continue;

    devm_free_irq(dev, irq, dp);

    dp->irq_vector[i] = 0;

    log_trace("successfully freed %s irq: %d\n", IrqNames[i], irq);
  }

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,8,0))
  pci_free_irq_vectors(pci_dev);
#else
  pci_disable_msix(pci_dev);
#endif
}

/******************************************************************************
 *
 * cs8_msg_info
 *
 *****************************************************************************/
static int cs8_msg_info(struct cs8_message_t *msg, char *p, int max)
{
  struct cs8_list_t *list = &msg->rx_list;
  unsigned long flags;
  struct cs8_packet_t *pkt;
  unsigned long exp = timer_pending(&msg->timer) ? msg->timer.expires - jiffies : 0;
  int len = 0;

  len += scnprintf(p+len, max-len, " message %-2d      %s [%02x] [%ld]\n", msg->idx, MsgStates[msg->state&7], msg->flags, exp);

  spin_lock_irqsave(&list->lock, flags);

  if (!list_empty(&list->head))
  {
    len += scnprintf(p+len, max-len, "  rx             %2d <%2d..%2d>", list->count, list->count_min, list->count_max);

    list_for_each_entry(pkt, &list->head, list)
    {
      len += scnprintf(p+len, max-len, " %02x", pkt->idx);
    }
    p[len++] = '\n';
  }

  spin_unlock_irqrestore(&list->lock, flags);

  p[len] = 0;
  return len;
}

/******************************************************************************
 *
 * cs8_msg_free
 *
 *****************************************************************************/
static int cs8_msg_free(struct cs8_message_t *msg)
{
  struct cs8_device_t   *dp = msg->dp;
  struct cs8_channel_t  *channel = &dp->rx_channel;
  struct cs8_packet_t   *pkt;
  int done = 0;

  // free all packets received for now
  while (cs8_pkt_get_first(&msg->rx_list, &pkt, NULL) == 0)
  {
    done |= pkt->info.last;
    cs8_pkt_put_last(&channel->free_list, pkt);
  }

  return done;
}

/******************************************************************************
 *
 * cs8_msg_lock
 *
 *****************************************************************************/
static inline int cs8_msg_lock(struct cs8_device_t *dp, struct cs8_message_t **pp_msg)
{
  struct cs8_list_t *list = &dp->free_list;
  struct cs8_message_t *msg;
  unsigned long flags;

  spin_lock_irqsave(&list->lock, flags);

  if ((msg = list_first_entry_or_null(&list->head, struct cs8_message_t, list)) == NULL)
  {
    spin_unlock_irqrestore(&list->lock, flags);
    *pp_msg = NULL;
    return -EBUSY;
  }

  list_del_init(&msg->list);

  list->count--;
  if (list->count < list->count_min) list->count_min = list->count;

  spin_unlock_irqrestore(&list->lock, flags);

  if (!list_empty(&msg->rx_list.head))
  {
    // free orphaned packets
    cs8_msg_free(msg);
  }

  msg->state = MSG_STATE_LOCKED;
  msg->flags = 0;

  *pp_msg = msg;
  return 0;
}

/******************************************************************************
 *
 * cs8_msg_unlock
 *
 *****************************************************************************/
static inline void cs8_msg_unlock(struct cs8_message_t *msg)
{
  struct cs8_device_t *dp = msg->dp;
  struct cs8_list_t *list = &dp->free_list;
  unsigned long flags;

  del_timer(&msg->timer);
  msg->state = MSG_STATE_FREE;
//msg->flags = 0;
  msg->session = NULL;

  spin_lock_irqsave(&list->lock, flags);

  list_add_tail(&msg->list, &list->head);

  list->count++;
  if (list->count > list->count_max) list->count_max = list->count;

  spin_unlock_irqrestore(&list->lock, flags);
}

/******************************************************************************
 *
 * cs8_msg_timeout_set
 *
 *****************************************************************************/
static inline void cs8_msg_timeout_set(struct cs8_message_t *msg, unsigned int timeout)
{
  mod_timer(&msg->timer, jiffies + timeout);
}

/******************************************************************************
 *
 * cs8_msg_expired
 *
 *****************************************************************************/
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,15,0))
static void cs8_msg_expired(struct timer_list *tmr)
{
  struct cs8_message_t *msg = from_timer(msg, tmr, timer);
#else
static void cs8_msg_expired(unsigned long data)
{
  struct cs8_message_t *msg = (struct cs8_message_t *)data;
#endif

  if (msg->state != MSG_STATE_FREE)
  {
    const char *txt[] = { "pending", "done" };
    int done = cs8_msg_free(msg);

    msg->flags |= MSG_FLAG_EXPIRED;

    cs8_msg_unlock(msg);  // release message even it is not complete

    log_info("channel#%d: message %d has expired [%s]\n", msg->dp->chnid, msg->idx, txt[done&1]);
  }
  else
  {
    log_info("channel#%d: message %d has already been released\n", msg->dp->chnid, msg->idx);
  }
}

/******************************************************************************
 *
 * cs8_msg_send
 *
 *****************************************************************************/
static int cs8_msg_send(struct cs8_device_t *dp, struct cs8_message_t *msg, struct pci_request_t *request)
{
  int                   err         = 0;
  struct cs8_channel_t  *channel    = &dp->tx_channel;
  struct cs8_list_t     *free_list  = &channel->free_list;
  unsigned int          rlen        = request->count;
  unsigned char         *p_out_buf  = (unsigned char*)request->data;
  unsigned int          out_len     = 0;
  unsigned int          seq_cnt     = 0;
  unsigned int          last        = 0;
  unsigned int          crc         = 0;
  struct cs8_packet_t   *pkt;
  unsigned char         *p_hdr;

  do
  {
    unsigned int count = MIN(rlen, MAX_PACKET_SIZE-8);

    if ((err = cs8_pkt_get_first_timeout(free_list, HZ, &pkt)) != 0)
    {
      log_error("channel#%d: cs8_pkt_get_first_timeout returned: %d\n", dp->chnid, err);
      goto cleanup;
    }

    p_hdr = pkt->base_ptr;

    // copy user data to packet buffer
    if (copy_from_user(p_hdr+8, p_out_buf, count) != 0)
    {
      cs8_pkt_put_last(free_list, pkt);
      CLEANUP(-EFAULT);
    }

    if (dp->flags & DEVICE_FLAG_CRC)
      crc = crc16_calc(crc, p_hdr+8, count);

    // packet header
    p_hdr[0] = 0;
    p_hdr[1] = msg->idx;
    p_hdr[2] = seq_cnt;
    p_hdr[3] = crc >> 8;
    p_hdr[4] = crc;
    p_hdr[5] = rlen >> 16;
    p_hdr[6] = rlen >> 8;
    p_hdr[7] = rlen;

    rlen -= count;
    if (rlen == 0) last = 1;

    pkt->info.pid     = pkt->idx;
    pkt->info.wlen    = DMA_WLEN(8 + count);
    pkt->info.addr    = dp->chnid;
    pkt->info.msg_idx = msg->idx;
    pkt->info.seq_cnt = seq_cnt;
    pkt->info.last    = last;

    if ((err = cs8_channel_pkt_submit(dp, channel, pkt)) != 0)
    {
      cs8_pkt_put_last(free_list, pkt);
      log_error("channel#%d: cs8_channel_pkt_submit(%d) returned: %d\n", dp->chnid, pkt->idx, err);
      goto cleanup;
    }

    seq_cnt++;

    out_len   += count;
    p_out_buf += count;
  }
  while (rlen != 0);

  // wait until message has been sent
  if (request->timeout != 0)
  {
    long timeout = msecs_to_jiffies(request->timeout);

    timeout = wait_event_interruptible_timeout(msg->tx_event,
                                               (  msg->state == MSG_STATE_SEND_DONE
                                               || dp->state != DEVICE_STATE_ENABLED
                                               ),
                                               timeout);

    if (dp->state != DEVICE_STATE_ENABLED) CLEANUP(-EPIPE);

    if (timeout <= 0)
    {
      log_error("channel#%d: wait_event_interruptible_timeout(%d,%d) returned: %ld\n", dp->chnid, msg->idx, request->timeout, timeout);
      CLEANUP((timeout < 0) ? timeout : -ETIMEDOUT);
    }
  }

cleanup:
  cs8_msg_timeout_set(msg, MessageExpiry); // set short message expiration time

  channel->error = err;

  log_back("%2d | out_len: %d [%d]", dp->chnid, out_len, request->count);

  request->count = out_len;
  return err;
}

/******************************************************************************
 *
 * cs8_msg_get_request
 *
 *****************************************************************************/
static int cs8_msg_get_request(struct cs8_device_t *dp, struct cs8_message_t *msg, struct pci_request_t *request)
{
  int                   err = 0;
  struct cs8_channel_t  *channel = &dp->rx_channel;
  struct cs8_packet_t   *pkt;
  unsigned char         *p_hdr;
  struct cs8_list_t     *rx_list = &msg->rx_list;
  long                  timeout = 0;

  // wait on first packet
  while ((pkt = list_first_entry_or_null(&rx_list->head, struct cs8_packet_t, list)) == NULL)
  {
    if (timeout == 0)
    {
      timeout = msecs_to_jiffies(request->timeout);
      cs8_msg_timeout_set(msg, timeout+MessageExpiry);
    }

    timeout = wait_event_interruptible_timeout(msg->rx_event,
                                               (  !list_empty(&rx_list->head)
                                               || dp->state != DEVICE_STATE_ENABLED
                                               ),
                                               timeout);

    if (dp->state != DEVICE_STATE_ENABLED) CLEANUP(-EPIPE);

    if (timeout <= 0)
    {
      log_trace("channel#%d: wait_event_interruptible_timeout(%d,%d) returned: %ld\n", dp->chnid, msg->idx, request->timeout, timeout);
      CLEANUP((timeout < 0) ? timeout : -ETIMEDOUT);
    }
  }

  // check packet count
  if (  pkt->info.wlen == 0
     || pkt->info.wlen > MAX_PACKET_SIZE/8
     )
  {
    log_error("channel#%d: invalid packet length: wlen=%d, pid=%d, user=0x%08x\n", dp->chnid, pkt->info.wlen, pkt->info.pid, pkt->info.user);
    CLEANUP(-EMSGSIZE);
  }

  // check address
  if (pkt->info.addr != dp->chnid)
  {
    log_error("channel#%d: invalid packet address: %d, expected: %d\n", dp->chnid, pkt->info.addr, dp->chnid);
    CLEANUP(-EFAULT);
  }

  // check message index
  if (pkt->info.msg_idx != msg->idx)
  {
    log_error("channel#%d: invalid packet message index: %d, expected: %d\n", dp->chnid, pkt->info.msg_idx, msg->idx);
    CLEANUP(-EFAULT);
  }

  // check sequence counter
  if (pkt->info.seq_cnt != 0)
  {
    log_error("channel#%d: invalid packet sequence counter: %d\n", dp->chnid, pkt->info.seq_cnt);
    CLEANUP(-EPROTO);
  }

  // check packet header
  p_hdr = pkt->base_ptr;

  if (p_hdr[0] != 0)
  {
    if (p_hdr[0] == 0xE0)
    {
      msg->type = MSG_TYPE_LOG;
    }
    else
    {
      err = p_hdr[0];
      log_error("channel#%d: request rejected: pid=%d, errno=%d\n", dp->chnid, pkt->idx, err);
      CLEANUP(-err);
    }
  }

  if (p_hdr[1] != msg->idx)
  {
    log_error("channel#%d: invalid message index: %d, expected: %d\n", dp->chnid, p_hdr[1], msg->idx);
    CLEANUP(-EPROTO);
  }

  if (p_hdr[2] != 0)
  {
    log_error("channel#%d: invalid sequence counter: %d\n", dp->chnid, p_hdr[2]);
    CLEANUP(-EPROTO);
  }

  // request->count = load_int3_be(p_hdr+5);
  msg->count = load_int3_be(p_hdr+5);

  // set short message expiration time
  cs8_msg_timeout_set(msg, MessageExpiry);

cleanup:
  channel->error = err;

  log_back("%2d | count: %d", dp->chnid, msg->count);

  return err;
}

/******************************************************************************
 *
 * cs8_msg_recv
 *
 *****************************************************************************/
static int cs8_msg_recv(struct cs8_device_t *dp, struct cs8_message_t *msg, struct pci_request_t *request)
{
  int                   err         = 0;
  struct cs8_channel_t  *channel    = &dp->rx_channel;
  struct cs8_list_t     *free_list  = &channel->free_list;
  struct cs8_list_t     *rx_list    = &msg->rx_list;
  unsigned char         *p_in_buf   = (unsigned char*)request->data;
  unsigned int          in_len      = 0;
  unsigned int          seq_cnt     = 0;
  long                  timeout     = 3*HZ;
  unsigned int          crc_calc    = 0;
  unsigned int          crc_last    = 0;
  unsigned int          rlen        = MIN(request->count, msg->count);

  do
  {
    struct cs8_packet_t *pkt;
    unsigned int        count;
    unsigned char       *p_hdr;

    while (cs8_pkt_get_first(rx_list, &pkt, NULL) != 0)
    {
      timeout = wait_event_timeout(msg->rx_event,
                                   (  !list_empty(&rx_list->head)
                                   || dp->state != DEVICE_STATE_ENABLED
                                   ),
                                   timeout);

      if (dp->state != DEVICE_STATE_ENABLED) CLEANUP(-EPIPE);

      if (timeout <= 0)
      {
        log_error("channel#%d: wait_event_timeout(%d,3000) returned: %ld\n", dp->chnid, msg->idx, timeout);
        CLEANUP((timeout < 0) ? timeout : -ETIMEDOUT);
      }
    }

    // check packet count
    if (  pkt->info.wlen == 0
       || pkt->info.wlen > MAX_PACKET_SIZE/8
       )
    {
      log_error("channel#%d: invalid packet length: wlen=%d, pid=%d, user=0x%08x\n", dp->chnid, pkt->info.wlen, pkt->info.pid, pkt->info.user);
      cs8_pkt_put_last(free_list, pkt);
      CLEANUP(-EMSGSIZE);
    }

    // check address
    if (pkt->info.addr != dp->chnid)
    {
      log_error("channel#%d: invalid packet address: %d, expected: %d\n", dp->chnid, pkt->info.addr, dp->chnid);
      cs8_pkt_put_last(free_list, pkt);
      CLEANUP(-EFAULT);
    }

    // check message index
    if (pkt->info.msg_idx != msg->idx)
    {
      log_error("channel#%d: invalid packet message index: %d, expected: %d\n", dp->chnid, pkt->info.msg_idx, msg->idx);
      cs8_pkt_put_last(free_list, pkt);
      CLEANUP(-EFAULT);
    }

    // check sequence counter
    if (pkt->info.seq_cnt != seq_cnt)
    {
      log_error("channel#%d: invalid packet sequence counter: %d, expected: %d\n", dp->chnid, pkt->info.seq_cnt, seq_cnt);
      cs8_pkt_put_last(free_list, pkt);
      CLEANUP(-EPROTO);
    }

    // check packet header
    p_hdr = pkt->base_ptr;

    if (p_hdr[0] != 0)
    {
      if (p_hdr[0] == 0xE0)
      {
        msg->type = MSG_TYPE_LOG;
      }
      else
      {
        err = p_hdr[0];
        log_error("channel#%d: request rejected at %d/%d: pid=%d, errno=%d\n", dp->chnid, in_len, request->count, pkt->idx, err);
        cs8_pkt_put_last(free_list, pkt);
        CLEANUP(-err);
      }
    }

    if (p_hdr[1] != msg->idx)
    {
      log_error("channel#%d: invalid message index: %d, expected: %d\n", dp->chnid, p_hdr[1], msg->idx);
      cs8_pkt_put_last(free_list, pkt);
      CLEANUP(-EPROTO);
    }

    if (p_hdr[2] != seq_cnt)
    {
      log_error("channel#%d: invalid sequence counter: %d, expected: %d\n", dp->chnid, p_hdr[2], seq_cnt);
      cs8_pkt_put_last(free_list, pkt);
      CLEANUP(-EPROTO);
    }

    /* if (seq_cnt == 0)
    {
      unsigned int msg_len = load_int3_be(p_hdr+5);
      if (rlen > msg_len) rlen = msg_len;
    } */

    count = (pkt->info.wlen * 8) - 8;
    if (count > rlen) count = rlen;

    if (dp->flags & DEVICE_FLAG_CRC)
    {
      crc_last = (p_hdr[3] << 8) + p_hdr[4];
      crc_calc = crc16_calc(crc_calc, p_hdr+8, count);
    }

    // copy packet buffer to user data
    if (copy_to_user(p_in_buf, p_hdr+8, count) != 0) err = -EFAULT;

    cs8_pkt_put_last(free_list, pkt);

    seq_cnt++;

    in_len   += count;
    p_in_buf += count;
    rlen     -= count;
  }
  while (rlen != 0);

  if (   dp->flags & DEVICE_FLAG_CRC
     &&  crc_calc != crc_last
     )
  {
    log_error("channel#%d: CRC mismatch: %04x, recalc := %04x\n", dp->chnid, crc_last, crc_calc);
    CLEANUP(-EILSEQ);
  }

cleanup:
  channel->error = err;

  log_back("%2d | in_len: %d [%d]", dp->chnid, in_len, request->count);

  request->count = in_len;
  return err;
}

/******************************************************************************
 *
 * cs8_msg_cancel
 *
 *****************************************************************************/
static int cs8_msg_cancel(struct cs8_message_t *msg)
{
  if (msg->state != MSG_STATE_FREE)
  {
    const char *txt[] = { "pending", "done" };
    int done;

    msg->flags |= MSG_FLAG_CANCEL;

    done = cs8_msg_free(msg);

    if (done)
    {
      cs8_msg_unlock(msg);
    }

    log_info("channel#%d: message %d was cancelled [%s]\n", msg->dp->chnid, msg->idx, txt[done & 1]);
  }
  else
  {
    log_warning("channel#%d: message %d has already been released\n", msg->dp->chnid, msg->idx);
  }

  return 0;
}

/******************************************************************************
 *
 * cs8_messages_info
 *
 *****************************************************************************/
static int cs8_messages_info(struct cs8_device_t *dp, char *p, int max)
{
  struct cs8_message_t *msg;
  struct cs8_list_t *free_list = &dp->free_list;
  int len = 0;
  int i;

  len += scnprintf(p+len, max-len, "Message pool     %2d <%2d..%2d>", free_list->count, free_list->count_min, free_list->count_max);

#ifdef DEBUG
  {
    unsigned long flags;

    spin_lock_irqsave(&free_list->lock, flags);

    list_for_each_entry(msg, &free_list->head, list)
    {
      len += scnprintf(p+len, max-len, " %02d", msg->idx);
    }

    spin_unlock_irqrestore(&free_list->lock, flags);
  }
#endif

  p[len++] = '\n';

  for (i=0; i<MaxMessages; i++)
  {
    msg = dp->messages + i;

    if (msg->rx_list.count == 0) continue;

    len += cs8_msg_info(msg, p+len, max-len);
  }

  p[len] = 0;
  return len;
}

/******************************************************************************
 *
 * cs8_messages_init
 *
 *****************************************************************************/
static int cs8_messages_init(struct cs8_device_t *dp)
{
  int i;

  cs8_list_init(&dp->free_list);

  for (i=0; i<MaxMessages; i++)
  {
    struct cs8_message_t *msg = dp->messages + i;

    memset(msg, 0, sizeof(struct cs8_message_t));

    msg->dp = dp;
    msg->idx = i;
    msg->mask = ((u64)1 << i);
    INIT_LIST_HEAD(&msg->list);

  //cs8_list_init(&msg->tx_list);
    cs8_list_init(&msg->rx_list);

    init_waitqueue_head(&msg->tx_event);
    init_waitqueue_head(&msg->rx_event);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,15,0))
    timer_setup(&msg->timer, cs8_msg_expired, 0);
#else
    init_timer(&msg->timer);
    setup_timer(&msg->timer, cs8_msg_expired, (unsigned long)msg);
#endif

    cs8_msg_unlock(msg);
  }

  dp->free_list.count_min = i;
  dp->free_list.count_max = i;

  return 0;
}

/******************************************************************************
 *
 * cs8_messages_cleanup
 *
 *****************************************************************************/
static int cs8_messages_cleanup(struct cs8_device_t *dp)
{
  int i;

  for (i=0; i<MaxMessages; i++)
  {
    struct cs8_message_t *msg = dp->messages + i;

    del_timer(&msg->timer);

    if (msg->state != MSG_STATE_FREE)
    {
      msg->flags |= MSG_FLAG_CANCEL;
      wake_up_all(&msg->rx_event);
    }
  }

  return 0;
}

/******************************************************************************
 *
 * cs8_device_healthy
 *
 *****************************************************************************/
static inline int cs8_device_healthy(struct cs8_device_t *dp)
{
  return readl(dp->bar + GDMA_HW_VERSION) != (u32)(-1);
}

/******************************************************************************
 *
 * cs8_device_info
 *
 *****************************************************************************/
static int cs8_device_info(struct cs8_device_t *dp, char *p, int max)
{
  int len = 0;
  u32 regval;
  struct pci_dev *pci_dev = dp->csdev.pci_dev;

  // device
  len += scnprintf(p+len, max-len, "Device           %s\n", dp->csdev.device_name);
  len += scnprintf(p+len, max-len, " model           %d\n", dp->csdev.model);
  len += scnprintf(p+len, max-len, " slot            %-16.16s\n", dp->csdev.slot);

  if (pci_dev->is_physfn)
  {
    len += scnprintf(p+len, max-len, " devfn           pf:%d\n", pci_dev->devfn);
  }
  else if (pci_dev->is_virtfn)
  {
    if (pci_dev->physfn != NULL)
      len += scnprintf(p+len, max-len, " devfn           vf:%d->pf:%d\n", pci_dev->devfn, pci_dev->physfn->devfn);
    else
      len += scnprintf(p+len, max-len, " devfn           vf:%d\n", pci_dev->devfn);
  }
  else
  {
    len += scnprintf(p+len, max-len, " devfn           %d\n", pci_dev->devfn);
  }

  len += scnprintf(p+len, max-len, " type            %s\n", DeviceTypes[dp->type&3]);
  len += scnprintf(p+len, max-len, " irq_mode        %s\n", IrqModes[dp->irq_mode&1]);
  len += scnprintf(p+len, max-len, " chnid           %d\n", dp->chnid);
  len += scnprintf(p+len, max-len, " state           %s\n", DeviceStates[dp->state & 3]);

  len += scnprintf(p+len, max-len, " flags           %04x [", dp->flags);
  if (dp->flags & DEVICE_FLAG_CRC) len += scnprintf(p+len, max-len, "crc ");
  if (dp->flags & DEVICE_FLAG_NET) len += scnprintf(p+len, max-len, "net");
  len += scnprintf(p+len, max-len, "]\n");

  if (dp->ddversion != 0)
  len += scnprintf(p+len, max-len, " ddversion       %d.%d.%x\n", (dp->ddversion & 0xF000) >> 12,
                                                                  (dp->ddversion & 0x0FF0) >> 4,
                                                                  (dp->ddversion & 0x000F) );
  if (dp->netdev)
  {
    bool present = netif_device_present(dp->netdev);
    bool carrier = netif_carrier_ok(dp->netdev);
    bool oper    = netif_oper_up(dp->netdev);
    bool queue   = netif_queue_stopped(dp->netdev);
    struct cs8_netdev *ndev = netdev_priv(dp->netdev);

  len += scnprintf(p+len, max-len, " network         device: %s, carrier: %s, operational: %s, queue: %s, used: %d\n",
                                                     present ? "present" : "removed",
                                                     carrier ? "ok" : "not ok",
                                                     oper    ? "yes" : "no",
                                                     queue   ? "stopped" : "running",
                                                     ndev->use_count);
  }

  len += scnprintf(p+len, max-len, " relaunch        %d\n", dp->relaunch);
  len += scnprintf(p+len, max-len, " lock count      %d\n", dp->sema.count);

  // Register
  len += scnprintf(p+len, max-len, "Register\n");
  len += scnprintf(p+len, max-len, " gdma version    %08x\n", readl(dp->bar + GDMA_VERSION));
  len += scnprintf(p+len, max-len, " fpga version    %08x\n", readl(dp->bar + GDMA_FPGA_VERSION));
  len += scnprintf(p+len, max-len, " hw version      %08x\n", readl(dp->bar + GDMA_HW_VERSION));
  len += scnprintf(p+len, max-len, " serial          %08x:%08x\n", readl(dp->bar + GDMA_SSN_HIGH),
                                                                  readl(dp->bar + GDMA_SSN_LOW));
  regval = readl(dp->bar + GDMA_TIMESTAMP);

  len += scnprintf(p+len, max-len, " timestamp       %08x", regval);
  if (cs_time_to_ascii != NULL)
    len += scnprintf(p+len, max-len, " [%s]", cs_time_to_ascii(regval));
  p[len++] = '\n';

  if (LogLevel >= LOG_LEVEL_TRACE)
  {
  len += scnprintf(p+len, max-len, " debug           %08x\n",    readl(dp->bar + GDMA_DEBUG));
  len += scnprintf(p+len, max-len, " error count     %08x\n",    readl(dp->bar + GDMA_ERROR_COUNT));
  len += scnprintf(p+len, max-len, " core control    %08x\n",    readl(dp->bar + GDMA_CORE_CONTROL));
  }

  len += scnprintf(p+len, max-len, " control         %08x\n",    readl(dp->bar + GDMA_CONTROL));

  len += scnprintf(p+len, max-len, " src_q_rd_idx    %2d [%2d]\n", readb(dp->bar + GDMA_SRC_Q_RD_IDX), dp->tx_channel.read_idx);
  len += scnprintf(p+len, max-len, " src_q_wr_idx    %2d [%2d]\n", readb(dp->bar + GDMA_SRC_Q_WR_IDX), dp->tx_channel.write_idx);
  len += scnprintf(p+len, max-len, " dst_q_rd_idx    %2d [%2d]\n", readb(dp->bar + GDMA_DST_Q_RD_IDX), dp->rx_channel.read_idx);
  len += scnprintf(p+len, max-len, " dst_q_wr_idx    %2d [%2d]\n", readb(dp->bar + GDMA_DST_Q_WR_IDX), dp->rx_channel.write_idx);

  len += scnprintf(p+len, max-len, " mailbox_in      %08x\n", readl(dp->bar + GDMA_MAILBOX_IN));
  len += scnprintf(p+len, max-len, " mailbox_out     %08x\n", readl(dp->bar + GDMA_MAILBOX_OUT));

  len += scnprintf(p+len, max-len, " irq regs        mask: %02x enbl: %02x stat: %02x test: %02x\n", readb(dp->bar + GDMA_IRQ_MASK),
                                                                                                     readb(dp->bar + GDMA_IRQ_ENABLE),
                                                                                                     readb(dp->bar + GDMA_IRQ_STATUS),
                                                                                                     readb(dp->bar + GDMA_IRQ_TEST));
  if (LogLevel >= LOG_LEVEL_TRACE)
  {
  len += scnprintf(p+len, max-len, " irq queue       %08x %08x %08x\n", readl(dp->bar + GDMA_IRQ_QUEUE_SRC),
                                                                        readl(dp->bar + GDMA_IRQ_QUEUE_DST),
                                                                        readl(dp->bar + GDMA_IRQ_QUEUE_MAILBOX));
  }

  len += scnprintf(p+len, max-len, "MSI-X Vect.-Tab  addr              data     mask\n");
  len += scnprintf(p+len, max-len, " tx              %08x:%08x %08x %08x\n", readl(dp->bar + GDMA_MSIX_IRQ_ADDR_SRC_HIGH),
                                                                             readl(dp->bar + GDMA_MSIX_IRQ_ADDR_SRC_LOW),
                                                                             readl(dp->bar + GDMA_MSIX_IRQ_DATA_SRC),
                                                                             readl(dp->bar + GDMA_MSIX_IRQ_MASK_SRC));

  len += scnprintf(p+len, max-len, " rx              %08x:%08x %08x %08x\n", readl(dp->bar + GDMA_MSIX_IRQ_ADDR_DST_HIGH),
                                                                             readl(dp->bar + GDMA_MSIX_IRQ_ADDR_DST_LOW),
                                                                             readl(dp->bar + GDMA_MSIX_IRQ_DATA_DST),
                                                                             readl(dp->bar + GDMA_MSIX_IRQ_MASK_DST));

  len += scnprintf(p+len, max-len, " mailbox         %08x:%08x %08x %08x\n", readl(dp->bar + GDMA_MSIX_IRQ_ADDR_MAIL_HIGH),
                                                                             readl(dp->bar + GDMA_MSIX_IRQ_ADDR_MAIL_LOW),
                                                                             readl(dp->bar + GDMA_MSIX_IRQ_DATA_MAIL),
                                                                             readl(dp->bar + GDMA_MSIX_IRQ_MASK_MAIL));
  // Statistics
  len += scnprintf(p+len, max-len, "IRQ-stats        tx         rx         mbx\n");
  len += scnprintf(p+len, max-len, " channel         %-10d %-10d %-10d\n", dp->irq_cnt[0], dp->irq_cnt[1], dp->irq_cnt[2]);

  if (  dp->irq_mode == IRQ_MODE_SHARED
     && dp->chnid == 0
     )
  {
  len += scnprintf(p+len, max-len, " shared          %-10d %-10d %-10d\n", dp->irq_shared[0], dp->irq_shared[1], dp->irq_shared[2]);
  len += scnprintf(p+len, max-len, " ghost           %-10d %-10d %-10d\n", dp->irq_ghost[0], dp->irq_ghost[1], dp->irq_ghost[2]);
  }

  len += scnprintf(p+len, max-len, "Netstats         packets    bytes      errors\n");
  len += scnprintf(p+len, max-len, " rx              %-10lld %-10lld %-10lld\n", dp->rx_stats.packets, dp->rx_stats.bytes, dp->rx_stats.errors);
  len += scnprintf(p+len, max-len, " tx              %-10lld %-10lld %-10lld\n", dp->tx_stats.packets, dp->tx_stats.bytes, dp->tx_stats.errors);

  p[len] = 0;
  return len;
}

/******************************************************************************
 *
 * cs8_device_reset
 *
 *****************************************************************************/
static void cs8_device_reset(struct cs8_device_t *dp)
{
  u32 regval = GDMA_CONTROL_CHNL_RESET;

  dp->state = DEVICE_STATE_DISABLED;

  if (Loopback)                        regval |= GDMA_CONTROL_LOOPBACK;
  if (dp->irq_mode == IRQ_MODE_SHARED) regval |= GDMA_CONTROL_SRIO_DISABLE;

  writel(regval, dp->bar + GDMA_CONTROL);

  udelay(1000);
}

/******************************************************************************
 *
 * cs8_device_init
 *
 *****************************************************************************/
static int cs8_device_init(struct cs8_device_t *dp)
{
  // initialize channels
  cs8_channel_init(&dp->tx_channel);
  cs8_channel_init(&dp->rx_channel);

  // initialize messages
  cs8_messages_init(dp);

  // initialize spm event
  init_waitqueue_head(&dp->spm_event);
  dp->spm_message = 0;

  // reset statistics
  memset(dp->irq_cnt, 0, sizeof(dp->irq_cnt));
  memset(dp->irq_shared, 0, sizeof(dp->irq_shared));
  memset(dp->irq_ghost, 0, sizeof(dp->irq_ghost));

  // enable interrupts
  cs8_irq_enable(dp);

  dp->flags = DeviceFlags & DEVICE_FLAG_CRC;

  if (dp->type == DEVICE_TYPE_PARENT) dp->flags |= DeviceFlags & DEVICE_FLAG_NET;

  dp->state = DEVICE_STATE_INITIALIZED;

  return 0;
}

/******************************************************************************
 *
 * cs8_session_msg_add
 *
 *****************************************************************************/
static inline void cs8_session_msg_add(struct cs_session_t *session, struct cs8_message_t *msg)
{
  spin_lock(&session->lock);
  MASK_SET(session->messages, msg->idx);
  spin_unlock(&session->lock);

  msg->session = session;
}

/******************************************************************************
 *
 * cs8_session_msg_remove
 *
 *****************************************************************************/
static inline void cs8_session_msg_remove(struct cs_session_t *session, struct cs8_message_t *msg)
{
  spin_lock(&session->lock);
  MASK_CLR(session->messages, msg->idx);
  spin_unlock(&session->lock);

  msg->session = NULL;
}

/******************************************************************************
 *
 * cs8_session_msg_check
 *
 *****************************************************************************/
static inline unsigned int cs8_session_msg_check(struct cs_session_t *session, struct cs8_message_t *msg)
{
  if (  msg->session == NULL
     || msg->session != session
     )
    return 0;

  return MASK_QUERY(session->messages, msg->idx);
}

/******************************************************************************
 *
 * cs8_reset_prepare
 *
 *****************************************************************************/
static void cs8_reset_prepare(struct cs8_device_t *dp)
{
  log_trace("channel#%d\n", dp->chnid);

  dp->state = DEVICE_STATE_DISABLED;

  cs8_irq_disable(dp);

  cs8_mailbox_send(dp, SPM_TYPE_DISABLED, dp->chnid, 0);

  cs8_messages_cleanup(dp);
}

/******************************************************************************
 *
 * cs8_reset_done
 *
 *****************************************************************************/
static void cs8_reset_done(struct cs8_device_t *dp)
{
  log_trace("channel#%d\n", dp->chnid);

  cs8_device_reset(dp);
  cs8_device_init(dp);
  cs8_mailbox_send(dp, SPM_TYPE_INITIALIZED, dp->chnid, dp->flags);
}

/******************************************************************************
 *
 * cs8_background_task
 *
 *****************************************************************************/
static void cs8_background_task(struct work_struct *work)
{
  struct delayed_work *dwork = container_of(work, struct delayed_work, work);
  struct cs8_device_t *dp = container_of(dwork, struct cs8_device_t, background_task);
  unsigned long delay = 5*HZ;

  if (dp->state != DEVICE_STATE_OFFLINE)
  {
    if (!cs8_device_healthy(dp))
    {
      log_error("channel#%d: offline\n", dp->chnid);

      dp->state = DEVICE_STATE_OFFLINE;
      cs8_messages_cleanup(dp);

      if (dp->type == DEVICE_TYPE_PARENT)
      {
        struct cs8_device_t *child;

        for (child = dp->next; child != NULL; child = child->next)
        {
          child->state = DEVICE_STATE_OFFLINE;
          cs8_messages_cleanup(child);
        }
      }
      goto end;
    }
  }

  if (dp->state == DEVICE_STATE_OFFLINE)
  {
    if (dp->pss != NULL)
    {
      pci_load_saved_state(dp->csdev.pci_dev, dp->pss);
      pci_restore_state(dp->csdev.pci_dev);

      if (!cs8_device_healthy(dp)) goto end;

      log_info("channel#%d: PCI state restored\n", dp->chnid);
    }

    if (cs8_device_healthy(dp))
    {
      log_info("channel#%d: online again\n", dp->chnid);

      cs8_reset_prepare(dp);
      cs8_reset_done(dp);

      dp->state |= DEVICE_STATE_RECOVER;
    }
  }

end:
  schedule_delayed_work(&dp->background_task, delay);
}

/******************************************************************************
 *
 * cs8_shutdown
 *
 *****************************************************************************/
static int cs8_shutdown(struct cs8_device_t *dp, int state)
{
  log_info("channel#%d: state:%d\n", dp->chnid, state);

  cs8_reset_prepare(dp);

  if (dp->type == DEVICE_TYPE_PARENT)
  {
    for ( ;dp != NULL; dp = dp->next)
    {
      if (dp->type == DEVICE_TYPE_CHILD)
      {
        log_info("channel#%d\n", dp->chnid);
        cs8_reset_prepare(dp);
      }
    }
  }

  return 0;
}

/******************************************************************************
 *
 * cs8_suspend
 *
 *****************************************************************************/
static int cs8_suspend(struct cs8_device_t *dp)
{
  log_info("channel#%d\n", dp->chnid);

  cs8_reset_prepare(dp);

  if (dp->type == DEVICE_TYPE_PARENT)
  {
    for ( ;dp != NULL; dp = dp->next)
    {
      if (dp->type == DEVICE_TYPE_CHILD)
      {
        log_info("channel#%d\n", dp->chnid);
        cs8_reset_prepare(dp);
      }
    }
  }

  return 0;
}

/******************************************************************************
 *
 * cs8_resume
 *
 *****************************************************************************/
static int cs8_resume(struct cs8_device_t *dp)
{
  log_info("channel#%d\n", dp->chnid);

  cs8_reset_done(dp);

  if (dp->type == DEVICE_TYPE_PARENT)
  {
    for ( ;dp != NULL; dp = dp->next)
    {
      if (dp->type == DEVICE_TYPE_CHILD)
      {
        log_info("channel#%d\n", dp->chnid);
        cs8_reset_done(dp);
      }
    }
  }

  return 0;
}

/******************************************************************************
 *
 * cs8_state
 *
 *****************************************************************************/
static const char *cs8_state(struct cs8_device_t *dp)
{
  return DeviceStates[dp->state&3];
}

/******************************************************************************
 *
 * cs8_open
 *
 *****************************************************************************/
static int cs8_open(struct cs_session_t *session)
{
  struct cs8_device_t *dp = (struct cs8_device_t*)session->dp;

  session->gen = dp->relaunch;
  get_random_bytes(&session->id, sizeof(session->id));

  return 0;
}

/******************************************************************************
 *
 * cs8_close
 *
 *****************************************************************************/
static int cs8_close(struct cs_session_t *session)
{
  struct cs8_device_t *dp = (struct cs8_device_t*)session->dp;
  int i;

  if (!SANITY_CHECK(dp, DEVICE_TAG))
  {
    log_error("no valid device\n");
    return -ENODEV;
  }

  for (i=0; i<MaxMessages; i++)
  {
    struct cs8_message_t *msg = dp->messages + i;

    if (cs8_session_msg_check(session, msg))
    {
      cs8_session_msg_remove(session, msg);
      cs8_msg_cancel(msg);
    }
  }

  if (session->has_sema)
  {
    session->has_sema = 0;
    up(&dp->sema);
    log_info("orphaned semaphore unlocked [%d]\n", dp->sema.count);
  }

  return 0;
}

/******************************************************************************
 *
 * cs8_read
 *
 *****************************************************************************/
static int cs8_read(struct cs_session_t *session, char __user *buf, size_t max_len)
{
  return -ENOSYS;
}

/******************************************************************************
 *
 * cs8_write
 *
 *****************************************************************************/
static int cs8_write(struct cs_session_t *session, const char __user *buf, size_t count)
{
  return -ENOSYS;
}

/******************************************************************************
 *
 * cs8_poll
 *
 *****************************************************************************/
static unsigned int cs8_poll(struct cs_session_t *session, struct file *filp, struct poll_table_struct *wait)
{
  struct cs8_device_t *dp = (struct cs8_device_t*)session->dp;
  struct cs8_message_t *msg;
  unsigned int mask = POLLOUT | POLLWRNORM;
  int i;

  if (session->gen != dp->relaunch)
  {
    log_error("cs8_ioctl: no valid session\n");
    return -EPIPE;
  }

  /*
   * according to comment in fs/eventpoll.c:
   *   epoll adds to the wakeup queue at EPOLL_CTL_ADD time only,
   * so we have to add all msg wait_queues to poll_table for polling to work
   */
  for (i = 0; i < MaxMessages; i++)
  {
    msg = dp->messages + i;
    poll_wait(filp, &msg->rx_event, wait);
  }

  for (i = 0; i < MaxMessages; i++)
  {
    msg = dp->messages + i;

    if (!MASK_QUERY(session->messages, msg->idx)) continue;

    if (!list_empty(&msg->rx_list.head))
    {
      mask |= POLLIN | POLLRDNORM;
      break;
    }
  }

  return mask;
}

/******************************************************************************
 *
 * cs8_ioctl
 *
 *****************************************************************************/
static int cs8_ioctl(struct cs_session_t *session, unsigned int cmd, unsigned long arg)
{
  int err = 0;
  struct cs8_device_t *dp = (struct cs8_device_t*)session->dp;

  if (!SANITY_CHECK(dp, DEVICE_TAG))
  {
    log_error("cs8_ioctl: no valid device\n");
    return -ENODEV;
  }

  /* if (session->gen != dp->relaunch)
  {
    log_error("cs8_ioctl: no valid session\n");
    return -EPIPE;
  } */

  switch (cmd)
  {
    //-------------------------------------------------------------------------
    case CSAR_CTL_RESET:
    //-------------------------------------------------------------------------
    {
      struct pci_reset_t *p_arg = (struct pci_reset_t *)arg;
      struct pci_reset_t reset;
      u32 mask = 0;

      if (copy_from_user(&reset, p_arg, sizeof(struct pci_reset_t)) != 0) return -EFAULT;

      if (reset.tag != RESET_TAG) return -EINVAL;

      if (reset.mask == 0) reset.mask = GDMA_CONTROL_CHNL_RESET;

      if (reset.mask & RESET_BLOCK) mask |= GDMA_CONTROL_BLOCK_RESET;
      if (reset.mask & RESET_CPU)   mask |= GDMA_CONTROL_CPU_RESET;

      if (mask != 0)
      {
        if (dp->type != DEVICE_TYPE_PARENT)
        {
          log_error("only CPU reset of physical (parent) function is allowed\n");
          return -EOPNOTSUPP;
        }

        // CPU | Block reset
        for ( ; dp != NULL; dp=dp->next)
        {
          cs8_reset_prepare(dp);

          if (dp->type == DEVICE_TYPE_PARENT)
          {
            if (dp->netdev)
            {
              netif_carrier_off(dp->netdev);
              log_info("%s is down\n", dp->netdev->name);
            }

            writel(mask, dp->bar + GDMA_CONTROL);
            udelay(1000);
            log_info("channel#%d: reset [0x%08x] done\n", dp->chnid, mask);
          }

          cs8_reset_done(dp);
        }
      }
      else if (reset.mask != 0)
      {
        // channel reset only
        cs8_reset_prepare(dp);
        cs8_reset_done(dp);
      }

      return 0;
    }

    //-------------------------------------------------------------------------
    case CSAR_CTL_INFO:
    //-------------------------------------------------------------------------
    {
      struct pci_info_t *p_arg = (struct pci_info_t *)arg;
      struct pci_info_t info;
      unsigned char     *buf;
      unsigned char     *p;
      unsigned int      max;
      unsigned int      len = 0;

      if (copy_from_user(&info, p_arg, sizeof(struct pci_info_t)) != 0) return(-EFAULT);

      if (  info.tag != INFO_TAG
         || (max = info.size) < PAGE_SIZE
         )
        return -EINVAL;

      if (info.mask == 0) info.mask = 1;

      if (LogLevel >= LOG_LEVEL_TRACE) info.mask |= 7;

      if ((p = buf = kmalloc(max, GFP_KERNEL)) == NULL) return(-ENOMEM);

      len += scnprintf(p+len, max-len, "Driver Version   %s\n", DriverVersionString);

      if (info.mask & 1)
      {
        // device
        len += cs8_device_info(dp, p+len, max-len);
      }

      if (info.mask & 2)
      {
        // messages
        len += cs8_messages_info(dp, p+len, max-len);
      }

      if (info.mask & 4)
      {
        // channels
        len += cs8_channel_info(&dp->tx_channel, p+len, max-len);
        len += cs8_channel_info(&dp->rx_channel, p+len, max-len);
      }

      // len += scnprintf(p+len, max-len, "[%d/%d]\n", len, max);

      if (copy_to_user((unsigned char*)info.data, buf, len) != 0) err = -EFAULT;

      kfree(buf);

      put_user(len, &p_arg->count);

      return 0;
    }

    //-------------------------------------------------------------------------
    case CSAR_CTL_SEND:
    //-------------------------------------------------------------------------
    {
      struct pci_request_t  *p_arg = (struct pci_request_t*)arg;
      struct pci_request_t  request;
      struct cs8_message_t  *msg;

      if (copy_from_user(&request, p_arg, sizeof(struct pci_request_t)) != 0) return -EFAULT;

      if (  !REQUEST_TAG_VALID(request.tag)
         || request.count > MAX_MSG_SIZE
         || request.data == 0
         )
       return -EINVAL;

      if (dp->state != DEVICE_STATE_ENABLED) return -EPIPE;

      // lock message
      if ((err = cs8_msg_lock(dp, &msg)) != 0)
      {
        log_error("cs8_msg_lock returned: %d", err);
        return err;
      }

      msg->state = MSG_STATE_SEND;

      // send message
      if ((err = cs8_msg_send(dp, msg, &request)) != 0)
      {
        if (request.count != 0)
        {
          // message was partially sent
          msg->flags |= MSG_FLAG_CANCEL;
        }
        else
        {
          cs8_msg_unlock(msg);
        }

        log_error("cs8_msg_send returned: %d\n", err);
        goto cleanup;
      }

      // add message to session
      cs8_session_msg_add(session, msg);

      msg->state = MSG_STATE_SEND_DONE;

      put_user(msg->idx,      &p_arg->handle);
      put_user(dp->chnid,     &p_arg->addr);
      put_user(request.count, &p_arg->count);
      return 0;
    }

    //-------------------------------------------------------------------------
    case CSAR_CTL_GET_REQUEST:
    //-------------------------------------------------------------------------
    {
      struct pci_request_t  *p_arg = (struct pci_request_t*)arg;
      struct pci_request_t  request;
      struct cs8_message_t  *msg;

      if (copy_from_user(&request, p_arg, sizeof(struct pci_request_t)) != 0) return -EFAULT;

      if (  !REQUEST_TAG_VALID(request.tag)
         || request.handle >= MaxMessages
         )
        return -EINVAL;

      if (dp->state != DEVICE_STATE_ENABLED) return -EPIPE;

      msg = dp->messages + request.handle;

      if (cs8_session_msg_check(session, msg) == 0) return -EACCES;

      msg->type = 0;
      msg->state = MSG_STATE_GETREQ;

      if ((err = cs8_msg_get_request(dp, msg, &request)) != 0)
      {
        cs8_session_msg_remove(session, msg);
        cs8_msg_cancel(msg);
        log_error("cs8_msg_get_request returned: %d\n", err);
        goto cleanup;
      }

      msg->state = MSG_STATE_GETREQ_DONE;

      put_user(msg->count, &p_arg->count);
      put_user(msg->type,  &p_arg->type);
      return 0;
    }

    //-------------------------------------------------------------------------
    case CSAR_CTL_RECV:
    //-------------------------------------------------------------------------
    {
      struct pci_request_t  *p_arg = (struct pci_request_t*)arg;
      struct pci_request_t  request;
      struct cs8_message_t  *msg;

      if (copy_from_user(&request, p_arg, sizeof(struct pci_request_t)) != 0) return -EFAULT;

      if (  !REQUEST_TAG_VALID(request.tag)
         || request.handle >= MaxMessages
         || request.count > MAX_MSG_SIZE
         || request.data == 0
         )
        return -EINVAL;

      if (dp->state != DEVICE_STATE_ENABLED) return -EPIPE;

      msg = dp->messages + request.handle;

      if (cs8_session_msg_check(session, msg) == 0) return -EACCES;

      msg->state = MSG_STATE_RECV;

      if ((err = cs8_msg_recv(dp, msg, &request)) != 0)
      {
        cs8_session_msg_remove(session, msg);
        cs8_msg_cancel(msg);
        log_error("cs8_msg_recv returned: %d\n", err);
        goto cleanup;
      }

      msg->state = MSG_STATE_RECV_DONE;

      if (msg->type == 0)
      {
        cs8_session_msg_remove(session, msg);
        cs8_msg_unlock(msg);
      }

      put_user(request.count, &p_arg->count);
      put_user(msg->type,     &p_arg->type);
      return 0;
    }

    //-------------------------------------------------------------------------
    case CSAR_CTL_CANCEL_REQUEST:
    //-------------------------------------------------------------------------
    {
      struct pci_request_t  *p_arg = (struct pci_request_t*)arg;
      struct pci_request_t  request;
      struct cs8_message_t  *msg;

      if (copy_from_user(&request, p_arg, sizeof(struct pci_request_t)) != 0) return -EFAULT;

      if (  !REQUEST_TAG_VALID(request.tag)
         || request.handle >= MaxMessages
         )
        return -EINVAL;

      msg = dp->messages + request.handle;

      if (cs8_session_msg_check(session, msg) == 0) return -EACCES;

      // remove message from session
      cs8_session_msg_remove(session, msg);

      // free & unlock message
      cs8_msg_cancel(msg);
      return 0;
    }

    //-------------------------------------------------------------------------
    case CSAR_CTL_SPM_SEND:
    //-------------------------------------------------------------------------
    {
      struct pci_spm_t *p_arg = (struct pci_spm_t*)arg;
      struct pci_spm_t spm;

      if (copy_from_user(&spm, p_arg, sizeof(struct pci_spm_t)) != 0) return(-EFAULT);

      if (spm.tag != SPM_TAG) return -EINVAL;

      cs8_mailbox_send(dp, SPM_TYPE_USER, dp->chnid, spm.message);

      put_user(dp->chnid, &p_arg->addr);
      return 0;
    }

    //-------------------------------------------------------------------------
    case CSAR_CTL_SPM_RECV:
    //-------------------------------------------------------------------------
    {
      struct pci_spm_t *p_arg = (struct pci_spm_t*)arg;
      struct pci_spm_t spm;

      if (copy_from_user(&spm, p_arg, sizeof(struct pci_spm_t)) != 0) return(-EFAULT);

      if (spm.tag != SPM_TAG) return -EINVAL;

      if (dp->spm_message == 0)
      {
        long timeout = msecs_to_jiffies(spm.timeout);

        long remain = wait_event_interruptible_timeout(dp->spm_event, (dp->spm_message != 0), timeout);

        if (remain <= 0)
        {
          log_debug("channel#%d: wait_event_interruptible_timeout(%d) returned: %ld\n", dp->chnid, spm.timeout, remain);
          return((remain < 0) ? remain : -ETIMEDOUT);
        }
      }

      put_user(dp->spm_message, &p_arg->message);
      put_user(dp->chnid, &p_arg->addr);

      dp->spm_message = 0;
      return 0;
    }

    //-------------------------------------------------------------------------
    case CSAR_CTL_GET_MODEL:
    case CS2IOC_GETHWTYPE:
    //-------------------------------------------------------------------------
      if (put_user(8, (int*)arg) != 0) return -EFAULT;
      return 0;

    //-------------------------------------------------------------------------
    case CSAR_CTL_LOG_LEVEL:
    //-------------------------------------------------------------------------
    {
      int old = LogLevel;
      int new = LogLevel;

      get_user(new, (int*)arg);

      if (new != old)
      {
        LogLevel = new;
        log_trace("changed log level from %d to %d\n", old, new);
      }

      put_user(old, (int*)arg);
      return 0;
    }

    //--------------------------------------------------------------------------------
    case CSAR_CTL_LOCK:
    //--------------------------------------------------------------------------------
    {
      if ((err = down_interruptible(&dp->sema)) != 0) return err;
      session->has_sema = 1;
      return 0;
    }

    //--------------------------------------------------------------------------------
    case CSAR_CTL_UNLOCK:
    //--------------------------------------------------------------------------------
    {
      session->has_sema = 0;
      up(&dp->sema);
      return 0;
    }

    //-------------------------------------------------------------------------
    case CSAR_CTL_TEST(66):
    case CSAR_CTL_TEST(67):
    case CSAR_CTL_TEST(68):
    //-------------------------------------------------------------------------
    {
      log_info("cmd: 0x%08x\n", cmd);
      return 0;
    }

    //-------------------------------------------------------------------------
    default:
    //-------------------------------------------------------------------------
    {
      if (_IOC_TYPE(cmd) != CSAR_IOC_MAGIC) return -ENOTTY;
      log_error("unknown control code: 0x%08x", cmd);
      return -EINVAL;
    }
  }

  return 0;

cleanup:
  log_back_print();
  return err;
}

/******************************************************************************
 *
 * cs8_proc_show
 *
 ******************************************************************************/
static int cs8_proc_show(struct seq_file *m, void *v)
{
  struct cs8_device_t *dp = (struct cs8_device_t*)m->private;
  struct pci_dev      *pci_dev;
  char                *buf = NULL;
  unsigned int        len;

  seq_printf(m, "Driver Version   %s\n", DriverVersionString);

  if (  dp == NULL
     || (pci_dev = dp->csdev.pci_dev) == NULL
     )
    return(-ENODEV);

  if ((buf = kmalloc(PAGE_SIZE, GFP_KERNEL)) == NULL) return -ENOMEM;

  if (LogLevel >= LOG_LEVEL_TRACE)
  {
    // Module Parameters
    len = cs8_param_info(buf, PAGE_SIZE);
    seq_printf(m, buf);

    // Power Management
    seq_printf(m, "Power management\n");
    seq_printf(m, " current_state   %s\n", pci_dev->current_state >= 0 ? PowerStates[pci_dev->current_state&7] : "error");
    seq_printf(m, " pm_cap          %d\n", pci_dev->pm_cap);
    seq_printf(m, " pme_support     0x%08x\n", pci_dev->pme_support);
    seq_printf(m, " d1_support      %d\n", pci_dev->d1_support);
    seq_printf(m, " d2_support      %d\n", pci_dev->d2_support);
    seq_printf(m, " no_d1d2         %d\n", pci_dev->no_d1d2);
    seq_printf(m, " wakeup_prepared %d\n", pci_dev->wakeup_prepared);
  //seq_printf(m, " ignore_hotplug  %d\n", pci_dev->ignore_hotplug);
  //seq_printf(m, " hotplug_user_i  %d\n", pci_dev->hotplug_user_indicators);
  }

  len = cs8_device_info(dp, buf, PAGE_SIZE);
  seq_printf(m, buf);

  len = cs8_messages_info(dp, buf, PAGE_SIZE);
  seq_printf(m, buf);

  len  = cs8_channel_info(&dp->tx_channel, buf,     PAGE_SIZE);
  len += cs8_channel_info(&dp->rx_channel, buf+len, PAGE_SIZE-len);
  seq_printf(m, buf);

  kfree(buf);
  return 0;
}

/******************************************************************************
 *
 * cs8_proc_write
 *
 ******************************************************************************/
static ssize_t cs8_proc_write(struct file *file, const char __user *user, size_t count, loff_t *off)
{
  struct cs8_device_t *dp = compat_file_to_PDE_DATA(file);
  int   err = 0;
  char  *buf = NULL;
  int   i;

  if ((buf = kzalloc(count+1, GFP_KERNEL)) == NULL) return -ENOMEM;

  if (copy_from_user(buf, user, count) != 0) CLEANUP(-EFAULT);

  //---------------------------------------------------------------------------
  if (strncasecmp(buf, "RESET", 5) == 0)
  //---------------------------------------------------------------------------
  {
    u32 mask = 0; // GDMA_CONTROL_CPU_RESET | GDMA_CONTROL_CHNL_RESET;
    cs_get_args(buf+5, "u", &mask);

    cs8_reset_prepare(dp);

    if (mask != 0)
    {
      writel(mask, dp->bar + GDMA_CONTROL);
      udelay(1000);
    }

    cs8_reset_done(dp);
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "REBOOT", 6) == 0)
  //---------------------------------------------------------------------------
  {
    if (dp->type != DEVICE_TYPE_PARENT)
    {
      log_error("only CPU reset of physical (parent) function is allowed\n");
      err = -EOPNOTSUPP;
    }
    else
    {
      for ( ;dp != NULL; dp = dp->next)
      {
        if (dp->netdev)
        {
          netif_carrier_off(dp->netdev);
          log_info("%s is down\n", dp->netdev->name);
        }

        cs8_reset_prepare(dp);

        if (dp->type == DEVICE_TYPE_PARENT)
        {
          writel(GDMA_CONTROL_CPU_RESET, dp->bar + GDMA_CONTROL);
          udelay(1000);
          log_info("CPU reset done\n");
        }

        cs8_reset_done(dp);
      }
    }
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "POWER", 5) == 0)
  //---------------------------------------------------------------------------
  {
    pci_power_t state = PCI_D0;
    cs_get_args(buf+5, "u", &state);

    log_info("setting power state to: %s ...\n", PowerStates[state]);
    err = pci_set_power_state(dp->csdev.pci_dev, state);
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "FLR", 3) == 0)
  //---------------------------------------------------------------------------
  {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,17,0))
    err = pcie_flr(dp->csdev.pci_dev);
#elif (LINUX_VERSION_CODE >= KERNEL_VERSION(4,12,0))
    pcie_flr(dp->csdev.pci_dev);
#endif
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "TEST", 4) == 0)
  //---------------------------------------------------------------------------
  {
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "ErrorReporting", 14) == 0)
  //---------------------------------------------------------------------------
  {
    cs_get_args(buf+14, "u", &ErrorReporting);
    err = cs8_set_error_reporting(dp->csdev.pci_dev);
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "MaxPayload", 10) == 0)
  //---------------------------------------------------------------------------
  {
    cs_get_args(buf+10, "u", &MaxPayload);
    err = cs8_set_maxpay(dp->csdev.pci_dev);
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "MaxReadReq", 10) == 0)
  //---------------------------------------------------------------------------
  {
    cs_get_args(buf+10, "u", &MaxReadReq);
    err = cs8_set_readrq(dp->csdev.pci_dev);
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "MessageExpiry", 13) == 0)
  //---------------------------------------------------------------------------
  {
    cs_get_args(buf+13, "u", &MessageExpiry);
    log_info("MessageExpiry has been set to: %d\n", MessageExpiry);
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "Control", 7) == 0)
  //---------------------------------------------------------------------------
  {
    u32 regval = 0;
    cs_get_args(buf+7, "u", &regval);
    writel(regval, dp->bar + GDMA_CONTROL);
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "DeviceMask", 10) == 0)
  //---------------------------------------------------------------------------
  {
    cs_get_args(buf+10, "u", &DeviceMask);
    log_info("DeviceMask has been set to: %d\n", DeviceMask);
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "CoreControl", 11) == 0)
  //---------------------------------------------------------------------------
  {
    u32 val = 0;
    cs_get_args(buf+11, "u", &val);
    cs8_set_core_control(dp, val);
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "ITR", 3) == 0)
  //---------------------------------------------------------------------------
  {
    u8 mask = 0;
    cs_get_args(buf+3, "u", &mask);
    writeb(mask, dp->bar + GDMA_IRQ_TEST);
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "SPM", 3) == 0)
  //---------------------------------------------------------------------------
  {
    u32 val1 = 0;
    u32 val2 = 0;

    if (cs_get_args(buf+3, "uu", &val1, &val2) == 2)
    {
      cs8_mailbox_send(dp, val1, dp->chnid, val2);
    }
    else
    {
      cs8_mailbox_send(dp, SPM_TYPE_USER, dp->chnid, val1);
    }
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "LOGLEVEL", 8) == 0)
  //---------------------------------------------------------------------------
  {
    unsigned int new = LogLevel;

    if (cs_get_args(buf+8, "u", &new) > 0)
    {
      printk("LogLevel changed from % d to %d\n", LogLevel, new);
      LogLevel = new;
    }
    else
    {
       printk("LogLevel: %d\n", LogLevel);
    }
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "READ", 4) == 0)
  //---------------------------------------------------------------------------
  {
    struct cs_iomem_t *bar = dp->iomem;
    unsigned int ofs = 0;
    unsigned int cnt = 265;  // bytes
    char line[80];
    char *p_x = NULL;

    cs_get_args(buf+4, "uu", &ofs, &cnt);

    printk("BAR%d: %08x:%08x\n", 0, bar->phys_addr_hi, bar->phys_addr_lo);

    for (i=0; i<cnt/4; i++, ofs += 4)
    {
      u32 regval = readl(bar->base_ptr + ofs);

      if ((i & 3) == 0) p_x = line + sprintf(line, "%4x |", ofs);

      p_x += sprintf(p_x, " %08X\n", regval) - 1;

      if ((i & 3) == 3)
      {
        printk(line);
        p_x = NULL;
      }
    }
  }
  //---------------------------------------------------------------------------
  else if (  strncasecmp(buf, "WRITEB", 6) == 0
          || strncasecmp(buf, "WRITEL", 6) == 0
          )
  //---------------------------------------------------------------------------
  {
    struct cs_iomem_t *bar = dp->iomem;
    unsigned int ofs = 0;
    unsigned int value;

    if (cs_get_args(buf+6, "uu", &ofs, &value) < 2) return -EINVAL;

    if ((buf[5] & 0x0F) == 0x02) writeb(value, bar->base_ptr + ofs);
    else                         writel(value, bar->base_ptr + ofs);
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "CFG", 3) == 0)
  //---------------------------------------------------------------------------
  {
    unsigned int config[64];
    int i;

    for (i=0; i<DIM(config); i++)
    {
      pci_read_config_dword(dp->csdev.pci_dev, i*4, config + i);
    }

    log_xprint("config", 0, (void*)config, sizeof(config));
  }
  //---------------------------------------------------------------------------
  else if (strncasecmp(buf, "HELP", 4) == 0)
  //---------------------------------------------------------------------------
  {
    printk("valid commands are:\n");
    printk("  loglevel  [=<level>]\n");
    printk("  read      [=<bar#>,<ofs>,<count>]\n");
    printk("  write     [=<bar#>,<ofs>,<value>]\n");
    printk("  cfg       [=<mode>]\n");
  }
  else
  {
    log_error("invalid argument: %s\n", buf);
    CLEANUP(-EINVAL);
  }

  if (err < 0)
  {
    log_error("'%s' returned: %d\n", buf, err);
    goto cleanup;
  }

  (void)dp;

cleanup:
  if (buf != NULL) kfree(buf);

  return (err != 0) ? err : count;
}

/******************************************************************************
 *
 * cs8_proc_open
 *
 ******************************************************************************/
static int cs8_proc_open(struct inode *inode, struct file *file)
{
  return single_open(file, cs8_proc_show, compat_PDE_DATA(inode));
}

//-----------------------------------------------------------------------------
static const proc_op_t cs8_proc_ops =
//-----------------------------------------------------------------------------
{
#ifdef HAVE_PROC_OPS
  .proc_open    = cs8_proc_open,
  .proc_read    = seq_read,
  .proc_write   = cs8_proc_write,
  .proc_lseek   = seq_lseek,
  .proc_release = single_release,
#else
  .owner   = THIS_MODULE,
  .open    = cs8_proc_open,
  .read    = seq_read,
  .write   = cs8_proc_write,
  .llseek  = seq_lseek,
  .release = single_release,
#endif
};

/******************************************************************************
 *
 * cs8_ethtool_get_drvinfo
 *
 *****************************************************************************/
static void cs8_ethtool_get_drvinfo(struct net_device *net, struct ethtool_drvinfo *info)
{
  scnprintf(info->driver, sizeof(info->driver), "%s-netdev [%s]", DRIVER_NAME, DRIVER_VERSION_STR);
}


/******************************************************************************
 *
 * cs8_ethtool_get_link
 *
 *****************************************************************************/
static u32 cs8_ethtool_get_link(struct net_device *net)
{
  struct cs8_netdev *ndev = netdev_priv(net);
  struct cs8_device_t *dp = ndev->dp;

  return dp->state == DEVICE_STATE_ENABLED;
}

static const struct ethtool_ops cs8_ethtool_ops = {
  .get_drvinfo        = cs8_ethtool_get_drvinfo,
  .get_link           = cs8_ethtool_get_link,
//.get_msglevel       = cs8_get_msglevel,
//.set_msglevel       = cs8_set_msglevel,
//.get_link_ksettings = cs8_get_link_ksettings,
//.set_link_ksettings = cs8_set_link_ksettings,
};

/******************************************************************************
 *
 * cs8_netdev_open
 *
 *****************************************************************************/
static int cs8_netdev_open(struct net_device *netdev)
{
  struct cs8_netdev *ndev = netdev_priv(netdev);
  netif_start_queue(netdev);
  ndev->use_count++;
  log_info("%s was opened\n", netdev->name);
  return 0;
}

/******************************************************************************
 *
 * cs8_netdev_stop
 *
 *****************************************************************************/
static int cs8_netdev_stop(struct net_device *netdev)
{
  struct cs8_netdev *ndev = netdev_priv(netdev);
  netif_stop_queue(netdev);
  ndev->use_count--;
  log_info("%s was stopped\n", netdev->name);
  return 0;
}

/******************************************************************************
 *
 * cs8_netdev_start_xmit
 *
 *****************************************************************************/
static int cs8_netdev_start_xmit(struct sk_buff *skb, struct net_device *netdev)
{
  struct cs8_netdev *ndev = netdev_priv(netdev);
  struct cs8_device_t *dp = ndev->dp;
  struct cs8_channel_t *ch = &dp->tx_channel;
  struct cs8_packet_t *pkt;
  int err;
  int len = skb->len;

  if ((err = cs8_pkt_get_first(&ch->free_list, &pkt, NULL)) != 0)
  {
    netif_stop_queue(netdev);
    dp->tx_stats.errors++;
    return NETDEV_TX_BUSY;
  }

  memcpy(pkt->base_ptr, skb->data, len);
  pkt->info.pid = pkt->idx;
  pkt->info.wlen = DMA_WLEN(len);
  pkt->info.addr = ndev->dp->chnid;
  // special marker for network traffic
  pkt->info.seq_cnt = 0;
  pkt->info.msg_idx = MSG_IDX_NET;
  pkt->info.last = 1;

  if ((err = cs8_channel_pkt_submit(dp, ch, pkt)) != 0)
  {
    cs8_pkt_put_last(&ch->free_list, pkt);
    netif_stop_queue(netdev);
    dp->tx_stats.errors++;
    log_error("cs8_channel_pkt_submit returned: %d\n", err);
    return NETDEV_TX_BUSY;
  }

  dp->tx_stats.packets++;
  dp->tx_stats.bytes += len;

  log_xtrace("%sent", skb->data, len);

  dev_kfree_skb_any(skb);

  return NETDEV_TX_OK;
}

/******************************************************************************
 *
 * cs8_netdev_get_stats
 *
 *****************************************************************************/
static struct net_device_stats *cs8_netdev_get_stats(struct net_device *netdev)
{
  struct cs8_netdev *ndev = netdev_priv(netdev);
  struct cs8_device_t *dp = ndev->dp;
  static struct net_device_stats stats;

  log_debug("\n");

  memset(&stats, 0, sizeof(stats));

  stats.rx_packets = dp->rx_stats.packets;
  stats.tx_packets = dp->tx_stats.packets;
  stats.rx_bytes   = dp->rx_stats.bytes;
  stats.tx_bytes   = dp->tx_stats.bytes;
  stats.rx_errors  = dp->rx_stats.errors;
  stats.tx_errors  = dp->tx_stats.errors;

  return &stats;
}

/******************************************************************************
 *
 * cs8_netdev_get_stats64
 *
 *****************************************************************************/
static void cs8_netdev_get_stats64(struct net_device *netdev, struct rtnl_link_stats64 *stats64)
{
  struct cs8_netdev *ndev = netdev_priv(netdev);
  struct cs8_device_t *dp = ndev->dp;

  log_debug("\n");

  stats64->rx_packets = dp->rx_stats.packets;
  stats64->tx_packets = dp->tx_stats.packets;
  stats64->rx_bytes   = dp->rx_stats.bytes;
  stats64->tx_bytes   = dp->tx_stats.bytes;
  stats64->rx_errors  = dp->rx_stats.errors;
  stats64->tx_errors  = dp->tx_stats.errors;
}

/******************************************************************************
 *
 * cs8_netdev_set_mac_addr
 *
 *****************************************************************************/
static int cs8_netdev_set_mac_addr(struct net_device *netdev, void *addr)
{
  struct sockaddr *sock_addr = addr;

  if (addr == NULL) return -EINVAL;

  log_trace("sa_family: %d\n", sock_addr->sa_family);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(6,2,0))
  log_xdebug("sa_data_min", sock_addr->sa_data_min, sizeof(sock_addr->sa_data_min));
#else
  log_xdebug("sa_data_min", sock_addr->sa_data, sizeof(sock_addr->sa_data));
#endif

  return eth_mac_addr(netdev, addr);
}

/******************************************************************************
 *
 * cs8_netdev_validate_addr
 *
 *****************************************************************************/
static int cs8_netdev_validate_addr(struct net_device *netdev)
{
  log_debug("\n");
  return eth_validate_addr(netdev);
}

//-----------------------------------------------------------------------------
// net_device_ops
//-----------------------------------------------------------------------------
static const struct net_device_ops cs8_netdev_ops =
{
  .ndo_open             = cs8_netdev_open,
  .ndo_stop             = cs8_netdev_stop,
  .ndo_start_xmit       = cs8_netdev_start_xmit,
  .ndo_set_mac_address  = cs8_netdev_set_mac_addr,
  .ndo_validate_addr    = cs8_netdev_validate_addr,
  .ndo_get_stats        = cs8_netdev_get_stats,
  .ndo_get_stats64      = cs8_netdev_get_stats64
};

/******************************************************************************
 *
 * cs8_netdev_setup
 *
 *****************************************************************************/
static void cs8_netdev_setup(struct net_device *netdev)
{
  ether_setup(netdev);
  netdev->netdev_ops = &cs8_netdev_ops;
  netdev->tx_queue_len = 48;
  netdev->ethtool_ops = &cs8_ethtool_ops;

  log_info("done\n");
}

/******************************************************************************
 *
 * cs8_netdev_create
 *
 *****************************************************************************/
static int cs8_netdev_create(struct cs8_device_t *dp)
{
  struct net_device *netdev = NULL;
  struct cs8_netdev *ndev = NULL;

  int err;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3,17,0)
  netdev = alloc_netdev(sizeof(struct cs8_netdev), "csar_net%d", NET_NAME_UNKNOWN, cs8_netdev_setup);
#else
  netdev = alloc_netdev(sizeof(struct cs8_netdev), "csar_net%d", cs8_netdev_setup);
#endif

  if (netdev == NULL) return -ENOMEM;

  SET_NETDEV_DEV(netdev, &dp->csdev.pci_dev->dev);
  eth_hw_addr_random(netdev);

  ndev = netdev_priv(netdev);
  ndev->dp = dp;
  ndev->use_count = 0;

  dp->netdev = netdev;
  netif_carrier_off(netdev);

  if ((err = register_netdev(netdev)))
  {
    log_error("register_netdev(%s) returned: %d\n", netdev->name, err);
    dp->netdev = NULL;
    free_netdev(netdev);
    return err;
  }

  log_info("%s was successfully created\n", netdev->name);

  return 0;
}


/******************************************************************************
 *
 * cs8_netdev_free
 *
 *****************************************************************************/
static int cs8_netdev_free(struct cs8_device_t *dp)
{
  if (dp->netdev == NULL) return -EINVAL;

  unregister_netdev(dp->netdev);
  free_netdev(dp->netdev);
  dp->netdev = NULL;
  return 0;
}

/******************************************************************************
 *
 * cs8_remove
 *
 *****************************************************************************/
static int cs8_remove(struct cs8_device_t *parent)
{
  struct cs8_device_t *dp;
  struct pci_dev      *pci_dev;

  if (!SANITY_CHECK(parent, DEVICE_TAG))
  {
    log_error("no valid device\n");
    return -ENODEV;
  }

  pci_dev = parent->csdev.pci_dev;

  // disconnect interrupt
  cs8_irq_free(parent);

  for (dp=parent; dp!=NULL; dp=dp->next)
  {
    int i,j;
    struct cs_device_t *csdev = &dp->csdev;

    log_trace("removing cs2.%d devfn:%d subfn:%d ...\n", csdev->minor, pci_dev->devfn, csdev->subfn);

    cancel_delayed_work_sync(&dp->background_task);

    dp->state = DEVICE_STATE_DISABLED;

    cs8_irq_disable(dp);

    cs8_mailbox_send(dp, SPM_TYPE_DISABLED, dp->chnid, 0);

    cs8_messages_cleanup(dp);

    cs8_netdev_free(dp);

    // cleanup channels
    for (i=0; i<2; i++)
    {
      struct cs8_channel_t *channel = (i==0) ? &dp->tx_channel : &dp->rx_channel;

      tasklet_kill(&channel->task);

      // free packets
      if (channel->packets != NULL)
      {
        for (j=0; j<channel->max_packets; j++)
        {
          struct cs8_packet_t *pkt = channel->packets + j;
          cs8_pkt_free_ex(dp, channel->dir, pkt);
        }

        kfree(channel->packets);
      }
    }

    // cleanup mailbox
    tasklet_kill(&dp->mailbox.task);

    if (dp->pss != NULL)
    {
      pci_load_and_free_saved_state(csdev->pci_dev, &dp->pss);
    }

    log_trace("remove cs2.%d devfn:%d subfn:%d OK\n", csdev->minor, pci_dev->devfn, csdev->subfn);
  }

  // unmap memory
  cs8_io_unmap(parent->iomem);

  // free memory
  for (dp=parent; dp!=NULL; )
  {
    struct cs8_device_t *next = dp->next;

    if (dp->devices != NULL) kfree(dp->devices);
    memset(dp, 0, sizeof(struct cs8_device_t));
    kfree(dp);

    dp = next;
  }

  pci_disable_device(pci_dev);
  pci_release_regions(pci_dev);

  return 0;
}

/******************************************************************************
 *
 * cs8_probe
 *
 *****************************************************************************/
int cs8_probe(struct pci_dev *pci_dev, struct cs8_device_t **pp_dp)
{
  int                   err = 0;
  struct cs8_device_t   *prev = NULL;
  struct cs8_device_t   *parent = NULL;
  struct cs8_device_t   *dp = NULL;
  struct cs8_channel_t  *channel;
  int                   idx;
  int                   i,j;
  unsigned int          dev_mask;

  log_trace("probing devfn:%d ...\n", pci_dev->devfn);

  if (MaxPackets < QUEUE_SIZE)  MaxPackets = QUEUE_SIZE;
  if (MaxPackets > 1024)        MaxPackets = 1024;

  if (Prefetch < 1)             Prefetch = 1;
  if (Prefetch > QUEUE_SIZE/2)  Prefetch = QUEUE_SIZE/2;

  if (MaxMessages < 1)            MaxMessages = 1;
  if (MaxMessages > MAX_MESSAGES) MaxMessages = MAX_MESSAGES;

  if (pci_dev->device == 0xC071)
  {
    dev_mask = DeviceMask | 1;
  }
  else
  {
    dev_mask = 1;
  }

  // enable device
  if ((err = pcim_enable_device(pci_dev)) != 0)
  {
    log_error("pcim_enable_device returned: %d\n", err);
    return err;
  }

  // set DMA mask
  if ((err = dma_set_mask_and_coherent(&pci_dev->dev, DMA_BIT_MASK(64))) != 0)
  {
    log_error("dma_set_mask_and_coherent returned: %d\n", err);
    return err;
  }

  // request memory regions
  if ((err = pcim_iomap_regions_request_all(pci_dev, 1, "cs8")) != 0)
  {
    log_error("pcim_iomap_regions_request_all returned: %d\n", err);
    return err;
  }

  // set master
  pci_set_master(pci_dev);

  // enable/disable error reporting
  if ((err = cs8_set_error_reporting(pci_dev)) != 0)
    log_error("cs8_set_error_reporting(0x%x) returned: %d\n", ErrorReporting, err);

  // configure max. payload
  if ((err = cs8_set_maxpay(pci_dev)) != 0)
    log_error("cs8_set_maxpay(%d) returned: %d\n", MaxPayload, err);

  // configure read request size
  if ((err = cs8_set_readrq(pci_dev)) != 0)
    log_error("cs8_set_readrq(%d) returned: %d\n", MaxReadReq, err);

  for (idx=0; idx<GDMA_MAX_CHANNELS; idx++)
  {
    struct cs_device_t *csdev;

    if ((dev_mask & BIT(idx)) == 0) continue;

    // create device
    if ((dp = kzalloc(sizeof(struct cs8_device_t), GFP_KERNEL)) == NULL) CLEANUP(-ENOMEM);

    dp->tag = DEVICE_TAG;

    // chain in
    if (prev != NULL) prev->next = dp;
    prev = dp;

    if (idx == 0)
    {
      if ((err = cs8_io_map(pci_dev, 0, dp->iomem)) != 0)
      {
        log_error("cs8_io_map(0) returned: %d", err);
        goto cleanup;
      }

      parent = dp;
    }

    dp->bar = parent->iomem[0].base_ptr + idx * PAGE_SIZE;

    dp->chnid = readl(dp->bar + GDMA_VERSION) >> 24;

    if (pci_dev->is_virtfn)
    {
      dp->type = DEVICE_TYPE_VIRT;
      dp->irq_mode = IRQ_MODE_CHANNEL;
    }
    else // if (pci_dev->is_physfn)
    {
      dp->type = (idx == 0) ? DEVICE_TYPE_PARENT : DEVICE_TYPE_CHILD;
      dp->irq_mode = (DeviceMask != 0) ? IRQ_MODE_SHARED : IRQ_MODE_CHANNEL;
    }

    if (dp->irq_mode == IRQ_MODE_SHARED)
    {
      if (dp->type == DEVICE_TYPE_PARENT)
      {
        unsigned int size = GDMA_MAX_CHANNELS * sizeof(struct cs8_device_t*);
        if ((dp->devices = kzalloc(size, GFP_KERNEL)) == NULL) CLEANUP(-ENOMEM);
      }

      parent->devices[dp->chnid] = dp;
    }

    // initialize base device
    csdev = &dp->csdev;

    csdev->minor = -1;
    csdev->subfn = dp->chnid;

    csdev->pci_dev = pci_dev;
    csdev->proc_ops = &cs8_proc_ops;

    csdev->p_open   = cs8_open;
    csdev->p_close  = cs8_close;
    csdev->p_ioctl  = cs8_ioctl;
    csdev->p_read   = cs8_read;
    csdev->p_write  = cs8_write;
    csdev->p_poll   = cs8_poll;

    csdev->p_remove        = (int(*) (struct cs_device_t*))cs8_remove;
    csdev->p_shutdown      = (int(*) (struct cs_device_t*, int))cs8_shutdown;
    csdev->p_suspend       = (int(*) (struct cs_device_t*))cs8_suspend;
    csdev->p_resume        = (int(*) (struct cs_device_t*))cs8_resume;
    csdev->p_reset_prepare = (void(*)(struct cs_device_t*))cs8_reset_prepare;
    csdev->p_reset_done    = (void(*)(struct cs_device_t*))cs8_reset_done;
    csdev->p_state         = (const char*(*) (struct cs_device_t*))cs8_state;

    // reset DMA core
    cs8_device_reset(dp);

    // set core control register
    cs8_set_core_control(dp, (CoreControl == 0) ? 0x80000 : CoreControl);

    // setup mailbox
    spin_lock_init(&dp->mailbox.lock);
    tasklet_init(&dp->mailbox.task, cs8_mailbox_task, (unsigned long)dp);

    // setup DMA channels
    channel = &dp->tx_channel;

    for (i=0; i<2; i++)
    {
      channel->bar = dp->bar;

      spin_lock_init(&channel->lock);

      if (i==0)
      {
        // TX
        channel->type = CHANNEL_TYPE_PCI_AXI;
        channel->dir = DMA_TO_DEVICE;
        channel->max_packets = MAX_PACKETS_TX;
        tasklet_init(&channel->task, cs8_channel_tx_task, (unsigned long)dp);
      }
      else
      {
        // RX
        channel->type = CHANNEL_TYPE_AXI_PCI;
        channel->dir = DMA_FROM_DEVICE;
        channel->max_packets = MaxPackets; // MAX_PACKETS_RX;
        tasklet_init(&channel->task, cs8_channel_rx_task, (unsigned long)dp);
      }

      // allocate packet pool
      channel->packets = kzalloc(channel->max_packets * sizeof(struct cs8_packet_t), GFP_KERNEL);

      if (channel->packets == NULL)
      {
        log_error("kzalloc(packets) failed\n");
        CLEANUP(-ENOMEM);
      }

      for (j=0; j<channel->max_packets; j++)
      {
        struct cs8_packet_t *pkt = channel->packets + j;

        memset(pkt, 0, sizeof(struct cs8_packet_t));

        if ((err = cs8_pkt_alloc_ex(dp, MAX_PACKET_SIZE, channel->dir, pkt)) != 0)
        {
          log_error("cs8_pkt_alloc_ex (channel[%d].packets[%d]) returned: %d\n", i, j, err);
          goto cleanup;
        }
      }

      channel = &dp->rx_channel;
    }

    sema_init(&dp->sema, 1);

    cs8_device_init(dp);

    INIT_DELAYED_WORK(&dp->background_task, cs8_background_task);

    schedule_delayed_work(&dp->background_task, 5*HZ);

    log_trace("probe devfn:%d subfn:%d OK\n", pci_dev->devfn, csdev->subfn);
  }

  if (  pci_dev->device == 0xC071
     && (parent->flags & DEVICE_FLAG_NET)
     )
  {
    if ((err = cs8_netdev_create(parent)) != 0)
    {
      log_error("cs8_netdev_create returned: %d\n", err);
      goto cleanup;
    }
  }

  // connect interrupt
  if ((err = cs8_irq_request(parent)) != 0)
  {
    log_error("cs8_irq_request returned: %d\n", err);
    goto cleanup;
  }

  pci_save_state(pci_dev);
  parent->pss = pci_store_saved_state(pci_dev);

  for (dp=parent; dp!=NULL; dp=dp->next)
  {
    // inform devices that we are initialized
    cs8_mailbox_send(dp, SPM_TYPE_INITIALIZED, dp->chnid, dp->flags);
  }

  *pp_dp = parent;
  return 0;

cleanup:
  cs8_remove(parent);
  *pp_dp = NULL;
  return err;
}

MODULE_AUTHOR("Sven Kaltschmidt <sven.kaltschmidt@utimaco.com>");
