// SPDX-License-Identifier: GPL-2.0-only
/*
 * CryptoServer Driver
 *
 * Copyright 2024 Utimaco IS GmbH
 * All Rights Reserved.
 *
 */
#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#include <linux/version.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/sched.h>
#include <linux/poll.h>
#include <linux/pm.h>
#include <linux/idr.h>

#include "cryptoserver.h"
#include "version.h"
#include "compat.h"

/******************************************************************************
 *
 * Definitions
 *
 ******************************************************************************/
//#define DRIVER_NAME           "cryptoserver"

#define DRIVER_BUILD          ""

#ifdef TRACE
#undef DRIVER_BUILD
#define DRIVER_BUILD          "[TRACE]"
#endif

#ifdef DEBUG
#undef DRIVER_BUILD
#define DRIVER_BUILD          "[DEBUG]"
#endif

#define CLASS_NAME            "hsm"
#define CHRDEV_NAME           "CryptoServer"

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,32)
#error Only for kernel versions >= 2.6.32
#endif


/******************************************************************************
 *
 * Globals
 *
 *****************************************************************************/

// globale module parameter
// unsigned int LogLevel = LOG_LEVEL_INFO;
module_param(LogLevel, int, 0664);
MODULE_PARM_DESC(LogLevel, " 0:none, 1:error, 2:warning, 3:info [default], 4:trace, 5:xtrace, 6:debug");

unsigned int MsiMode = 1;
module_param(MsiMode, int, 0444);
MODULE_PARM_DESC(MsiMode, " 0:legacy, 1:MSI/MSI-X [default: 1]");

unsigned int ErrorReporting = 0;
module_param(ErrorReporting, int, 0444);
MODULE_PARM_DESC(ErrorReporting, " PCIe AER, bit field: 1:device, 2:bridge [default: 0]");



const char DriverVersionString[] = DRIVER_VERSION_STR " " DRIVER_BUILD;

#ifdef BACKLOG
struct cs_backlog_t Backlog;
#pragma message "Backlog is activated"
#endif

static int          MajorNumber = -1;
static struct class *DeviceClass = NULL;

#define PROCNAME "driver/cryptoserver"
static struct proc_dir_entry *ProcEntry = NULL;

static unsigned int NumCards = 0;

#define DEVICE_TYPE_PHYS       0
#define DEVICE_TYPE_VIRT       1
#define DEVICE_TYPE_LEGACY     2


const char *LogLevelTxt[8] =
{
  "[0] None",
  "[1] Error",
  "[2] Warning",
  "[3] Info",
  "[4] Trace",
  "[5] Debug",
  "?6!",
  "?7!"
};

const char *IrqModeTxt[4] =
{
   "legacy",
   "MSI",
   "MSI-multi",
   "MSI-X"
};

const char *ModelTxt[16] =
{
  "0 [CS2000]",
  "1 [CS2]",
  "2 [Se-Series]",
  "3 [NQ3]",
  "4 [CSe-Series]",
  "5 [Se-Series Gen2]",
  "6 [?]",
  "7 [CSAR-Prototype]",
  "8 [CSAR-Series]",
  "9 [?]",
  "10 [?]",
  "11 [?]",
  "12 [?]",
  "13 [?]",
  "14 [?]",
  "15 [?]"
};

int cs2_probe(struct pci_dev *pci_dev, struct cs_device_t **pp_dp);
int cs3_probe(struct pci_dev *pci_dev, struct cs_device_t **pp_dp);
int cs8_probe(struct pci_dev *pci_dev, struct cs_device_t **pp_dp); //  __attribute__((weak));

static int (*probe_func[])(struct pci_dev *pci_dev, struct cs_device_t **pp_dp) =
{
  NULL,
  cs2_probe,
  cs3_probe,
  NULL,
  cs3_probe,
  cs3_probe,
  NULL,
  NULL,
  cs8_probe,
};

/******************************************************************************
 *
 * Macros
 *
 *****************************************************************************/

// force newer defination in order to prevent compiler warnings on some platforms
#undef IS_ERR_VALUE
#define IS_ERR_VALUE(x) unlikely((unsigned long)(void *)(x) >= (unsigned long)-MAX_ERRNO)

// IDR API

#define CS_MAX_MINORS 256

#define IDR_MIN_VERSION KERNEL_VERSION(3,9,0)

static DEFINE_MUTEX(cs_devs_idr_lock);
#define cs_id_lock()    mutex_lock(&cs_devs_idr_lock)
#define cs_id_unlock()  mutex_unlock(&cs_devs_idr_lock)

#if LINUX_VERSION_CODE >= IDR_MIN_VERSION
  static DEFINE_IDR(cs_devs_idr);
  #define cs_id_destroy()   idr_destroy(&cs_devs_idr)
#else
  static struct cs_device_t *DeviceTab[CS_MAX_MINORS] = { 0 };
  #define cs_id_destroy()
#endif

/******************************************************************************
 *
 * cs_id_alloc
 *
 *****************************************************************************/
static int cs_id_alloc(struct cs_device_t *dp)
{
  int minor = -ENOSPC;

  cs_id_lock();

#if LINUX_VERSION_CODE >= IDR_MIN_VERSION
  minor = idr_alloc(&cs_devs_idr, dp, 0, CS_MAX_MINORS, GFP_KERNEL);
#else
  {
    int id;

    for (id=0; id<CS_MAX_MINORS; id++)
    {
      if (DeviceTab[id] == NULL)
      {
        minor = id;
        DeviceTab[id] = dp;
        break;
      }
    }
  }
#endif

  cs_id_unlock();
  return minor;
}

/******************************************************************************
 *
 * cs_id_free
 *
 *****************************************************************************/
static void cs_id_free(struct cs_device_t *dp)
{
  if (  dp->minor >= 0
     && dp->minor < CS_MAX_MINORS
     )
  {
    cs_id_lock();

#if LINUX_VERSION_CODE >= IDR_MIN_VERSION
    idr_remove(&cs_devs_idr, dp->minor);
#else
    if (  DeviceTab[dp->minor] != NULL
       && DeviceTab[dp->minor]->minor == dp->minor
       )
    {
      DeviceTab[dp->minor] = NULL;
    }
#endif

    cs_id_unlock();
  }
}

/******************************************************************************
 *
 * cs_id_get_next
 *
 *****************************************************************************/
static inline struct cs_device_t *cs_id_get_next(int *pid)
{
#if LINUX_VERSION_CODE >= IDR_MIN_VERSION
  return idr_get_next(&cs_devs_idr, pid);
#else
  {
    int id;

    for (id=*pid; id<CS_MAX_MINORS; id++)
    {
      if (DeviceTab[id] != NULL)
      {
        *pid = id;
        return DeviceTab[id];
      }
    }

    return NULL;
  }
#endif
}

/******************************************************************************
 *
 * cs_id_find
 *
 *****************************************************************************/
static inline struct cs_device_t *cs_id_find(int minor)
{
  struct cs_device_t *dp;

  cs_id_lock();

#if LINUX_VERSION_CODE >= IDR_MIN_VERSION
  dp = idr_find(&cs_devs_idr, minor);
#else
  if (  DeviceTab[minor] != NULL
     && DeviceTab[minor]->minor == minor
     )
  {
    dp = DeviceTab[minor];
  }
  else
  {
    dp = NULL;
  }
#endif

  cs_id_unlock();

  return dp;
}

/******************************************************************************
 *
 * cs_pm_suspend
 *
 *****************************************************************************/
static int cs_pm_suspend(struct device *dev)
{
  struct pci_dev *pci_dev;
  struct cs_device_t *dp;

  if (  dev == NULL
     || (pci_dev = to_pci_dev(dev)) == NULL
     || (dp = pci_get_drvdata(pci_dev)) == NULL
     )
    return -EINVAL;

  log_info("device [%s] is going to suspend\n", dp->device_name);

  if (dev_is_pf(&pci_dev->dev))
  {
    // remove virtual functions
    unsigned int num_vf = pci_num_vf(pci_dev);
    pci_disable_sriov(pci_dev);
    log_info("%d virtual functions have been disabled\n", num_vf);
  }

  if (dp->p_suspend != NULL) dp->p_suspend(dp);

  pci_save_state(pci_dev);

  return 0;
}

/******************************************************************************
 *
 * cs_pm_resume
 *
 *****************************************************************************/
static int cs_pm_resume(struct device *dev)
{
  struct pci_dev *pci_dev;
  struct cs_device_t *dp;

  if (  dev == NULL
     || (pci_dev = to_pci_dev(dev)) == NULL
     || (dp = pci_get_drvdata(pci_dev)) == NULL
     )
    return -EINVAL;

  log_info("device [%s] was resumed\n", dp->device_name);

  pci_restore_state(pci_dev);

  if (dp->p_resume != NULL) dp->p_resume(dp);

  return 0;
}

/******************************************************************************
 *
 * cs_pm_poweroff
 *
 *****************************************************************************/
static int cs_pm_poweroff(struct device *dev)
{
  struct pci_dev *pci_dev;
  struct cs_device_t *dp;

  if (  dev == NULL
     || (pci_dev = to_pci_dev(dev)) == NULL
     || (dp = pci_get_drvdata(pci_dev)) == NULL
     )
    return -EINVAL;

  log_info("device [%s] is going to power off\n", dp->device_name);

  pci_save_state(pci_dev);

  if (dp->p_suspend != NULL) dp->p_suspend(dp);

  return 0;
}

/******************************************************************************
 *
 * cs_pm_restore
 *
 *****************************************************************************/
static int cs_pm_restore(struct device *dev)
{
  struct pci_dev *pci_dev;
  struct cs_device_t *dp;

  if (  dev == NULL
     || (pci_dev = to_pci_dev(dev)) == NULL
     || (dp = pci_get_drvdata(pci_dev)) == NULL
     )
    return -EINVAL;

  log_info("device [%s] was restored\n", dp->device_name);

  pci_restore_state(pci_dev);

  if (dp->p_resume != NULL) dp->p_resume(dp);

  return 0;
}

/******************************************************************************
 *
 * cs_eh_error_detected
 *
 *****************************************************************************/
static pci_ers_result_t cs_eh_error_detected(struct pci_dev *pci_dev, pci_channel_state_t state)
{
  const char *states[] = { "IO channel is normal", "IO channel is frozen", "PCI card is dead", "?3!" };
  struct cs_device_t *dp = pci_get_drvdata(pci_dev);

  log_error("device [%s]: error detected, state: %d [%s]\n", dp->device_name, state, states[state&3]);

//return PCI_ERS_RESULT_CAN_RECOVER;
  return PCI_ERS_RESULT_NEED_RESET;
//return PCI_ERS_RESULT_DISCONNECT;
}

/******************************************************************************
 *
 * cs_eh_mmio_enabled
 *
 *****************************************************************************/
static pci_ers_result_t cs_eh_mmio_enabled(struct pci_dev *pci_dev)
{
  struct cs_device_t *dp = pci_get_drvdata(pci_dev);

  log_info("device [%s]: MMIO has been re.enabled\n", dp->device_name);

  return PCI_ERS_RESULT_NONE;
}

/******************************************************************************
 *
 * cs_eh_slot_reset
 *
 *****************************************************************************/
static pci_ers_result_t cs_eh_slot_reset(struct pci_dev *pci_dev)
{
  struct cs_device_t *dp = pci_get_drvdata(pci_dev);

  log_info("device [%s]: slot has been reset\n", dp->device_name);

  return PCI_ERS_RESULT_NONE;
}

/******************************************************************************
 *
 * cs_eh_reset_prepare
 *
 *****************************************************************************/
static void cs_eh_reset_prepare(struct pci_dev *pci_dev)
{
  struct cs_device_t *dp = pci_get_drvdata(pci_dev);

  log_info("device [%s]\n", dp->device_name);

  if (dp->p_reset_prepare != NULL) dp->p_reset_prepare(dp);
}

/******************************************************************************
 *
 * cs_eh_reset_done
 *
 *****************************************************************************/
static void cs_eh_reset_done(struct pci_dev *pci_dev)
{
  struct cs_device_t *dp = pci_get_drvdata(pci_dev);

  log_info("device [%s]\n", dp->device_name);

  if (dp->p_reset_done != NULL) dp->p_reset_done(dp);
}

/******************************************************************************
 *
 * cs_eh_reset_notify
 *
 *****************************************************************************/
static void __maybe_unused cs_eh_reset_notify(struct pci_dev *pci_dev, bool prepare)
{
  if (prepare)
  {
    cs_eh_reset_prepare(pci_dev);
  }
  else
  {
    cs_eh_reset_done(pci_dev);
  }
}

/******************************************************************************
 *
 * cs_eh_resume
 *
 *****************************************************************************/
static void cs_eh_resume(struct pci_dev *pci_dev)
{
  struct cs_device_t *dp = pci_get_drvdata(pci_dev);

  log_info("device [%s]: may resume now\n", dp->device_name);
}

/******************************************************************************
 *
 * cs_shutdown
 *
 ******************************************************************************/
static void cs_shutdown(struct pci_dev *pci_dev)
{
  struct cs_device_t *dp = pci_get_drvdata(pci_dev);

  log_info("device [%s] going to shut down [%d]\n", dp->device_name, system_state);

  switch (system_state)
  {
    case SYSTEM_HALT:
    case SYSTEM_POWER_OFF:
    case SYSTEM_RESTART:
      if (dp->p_shutdown != NULL) dp->p_shutdown(dp, system_state);
      break;

    default:
      break;
  }
}

/******************************************************************************
 *
 * cs_sriov_configure
 *
 *****************************************************************************/
static int cs_sriov_configure(struct pci_dev *pci_dev, int numvfs)
{
  int err;

  log_info("devfn:%d, numvfs:%d\n", pci_dev->devfn, numvfs);

  if (numvfs > 0)
  {
    if ((err = pci_enable_sriov(pci_dev, numvfs)) != 0)
    {
      log_error("pci_enable_sriov(%d) returned: %d\n", numvfs, err);
      return err;
    }

    return numvfs;
  }
  else
  {
    pci_disable_sriov(pci_dev);
    return 0;
  }
}

/******************************************************************************
 *
 * cs_dev_find
 *
 *****************************************************************************/
static struct cs_device_t *cs_dev_find(struct pci_dev *pci_dev)
{
  struct pci_dev *physfn;
  struct cs_device_t *dp;
  int i;

  if ((physfn = pci_dev->physfn) == NULL) return NULL;

  cs_id_lock();

  for (i = 0; (dp = cs_id_get_next(&i)) != NULL; ++i)
  {
    if (dp == NULL) continue;

    if (  dp->pci_dev == physfn           // same physical function
       && dp->subfn == pci_dev->devfn     // same channel
       )
    {
      goto end;
    }
  }

  dp = NULL;

end:
  cs_id_unlock();
  return dp;
}

/******************************************************************************
 *
 * cs_dev_attr_channel_show
 *
 *****************************************************************************/
static ssize_t cs_dev_attr_channel_show(struct device *dev, struct device_attribute *attr, char *buf)
{
  struct cs_device_t *dp;

  if (  attr == NULL
     || (dp = container_of(attr, struct cs_device_t, dev_attr_channel)) == NULL
     )
    return 0;

  return scnprintf(buf, PAGE_SIZE, "%d\n", dp->subfn);
}

/******************************************************************************
 *
 * cs_dev_attr_parent_show
 *
 *****************************************************************************/
static ssize_t cs_dev_attr_parent_show(struct device *dev, struct device_attribute *attr, char *buf)
{
  struct cs_device_t *dp;
  struct cs_device_t *parent;

  if (  attr == NULL
     || (dp = container_of(attr, struct cs_device_t, dev_attr_parent)) == NULL
     || (parent = dp->parent) == NULL
     )
    return 0;

  return scnprintf(buf, PAGE_SIZE, "%s\n", parent->device_name);
}

/******************************************************************************
 *
 * cs_open
 *
 *****************************************************************************/
static int cs_open(struct inode *inode, struct file *file)
{
  int err;
  struct cs_device_t *dp;
  struct cs_session_t *session = NULL;
  int minor = iminor(inode);

  if ((dp = cs_id_find(minor)) == NULL)
  {
    log_error("no such device: %d\n", minor);
    return -ENODEV;
  }

  if (dp->pci_dev == NULL)
  {
    log_error("no such PCI device: %d\n", minor);
    return -ENODEV;
  }

  if ((session = kzalloc(sizeof(struct cs_session_t), GFP_KERNEL)) == NULL) return(-ENOMEM);

  session->dp = dp;
  session->file = file;
  spin_lock_init(&session->lock);

  if ((err = dp->p_open(session)) != 0)
  {
    log_error("open(%s)returned: %d\n", dp->device_name, err);
    goto cleanup;
  }

  dp->use_cnt++;

  file->private_data = session;
  return 0;

cleanup:
  if (session != NULL)
  {
    memset(session, 0, sizeof(struct cs_session_t));
    kfree(session);
  }
  return err;
}

/******************************************************************************
 *
 * cs_close
 *
 *****************************************************************************/
static int cs_close(struct inode *inode, struct file *file)
{
  int err;
  struct cs_session_t *session = file->private_data;
  struct cs_device_t *dp;

  if (session == NULL) return -EINVAL;

  if ((dp = session->dp) == NULL) CLEANUP(-ENODEV);

  if (dp->minor != iminor(inode))
  {
    log_error("no such device: %d\n", dp->minor);
    CLEANUP(-ENODEV);
  }

  dp->use_cnt--;

  dp->p_close(session);

cleanup:
  memset(session, 0, sizeof(struct cs_session_t));
  kfree(session);
  file->private_data = NULL;
  return err;
}


/******************************************************************************
 *
 * cs_ioctl_unlocked
 *
 *****************************************************************************/
static long cs_ioctl_unlocked(struct file *file, unsigned int cmd, unsigned long arg)
{
  struct cs_session_t *session = file->private_data;
  struct cs_device_t *dp;

  if (session == NULL) return -EINVAL;

  if ((dp = session->dp) == NULL) return -ENODEV;

  return dp->p_ioctl(session, cmd, arg);
}

/******************************************************************************
 *
 * cs_read
 *
 *****************************************************************************/
static ssize_t cs_read(struct file *file, char __user *buf, size_t count, loff_t *ppos)
{
  struct cs_session_t *session = file->private_data;
  struct cs_device_t *dp;

  if (session == NULL) return -EINVAL;

  if ((dp = session->dp) == NULL) return -ENODEV;

  return dp->p_read(session, buf, count);
}

/******************************************************************************
 *
 * cs_write
 *
 *****************************************************************************/
static ssize_t cs_write(struct file *file, const char __user *buf, size_t count, loff_t *ppos)
{
  struct cs_session_t *session = file->private_data;
  struct cs_device_t *dp;

  if (session == NULL) return -EINVAL;

  if ((dp = session->dp) == NULL) return -ENODEV;

  return dp->p_write(session, buf, count);
}

/******************************************************************************
 *
 * cs_poll
 *
 *****************************************************************************/
static unsigned int cs_poll(struct file *file, poll_table *wait)
{
  struct cs_session_t *session = file->private_data;
  struct cs_device_t *dp;

  if (session == NULL) return -EINVAL;

  if ((dp = session->dp) == NULL) return -ENODEV;

  if (dp->p_poll == NULL) return -ENOSYS;

  return dp->p_poll(session, file, wait);
}

//-----------------------------------------------------------------------------
static struct file_operations cs_file_ops =
//-----------------------------------------------------------------------------
{
  owner:          THIS_MODULE,
  open:           cs_open,
  release:        cs_close,
  read:           cs_read,
  write:          cs_write,
  poll:           cs_poll,
  unlocked_ioctl: cs_ioctl_unlocked,
  compat_ioctl:   cs_ioctl_unlocked
};

/******************************************************************************
 *
 * cs_remove
 *
 ******************************************************************************/
static void cs_remove(struct pci_dev *pci_dev)
{
  int err;
  struct cs_device_t *parent = pci_get_drvdata(pci_dev);
  struct cs_device_t *dp;
  int minor;

  if (parent == NULL) return;

  minor = parent->minor;

  log_trace("removing: cs2.%d ...\n", minor);

  // remove virtual functions
  if (dev_is_pf(&pci_dev->dev))
  {
    pci_disable_sriov(pci_dev);
  }

  for (dp=parent; dp != NULL; dp=dp->next)
  {
    // remove proc entry
    if (dp->proc_entry != NULL)
    {
      remove_proc_entry(dp->proc_name, NULL);
      dp->proc_entry = NULL;
      log_trace("removed proc entry: %s\n", dp->proc_name);
    }

    // remove sysfs files
    if (dp->dev_attr_channel.attr.name != NULL)
      device_remove_file(dp->device, &dp->dev_attr_channel);

    if (dp->dev_attr_parent.attr.name != NULL)
      device_remove_file(dp->device, &dp->dev_attr_parent);

    // delete device node
    device_destroy(DeviceClass, MKDEV(MajorNumber, dp->minor));

    // free minor id
    cs_id_free(dp);
  }

  // call device specific remove function
  if (parent->p_remove != NULL)
  {
    if ((err = parent->p_remove(parent)) != 0)
    {
      log_error("remove of cs2.%d returned: %d\n", minor, err);
    }
  }

  log_info("removed cs2.%d\n", minor);

  pci_set_drvdata(pci_dev, NULL);
}

/******************************************************************************
 *
 * cs_probe
 *
 ******************************************************************************/
static int cs_probe(struct pci_dev *pci_dev, const struct pci_device_id *id)
{
  int                 err = 0;
  unsigned int        model = id->driver_data & 0xF;
  unsigned int        cardno;
  unsigned int        type;
  char                *types[] = { "phys", "virt", "legacy", "???" };
  struct device       *device;
  struct pci_dev      *physfn = NULL;
  struct cs_device_t  *parent = NULL;
  struct cs_device_t  *dp;

  int (*p_probe)(struct pci_dev *pci_dev, struct cs_device_t **pp_dp);

  if (pci_dev->is_physfn)
  {
    type = DEVICE_TYPE_PHYS;
    cardno = NumCards++;
  }
  else if (pci_dev->is_virtfn)
  {
    if ((physfn = pci_dev->physfn) == NULL) return -ENODEV;
    type = DEVICE_TYPE_VIRT;
    cardno = ((struct cs_device_t *)pci_get_drvdata(physfn))->cardno;
  }
  else
  {
    // legacy device without SRIOV support
    type = DEVICE_TYPE_LEGACY;
    cardno = NumCards++;
  }

  if (  model >= DIM(probe_func)
     || (p_probe = probe_func[model]) == NULL
     )
  {
    log_error("unsupported device/model: %d\n", model);
    return -ENODEV;
  }

  log_info("device found: %04x:%04x %04x:%04x model:%s devfn:%d %s\n",
                   pci_dev->vendor,
                   pci_dev->device,
                   pci_dev->subsystem_vendor,
                   pci_dev->subsystem_device,
                   ModelTxt[model],
                   pci_dev->devfn, types[type&3]);

  // check if function is already used in non-SRIOV mode
  if (cs_dev_find(pci_dev) != NULL)
  {
    log_error("devfn %d is already in use\n", pci_dev->devfn);
    CLEANUP(-EEXIST);
  }

  // call device specific probe function
  if ((err = p_probe(pci_dev, &parent)) != 0)
  {
    log_error("probe returned: %d\n", err);
    return err;
  }

  pci_set_drvdata(pci_dev, parent);

  for (dp=parent; dp != NULL; dp=dp->next)
  {
    if ((dp->minor = cs_id_alloc(dp)) < 0)
    {
      log_error("can't allocate new minor number\n");
      CLEANUP(dp->minor);
    }

    sprintf(dp->device_name, "cs2.%d", dp->minor);
    sprintf(dp->proc_name, "driver/cs2.%d", dp->minor);

    dp->model = model;

    if (type == DEVICE_TYPE_VIRT)
    {
      dp->parent = (struct cs_device_t *)pci_get_drvdata(physfn);
    }
    else
    {
      dp->parent = parent;
    }

    dp->cardno = cardno;

    strncpy(dp->slot, dev_name(&pci_dev->dev), sizeof(dp->slot));

    spin_lock_init(&dp->lock);

    // create proc entry for device
    if ((dp->proc_entry = proc_create_data(dp->proc_name, 0666, NULL, dp->proc_ops, dp)) == NULL)
    {
      log_error("proc_create_data(%s) failed\n", dp->proc_name);
      CLEANUP(-ENOMEM);
    }

    log_trace("successfully created proc entry: %s", dp->proc_name);

    // create device node
    device = device_create(DeviceClass, &pci_dev->dev, MKDEV(MajorNumber, dp->minor), NULL, dp->device_name);

    if (IS_ERR_VALUE(device))
    {
      err = PTR_ERR(device);
      log_error("device_create(%s) returned: %d\n", dp->device_name, err);
      goto cleanup;
    }

    dp->device = device;

    // create sysfs files
    dp->dev_attr_channel.attr.name = "channel";
    dp->dev_attr_channel.attr.mode = 0444;
    dp->dev_attr_channel.show = cs_dev_attr_channel_show;

    if ((err = device_create_file(device, &dp->dev_attr_channel)) < 0)
    {
      dp->dev_attr_channel.attr.name = NULL;
      log_error("device_create_file(dev_attr_channel) returned: %d\n", err);
    }

    dp->dev_attr_parent.attr.name = "parent";
    dp->dev_attr_parent.attr.mode = 0444;
    dp->dev_attr_parent.show = cs_dev_attr_parent_show;

    if ((err = device_create_file(device, &dp->dev_attr_parent)) < 0)
    {
      dp->dev_attr_parent.attr.name = NULL;
      log_error("device_create_file(dev_attr_parent) returned: %d\n", err);
    }

    log_info("probe %s OK\n", dp->device_name);
  }

  return 0;

cleanup:
  cs_remove(pci_dev);
  return err;
}


//-----------------------------------------------------------------------------
static struct pci_device_id cs_pci_tbl[] =
//-----------------------------------------------------------------------------
{
  { 0x11e3, 0x0006,             // QuickLogic vendor / device id
    0x168a, 0x1110,             // Utimaco sub-vendor / sub-device id
    0, 0, 1                     // model := 1
  },
  { 0x168a, 0x2086,             // Utimaco vendor / device id
    PCI_ANY_ID, PCI_ANY_ID,
    0, 0,                       // class, class mask
    2                           // model := 2 (Se-Series)
  },
  { 0x168a, 0xC040,             // Utimaco vendor / device id
    0x168a, 0x2708,             // Utimaco sub-vendor / sub-device id
    0, 0,                       // class, class mask
    4                           // model := 4 (CSe-Series)
  },
  { 0x168a, 0xC051,             // Utimaco vendor / device id
    0x168a, 0x2708,             // Utimaco sub-vendor / sub-device id
    0, 0,                       // class, class mask
    5                           // model := 5 (Se2-Series)
  },
  {
    0x168A, 0xC070,
    PCI_ANY_ID, PCI_ANY_ID,
    0, 0,                       // class, class mask
    7                           // model := 7
  },
  {
    0x168A, 0xC071,
    PCI_ANY_ID, PCI_ANY_ID,
    0, 0,                       // class, class mask
    8                           // model := 8 [PF]
  },
  {
    0x168A, 0xC072,
    PCI_ANY_ID, PCI_ANY_ID,
    0, 0,                       // class, class mask
    8                           // model := 8 [VF]
  },
  { 0 }
};

//-----------------------------------------------------------------------------
static const struct pci_error_handlers cs_error_handler =
//-----------------------------------------------------------------------------
{
  .error_detected = cs_eh_error_detected,
  .mmio_enabled   = cs_eh_mmio_enabled,
  .slot_reset     = cs_eh_slot_reset,

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,13,0)
  .reset_prepare  = cs_eh_reset_prepare,
  .reset_done     = cs_eh_reset_done,
#elif (  LINUX_VERSION_CODE >= KERNEL_VERSION(3,16,0) \
      && !defined(CONFIG_SUSE_KERNEL) \
      )
  .reset_notify   = cs_eh_reset_notify,
#endif

  .resume         = cs_eh_resume
};


MODULE_DEVICE_TABLE(pci, cs_pci_tbl);
MODULE_ALIAS("cs2");

//-----------------------------------------------------------------------------
static struct dev_pm_ops cs_pm_ops =
//-----------------------------------------------------------------------------
{
//.prepare          =
//.complete         =
  .suspend          = cs_pm_suspend,
  .resume           = cs_pm_resume,
//.freeze           =
//.thaw             =
  .poweroff         = cs_pm_poweroff,
  .restore          = cs_pm_restore,
//.suspend_late     =
//.resume_early     =
//.freeze_late      =
//.thaw_early       =
//.poweroff_late    =
//.restore_early    =
//.suspend_noirq    =
//.resume_noirq     =
//.freeze_noirq     =
//.thaw_noirq       =
//.poweroff_noirq   =
//.restore_noirq    =
//.runtime_suspend  =
//.runtime_resume   =
//.runtime_idle     =
};

/*
 * PM operation sequences:
 *
 * suspend            : prepare -> suspend -> suspend_late -> suspend_noirq
 * resume             : resume_noirq -> resume_early -> resume -> complete
 * hibernate stage 1  : prepare -> freeze -> freeze_late -> freeze_noirq -> thaw_early -> thaw -> complete
 * hibernate stage 2  : prepare -> poweroff -> poweroff_late -> poweroff_noriq -> complete
 * restore            : restore_noirq -> restore_early -> restore -> complete
 */

// static SIMPLE_DEV_PM_OPS(cs_pm_ops, cs_pm_suspend, cs_pm_resume);


//-----------------------------------------------------------------------------
static struct pci_driver cs_driver =
//-----------------------------------------------------------------------------
{
  .name             = DRIVER_NAME,
  .id_table         = cs_pci_tbl,
  .probe            = cs_probe,
  .remove           = cs_remove,
  .shutdown         = cs_shutdown,
//.suspend          = cs_suspend_legacy,      // legacy
//.resume           = cs_resume_legacy,       // legacy
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 8, 0)
  .sriov_configure  = cs_sriov_configure,
#endif
  .err_handler      = &cs_error_handler,
  .driver.pm        = &cs_pm_ops
};

/******************************************************************************
 *
 * cs_devnode
 *
 *****************************************************************************/
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 33)
#if (  LINUX_VERSION_CODE >= KERNEL_VERSION(6, 2, 0) \
    || (RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION(8,9) ) \
    )
static char *cs_devnode(const struct device *dev, umode_t *mode)
#else
static char *cs_devnode(struct device *dev, umode_t *mode)
#endif
{
  if (mode != NULL)
  {
    *mode = 0666;
    log_debug("cs_devnode: [%s] mode: %u\n", dev_name(dev), *mode);
  }

  return kasprintf(GFP_KERNEL, "%s", dev_name(dev));
}
#endif

/******************************************************************************
 *
 * cs_proc_show
 *
 ******************************************************************************/
static int cs_proc_show(struct seq_file *m, void *v)
{
  struct pci_dev *pci_dev;
  struct cs_device_t *dp;
  int i;
  char buf[256];
  int max = sizeof(buf)-1;
  int len;
  unsigned int cardno;
  char alias[32];

  seq_printf(m, DRIVER_NAME" driver version "DRIVER_VERSION_STR" "DRIVER_BUILD"\n");

  seq_printf(m, "\n %4s %-14s %-5s %-7s %-9s %-4s %-8s %s\n", "card", "slot", "model", "device", "alias", "used", "state", "type");

  cs_id_lock();

  for (cardno = 0; cardno < NumCards; cardno++)
  {
    seq_printf(m, "---------------------------------------------------------------------------------\n");

    for (i = 0; (dp = cs_id_get_next(&i)) != NULL; ++i)
    {
      if (  dp == NULL
         || dp->cardno != cardno
         )
        continue;

      pci_dev = dp->pci_dev;

      if (dp->model == 8)
        scnprintf(alias, sizeof(alias)-1, "%s.%d", dp->parent->device_name, dp->subfn);
      else
        strcpy(alias, "-");

      len = scnprintf(buf, max, " %-4d %-14.14s %-5d %-7s %-9s %-4d", dp->cardno, dp->slot, dp->model, dp->device_name, alias, dp->use_cnt);

      len += scnprintf(buf+len, max-len, " %-8s", (dp->p_state != NULL) ? dp->p_state(dp) : "n/a");

      if (pci_dev->is_virtfn)
      {
        len += scnprintf(buf+len, max-len, " virtfn -> %s", dp->parent->device_name);
      }
      else if (pci_dev->is_physfn)
      {
        if (dp->subfn == 0)
          len += scnprintf(buf+len, max-len, " physfn (%d/%d vf used)", pci_num_vf(pci_dev), pci_sriov_get_totalvfs(pci_dev));
        else
          len += scnprintf(buf+len, max-len, " child  -> %s", dp->parent->device_name);
      }
      else
      {
        if (dp->subfn == 0)
          len += scnprintf(buf+len, max-len, " legacy");
        else
          len += scnprintf(buf+len, max-len, " child -> %s", dp->parent->device_name);
      }

      buf[len] = 0;

      seq_printf(m, "%s\n", buf);
    }
  }

  cs_id_unlock();

  return 0;
}

/******************************************************************************
 *
 * cs_proc_write
 *
 *****************************************************************************/
static ssize_t cs_proc_write(struct file *file, const char __user *user, size_t count, loff_t *off)
{
  int   err = 0;
  char  *buf = NULL;

  if ((buf = kzalloc(count+1, GFP_KERNEL)) == NULL) return -ENOMEM;

  if (copy_from_user(buf, user, count) != 0) CLEANUP(-EFAULT);

  if (strncasecmp(buf, "TEST", 4) == 0)
  {
    log_info("testing ...\n");

    // ...

    log_info("done\n");
  }
#ifdef BACKLOG
  else if (strncasecmp(buf, "BACKLOG", 7) == 0)
  {
    cs_backlog_print(&Backlog);
  }
#endif
  else if (strncasecmp(buf, "RESET", 5) == 0)
  {
    struct cs_device_t *dp;
    unsigned int cardno;
    int i;

    cs_id_lock();

    for (cardno = 0; cardno < NumCards; cardno++)
    {
      for (i = 0; (dp = cs_id_get_next(&i)) != NULL; ++i)
      {
        if (  dp == NULL
           || dp->cardno != cardno
           )
          continue;

        if (dp->p_reset_prepare != NULL)
        {
          dp->p_reset_prepare(dp);

          if (dp->p_reset_done != NULL)
          {
            dp->p_reset_done(dp);
          }
        }
      }
    }

    cs_id_unlock();
  }
  else
  {
    log_error("invalid argument: %s\n", buf);
    CLEANUP(-EINVAL);
  }

  if (err < 0)
  {
    log_error("'%s' returned: %d\n", buf, err);
    goto cleanup;
  }

cleanup:
  if (buf != NULL) kfree(buf);

  return (err != 0) ? err : count;
}

/******************************************************************************
 *
 * cs_proc_open
 *
 *****************************************************************************/
static int cs_proc_open(struct inode *inode, struct file *file)
{
  return single_open(file, cs_proc_show, compat_PDE_DATA(inode));
}

//-----------------------------------------------------------------------------
static const proc_op_t ProcOps =
//-----------------------------------------------------------------------------
{
#ifdef HAVE_PROC_OPS
  .proc_open    = cs_proc_open,
  .proc_read    = seq_read,
  .proc_write   = cs_proc_write,
  .proc_lseek   = seq_lseek,
  .proc_release = single_release,
#else
  .owner   = THIS_MODULE,
  .open    = cs_proc_open,
  .read    = seq_read,
  .write   = cs_proc_write,
  .llseek  = seq_lseek,
  .release = single_release,
#endif
};

/******************************************************************************
 *
 * cs_init_module
 *
 ******************************************************************************/
static int cs_init_module(void)
{
  int err;

  // init global variables
  DeviceClass = NULL;

  log_info(""DRIVER_NAME" driver version "DRIVER_VERSION_STR" "DRIVER_BUILD"\n");

  if (LogLevel > LOG_LEVEL_DEBUG) LogLevel = LOG_LEVEL_DEBUG;
  MsiMode &= 3;

  crc16_init();

#ifdef BACKLOG
  log_info("backlog is activated\n");
  memset(&Backlog, 0, sizeof(Backlog));
#endif

#if (RHEL_RELEASE_CODE != 0)
  log_info("RHEL_RELEASE_CODE: 0x%x\n", RHEL_RELEASE_CODE);
  log_trace("RHEL_MAJOR       : 0x%x\n", RHEL_MAJOR);
  log_trace("RHEL_MINOR       : 0x%x\n", RHEL_MINOR);
#endif

#if (SLE_VERSION_CODE != 0)
  log_info("SLE_VERSION_CODE: 0x%x\n", SLE_VERSION_CODE);
#endif

  // create device class
#if LINUX_VERSION_CODE >= KERNEL_VERSION(6, 4, 0)
  DeviceClass = class_create(CLASS_NAME);
#else
  DeviceClass = class_create(THIS_MODULE, CLASS_NAME);
#endif

  if (IS_ERR_VALUE(DeviceClass))
  {
    err = PTR_ERR(DeviceClass);
    DeviceClass = NULL;
    log_error("class_create(%s) returned: %d\n", CLASS_NAME, err);
    return err;
  }

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 33)
  DeviceClass->devnode = cs_devnode;
#endif

  // register character device
  MajorNumber = register_chrdev(0, CHRDEV_NAME, &cs_file_ops);

  if (MajorNumber < 0)
  {
    err = MajorNumber;
    log_error("register_chrdev(%s) returned: %d\n", CHRDEV_NAME, err);
    return err;
  }

  log_info("Major Number: %d\n", MajorNumber);

  // register driver
  if ((err = pci_register_driver(&cs_driver)) != 0)
  {
    log_error("pci_register_driver returned: %d\n", err);
    return err;
  }

  // create global proc entry
  if ((ProcEntry = proc_create_data(PROCNAME, 0666, NULL, &ProcOps, NULL)) == NULL)
  {
    log_error("proc_create_data(%s) failed\n", PROCNAME);
    return -ENOMEM;
  }

  log_info(""DRIVER_NAME" driver was successfully registered\n");

  usleep_range(10000,100000);

  return 0;
}

/******************************************************************************
 *
 * cs_cleanup_module
 *
 ******************************************************************************/
static void cs_cleanup_module(void)
{
  if (ProcEntry != NULL) remove_proc_entry(PROCNAME, NULL);

  // unregister driver
  pci_unregister_driver(&cs_driver);

  if (DeviceClass != NULL)
  {
    if (MajorNumber >= 0)
    {
      // unregister character device
      unregister_chrdev(MajorNumber, CHRDEV_NAME);
    }

    class_destroy(DeviceClass);
  }

  cs_id_destroy();

  log_info(""DRIVER_NAME" driver unregistered\n");
}


module_init(cs_init_module);
module_exit(cs_cleanup_module);

MODULE_LICENSE("GPL v2");
MODULE_AUTHOR("Utimaco IS GmbH");
MODULE_DESCRIPTION("CryptoServer PCIe device driver");
MODULE_VERSION(DRIVER_VERSION_STR);
