package defaults;

import CryptoServerJCE.*;
import java.security.*;
import java.security.spec.*;

/**
 * This program demonstrates the usage of the Utimaco's JCE Provider 
 * for the CryptoServer Hardware Security Module.
 *
 * RSA signature creation / verification
 *
 */
public class sign_RSA
{      
  public static void main(String[] args) throws Exception 
  {
    System.out.println("\n--- Utimaco CryptoServer JCE - sign_RSA ---\n");
    
    int sizes[] = { 1024, 2048, 3072 };
    
    Algorithm modes[] = 
    {                   
      new Algorithm("SHA1withRSA",   null),
      new Algorithm("SHA224withRSA", null),
      new Algorithm("SHA256withRSA", null),      
      new Algorithm("SHA384withRSA", null),           
      new Algorithm("SHA512withRSA", null),
      new Algorithm("SHA3-224withRSA", null),
      new Algorithm("SHA3-256withRSA", null),      
      new Algorithm("SHA3-384withRSA", null),           
      new Algorithm("SHA3-512withRSA", null),
      new Algorithm("MD5withRSA",    null),
      new Algorithm("SHA1withRSA",   PSSParameterSpec.DEFAULT ),
      new Algorithm("SHA1withRSA",   new PSSParameterSpec("SHA-1", "MGF1", MGF1ParameterSpec.SHA256, 32, 1)),
      new Algorithm("SHA224withRSA", new PSSParameterSpec("SHA-224", "MGF1", MGF1ParameterSpec.SHA1, 32, 1)),
      new Algorithm("SHA256withRSA", new PSSParameterSpec("SHA-256", "MGF1", MGF1ParameterSpec.SHA256, 0, 1)),
      new Algorithm("SHA384withRSA", new PSSParameterSpec("SHA-384", "MGF1", MGF1ParameterSpec.SHA224, 48, 1)),       
      new Algorithm("SHA512withRSA", new PSSParameterSpec("SHA-512", "MGF1", MGF1ParameterSpec.SHA512, 20, 1)),
      new Algorithm("SHA3-224withRSA", new PSSParameterSpec("SHA3-224", "MGF1", MGF1ParameterSpec.SHA1, 32, 1)),
      new Algorithm("SHA3-256withRSA", new PSSParameterSpec("SHA3-256", "MGF1", MGF1ParameterSpec.SHA384, 0, 1)),
      new Algorithm("SHA3-384withRSA", new PSSParameterSpec("SHA3-384", "MGF1", MGF1ParameterSpec.SHA256, 48, 1)),       
      new Algorithm("SHA3-512withRSA", new PSSParameterSpec("SHA3-512", "MGF1", MGF1ParameterSpec.SHA512, 20, 1)),
    };       
    
    CryptoServerProvider csProvider = null;
    
    try
    {
      // load CryptoServer provider    
      csProvider = new CryptoServerProvider(args.length > 0 ? args[0] : "CryptoServer.cfg");
      System.out.println("Device : " + csProvider.getCryptoServer().getDevice());
    
      // authenticate
      csProvider.loginPassword("JCE", "12345678");

      // list of providers
      Provider[] provider = { csProvider, Security.getProvider("SunRsaSign") };

      // for all combinations
      for (int ct = 0; ct <= 2; ct++)
      {
        System.out.println("\nSign on  : " + provider[ct & 1].getName());
        System.out.println("Verify on: " + provider[(ct >> 1) & 1].getName());               

        // for all key sizes
        for (int size : sizes)
        {
          System.out.println("  size: " + size);

          // generate RSA key
          KeyPairGenerator kg = KeyPairGenerator.getInstance("RSA", provider[ct & 1]);
          kg.initialize(size);
          KeyPair keyPair = kg.generateKeyPair(); 
          PrivateKey privateKey = keyPair.getPrivate();          
          
          // export public key      
          KeyFactory kf = KeyFactory.getInstance("RSA", provider[ct & 1]);
          RSAPublicKeySpec publicKeySpec = kf.getKeySpec(keyPair.getPublic(), RSAPublicKeySpec.class);
          
          // import public key into other provider
          kf = KeyFactory.getInstance("RSA", provider[(ct >> 1) & 1]);
          PublicKey publicKey = (PublicKey)kf.generatePublic(publicKeySpec);
      
          // for all modes
          for (Algorithm mode : modes)
          {
            String algo = mode.algo;
            if (ct > 0 && algo.startsWith("SHA3-"))  // SunRsaSign does not know SHA3 hashes yet
              continue;
            System.out.println("    algo: " + algo + (mode.param != null ? ",PSS" : ""));
          
            // do test        
            Signature sig = Signature.getInstance((ct & 1) != 0 && mode.param != null ? "RSASSA-PSS" : algo, provider[ct & 1]);
            Signature ver = Signature.getInstance(((ct >> 1) & 1) != 0 && mode.param != null ? "RSASSA-PSS" : algo, provider[(ct >> 1) & 1]);

            for (int len1=1; len1<33; len1+=3)
            {    
              byte [] data1 = getRandom(len1);
              
              for (int len2=1; len2<22; len2+=2)
              {        
                byte [] data2 = getRandom(len2);            
                
                // sign
                sig.initSign(privateKey);              
                if (mode.param != null)
                  sig.setParameter(mode.param);             
                sig.update(data1);
                sig.update(data2);
                byte [] sign = sig.sign();
              
                // verify
                byte [] data = cat(data1, data2);
                
                ver.initVerify(publicKey);                
                if (mode.param != null)
                  ver.setParameter(mode.param);
                ver.update(data);
                if (ver.verify(sign) == false)
                  throw new Exception("Signature Verification failed");                                     
              }
            }     
          }
        }
      }
    }
    catch (Exception ex)
    {
      throw ex;
    }
    finally
    {
      // logoff
      if (csProvider != null)
        csProvider.logoff();
    }
    
    System.out.println("Done");
  }        
  
  private static class Algorithm
  {
    String algo;
    AlgorithmParameterSpec param;
    
    public Algorithm(String algo, AlgorithmParameterSpec param)
    {
      this.algo = algo;
      this.param = param;
    }
  }
  
  private static byte [] getRandom(int length)
  {       
    try
    {
      byte[] buf = new byte[length];      
      SecureRandom rng = SecureRandom.getInstance("SHA1PRNG");
      
      do
      {
        rng.nextBytes(buf);
      }
      while (buf[0] == 0);
      
      return buf;
    }
    catch (Exception ex)
    {
      return null;
    }    
  }
  
  private static byte [] cat(byte [] a, byte [] b)
  {
    if(a == null) return(b);
    if(b == null) return(a);

    byte [] res = new byte[a.length + b.length];
    System.arraycopy(a,0,res,0,a.length);
    System.arraycopy(b,0,res,a.length,b.length);

    return(res);
  }
}
