/* SPDX-License-Identifier: GPL-2.0-only
 *
 * Copyright 2024 Utimaco IS GmbH
 * All Rights Reserved.
 *
 */
#ifndef __LOAD_STORE_H_INCLUDED__
#define __LOAD_STORE_H_INCLUDED__

#ifdef WIN32
#define _LD_ST_INLINE _inline
#else
#define _LD_ST_INLINE static inline
#endif

#ifdef LD_ST_LITTLE_ENDIAN

#define store_int2 store_int2_le
#define store_int3 store_int3_le
#define store_int4 store_int4_le

#define load_int2  load_int2_le
#define load_int3  load_int3_le
#define load_int4  load_int4_le

#else

#define store_int2 store_int2_be
#define store_int3 store_int3_be
#define store_int4 store_int4_be

#define load_int2  load_int2_be
#define load_int3  load_int3_be
#define load_int4  load_int4_be

#endif


/******************************************************************************
 *
 * Little Endian
 *
 *****************************************************************************/
_LD_ST_INLINE void store_int2_le(unsigned int val, unsigned char *x)
{
  *x++ = (unsigned char)val;
  *x   = (unsigned char)(val >> 8);
}

_LD_ST_INLINE void store_int3_le(unsigned int val, unsigned char *x)
{
  *x++ = (unsigned char)val;
  *x++ = (unsigned char)(val >> 8);
  *x++ = (unsigned char)(val >> 16);
}

_LD_ST_INLINE void store_int4_le(unsigned int val, unsigned char *x)
{
  *x++ = (unsigned char)val;
  *x++ = (unsigned char)(val >> 8);
  *x++ = (unsigned char)(val >> 16);
  *x   = (unsigned char)(val >> 24);
}

_LD_ST_INLINE unsigned int load_int2_le(unsigned char *x)
{
  return( (x[1] << 8) + x[0] );
}

_LD_ST_INLINE unsigned int load_int3_le(unsigned char *x)
{
  return( (x[2] << 16) + (x[1] << 8) + x[0] );
}

_LD_ST_INLINE unsigned int load_int4_le(unsigned char *x)
{
  return( (x[3] << 24) + (x[2] << 16) + (x[1] << 8) + x[0] );
}


/******************************************************************************
 *
 * Big Endian
 *
 *****************************************************************************/
_LD_ST_INLINE void store_int2_be(unsigned int val,unsigned char *x)
{
  *x++ = (unsigned char)(val >> 8);
  *x   = (unsigned char)val;
}

_LD_ST_INLINE void store_int3_be(unsigned int val,unsigned char *x)
{
  *x++ = (unsigned char)(val >> 16);
  *x++ = (unsigned char)(val >> 8);
  *x   = (unsigned char)val;
}

_LD_ST_INLINE void store_int4_be(unsigned int val,unsigned char *x)
{
  *x++ = (unsigned char)(val >> 24);
  *x++ = (unsigned char)(val >> 16);
  *x++ = (unsigned char)(val >> 8);
  *x   = (unsigned char)val;
}

_LD_ST_INLINE unsigned int load_int2_be(unsigned char *x)
{
  return( (x[0] << 8) + x[1] );
}

_LD_ST_INLINE unsigned int load_int3_be(unsigned char *x)
{
  return( (x[0] << 16) + (x[1] << 8) + x[2] );
}

_LD_ST_INLINE unsigned int load_int4_be(unsigned char *x)
{
  return( (x[0] << 24) + (x[1] << 16) + (x[2] << 8) + x[3] );
}

#undef _LD_ST_INLINE

#endif
