// SPDX-License-Identifier: GPL-2.0-only
/*
 * CryptoServer CS-Series (Classic, PCI) Driver
 *
 * Copyright 2024 Utimaco IS GmbH
 * All Rights Reserved.
 *
 */
#include <linux/version.h>
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/uaccess.h>
#include <linux/seq_file.h>
#include <linux/sched.h>
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,11,0))
#include <linux/sched/signal.h>
#endif

#include "cryptoserver.h"
#include "cs2_drv.h"

/******************************************************************************
 *
 * Definitions
 *
 *****************************************************************************/

// PCI register
#define REG_TRST        0xe8            // total reset
#define BIT_TRST        1
#define REG_SWRST       0xec            // FPGA reset
#define BIT_SWRST       1

#define REG_READ_ADDR_H 0x4c            // master read address (high word)
#define REG_READ_ADDR_L 0x48            // master read address (low word)
#define REG_READ_COUNT  0x50            // master read count

#define REG_READ_ADDR1_H  0x5c          // master read address1 (high word)
#define REG_READ_ADDR1_L  0x58          // master read address1 (low word)
#define REG_READ_COUNT1   0x60          // master read count1

#define REG_WRITE_ADDR_H 0x04           // master write address 0 (high word)
#define REG_WRITE_ADDR_L 0x00           // master write address 0 (low word)
#define REG_WRITE_COUNT  0x08           // master write count 0

#define REG_WRITE_ADDR1_H 0x14          // master write address 1 (high word)
#define REG_WRITE_ADDR1_L 0x10          // master write address 1 (low word)
#define REG_WRITE_COUNT1  0x18          // master write count 1

#define REG_MBR_TX_STAT 0x90            // Outgoing MB Status
#define BIT_MBR_TX_FULL 0x01

#define REG_MBR_TX_H    0x74            // Outgoing MB byte 7 - 4
#define REG_MBR_TX_L    0x70            // Outgoing MB byte 3 - 0

#define REG_MBR_RX_STAT 0x91            // Incomming MB Status
#define BIT_MBR_RX_FULL    1

#define REG_MBR_RX_H    0x7c            // Incomming MB byte 7 - 4
#define REG_MBR_RX_L    0x78            // Incomming MB byte 3 - 0

#define REG_IRQM_DMA    0x8e            // IRQ mask for DMA
#define BIT_IRQM_DMA_RX    1
#define BIT_IRQM_DMA_TX    4
#define REG_IRQM_MBR_RX 0x8d            // IRQ mask for Incomming MBR full
#define BIT_IRQM_MBR_RX    1

#define REG_IRQS_DMA    0x86            // IRQ status for DMA
#define BIT_IRQS_DMA_RX    1
#define BIT_IRQS_DMA_TX    4
#define REG_IRQS_MBR_RX 0x85            // IRQ status for Incomming MBR full
#define BIT_IRQS_MBR_RX    1

#define REG_DMA_ERR     0x96            // DMA error status
#define BIT_DMA_ERR_RX     1
#define BIT_DMA_ERR_TX     4

#define REG_PCI_ERR     0x97            // PCI error status

#define REG_DMA_START   0x92            // DMA Start / Done
#define BIT_DMA_RX_START   1
#define BIT_DMA_TX_START   4

#define REG_DMA_CANCEL  0xd2            // DMA Cancel
#define BIT_DMA_RX_CANCEL  1
#define BIT_DMA_TX_CANCEL  4

#define REG_BATT_STATE  0x1000          // Battery state register
#define BIT_CAR_BATT       1            // carrier battery
#define BIT_EXT_BATT       2            // external battery

// mailbox register commands
#define MBR_TX_REQ      0x00100010
#define MBR_TX_ACK      0x00200010
#define MBR_TX_EOT      0x00400010
#define MBR_TX_CRC_FLG  0x00000002

#define MBR_RX_REQ      0x00100020
#define MBR_RX_ACK      0x00200020
#define MBR_RX_NACK     0x00200120
#define MBR_RX_SPM      0x00300000
#define MBR_RX_EOT      0x00400020
#define MBR_RX_CRC      0x00400220
#define MBR_RX_CRC_FLG  0x00000001

#define MBR_STOP_H      0x00300000
#define MBR_STOP_L      0x53544F50

#define MBR_HALT_H      MBR_STOP_H
#define MBR_HALT_L      0x48414C54

#define MBR_ABORT       0x00500000
#define MBR_BL_GO       0x0050474F

#define MBR_TYP_MASK    0x00F00000
#define MBR_ERR_MASK    0x0000FF00

// state definitions
#define ST_IDLE         0
#define ST_WAIT_FOR_ACK 1
#define ST_WAIT_FOR_DMA 2
#define ST_DMA_FINISH   3
#define ST_GOT_EOT      4
#define ST_REQ_PENDING  5
#define ST_RESET        6
#define ST_TEMP         7
#define ST_DOWN         8
#define ST_PANIC        9
#define state_reset(x)  {if ((x) < ST_RESET) (x) = ST_IDLE; }

// default timeouts
#define TIMEOUT_MBR_TX      (HZ/10)         // 100 ms
#define TIMEOUT_TX_RX       (HZ*3)          // 3 sec.
#define MAX_TIMEOUT         ((MAX_SCHEDULE_TIMEOUT / HZ / 4) * 100)

#define CS2_DMA_SIZE        (2 * PAGE_SIZE)
#define CS2_DMA_BLOCK_SIZE  8               // DMA blocking

#define CS2_MAX_RETRIES     10              // max. retransmisions on CRC error


//--------------------------------------------------------------------------------
// Lock Queue
//--------------------------------------------------------------------------------
struct lock_queue_elem_t
{
  pid_t pid;
  struct lock_queue_elem_t *next;
};

#define QTAB_STEP 8

struct lock_queue_t
{
  struct lock_queue_elem_t  *first;
  struct lock_queue_elem_t  *last;
  struct lock_queue_elem_t  *free;
  struct lock_queue_elem_t  *mem;
  spinlock_t                lock;
  wait_queue_head_t         event;
};

//--------------------------------------------------------------------------------
// Device Structure
//--------------------------------------------------------------------------------
struct cs2_device_t
{
  struct cs_device_t csdev;

  // device specific part
  struct cs_iomem_t       bar0;

  volatile unsigned char __iomem *bar;

  long                    timeout;
  unsigned int            crc_flag;
  unsigned int            irqs;

  // tx
  struct cs_dmabuf_t      tx_buf;
  wait_queue_head_t       tx_queue;
  volatile int            tx_state;
  int                     tx_error;
  int                     tx_reset;
  unsigned int            tx_count;
  unsigned int            tx_crc_err;
  unsigned int            tx_crc_rt;

  // rx
  struct cs_dmabuf_t      rx_buf;
  wait_queue_head_t       rx_queue;
  volatile int            rx_state;
  int                     rx_error;
  int                     rx_reset;
  unsigned int            rx_count;
  unsigned int            rx_crc_err;
  unsigned int            rx_crc_rt;

  wait_queue_head_t       rst_queue;
  int                     spm_state;

  // mbr
  volatile unsigned int   request;
  volatile unsigned int   req_len;
  volatile unsigned int   mbr_h;
  volatile unsigned int   mbr_l;

  struct lock_queue_t     lock_queue;
  struct tasklet_struct   *task;

  // statistics
  struct
  {
    unsigned int irq_count[5];
  }
  stats;
};

/******************************************************************************
 *
 * Globals
 *
 *****************************************************************************/
static unsigned int CrcMode = 1;
// module_param(CrcMode, int, 0);
// MODULE_PARM_DESC(CrcMode, " 0:off, 1:on [default]");

static char *StateTxt[16] = { "idle",
                              "wait for ACK",
                              "wait for DMA",
                              "dma finished",
                              "EOT received",
                              "request pending",
                              "reset",
                              "temperature alarm",
                              "down",
                              "panic",
                              "10?", "11?", "12?", "13?", "14?", "15?"};

int cs2_probe(struct pci_dev *pci_dev, struct cs2_device_t **pp_dp);

/******************************************************************************
 *
 * cs2_calc_crc
 *
 ******************************************************************************/
static inline void cs2_calc_crc(unsigned char *data, unsigned int len, unsigned int *crc)
{
  unsigned int *x = (unsigned int *)data;
  unsigned int crc0 = crc[1];
  unsigned int crc1 = crc[0];

  len >>= 2;

  while (len != 0)
  {
    crc0 ^= (crc1 & *x);
    crc1 ^= *x++;
    len--;
  }

  *crc++ = crc1;
  *crc = crc0;
}

/******************************************************************************
 *
 * cs2_io_map
 *
 *****************************************************************************/
static int cs2_io_map(struct cs2_device_t *dp)
{
  struct pci_dev    *pci_dev = dp->csdev.pci_dev;
  unsigned long     addr;
  unsigned long     size;
  struct cs_iomem_t *bar;

  if (  (addr = pci_resource_start(pci_dev, 0)) == 0
     || (size = pci_resource_len  (pci_dev, 0)) == 0
     )
  {
    log_error("no resource BAR0\n");
    return -ENODEV;
  }

  log_trace("BAR0: phys_addr: %lx, size: 0x%lx\n", addr, size);

  bar = &dp->bar0;
  bar->phys_addr = addr;
  bar->size = size;

  if ((bar->base_ptr = ioremap(bar->phys_addr, bar->size)) == NULL)
  {
    log_error("can't remap io memory for BAR0\n");
    return -ENODEV;
  }

  dp->bar = bar->base_ptr;

  log_trace("bar0: phys.: %08X:%08X -> virt.: %p, size = 0x%08lX\n", bar->phys_addr_hi,
                                                                     bar->phys_addr_lo,
                                                                     bar->base_ptr,
                                                                     bar->size);
  return 0;
}

/******************************************************************************
 *
 * cs2_io_unmap
 *
 *****************************************************************************/
static int cs2_io_unmap(struct cs2_device_t *dp)
{
  struct cs_iomem_t *bar = &dp->bar0;

  iounmap(bar->base_ptr);
  bar->base_ptr = NULL;

  dp->bar = NULL;

  return 0;
}

/******************************************************************************
 *
 * cs2_dma_alloc
 *
 *****************************************************************************/
static int cs2_dma_alloc(struct cs2_device_t *dp, unsigned int size, struct cs_dmabuf_t *buf)
{
  struct pci_dev *pci_dev = dp->csdev.pci_dev;

  buf->base_ptr = dma_alloc_coherent(&pci_dev->dev, size, &buf->phys_addr, GFP_ATOMIC);

  if (buf->base_ptr == NULL)
  {
    log_error("dma_alloc_coherent failed\n");
    return -ENOMEM;
  }

  buf->size = size;

  log_debug("Virtual Address = %p, Physical Address = %08X:%08X, size = %d\n",
                    buf->base_ptr,
                    buf->phys_addr_hi,
                    buf->phys_addr_lo,
                    buf->size);
  return 0;
}

/******************************************************************************
 *
 * cs2_dma_free
 *
 *****************************************************************************/
static void cs2_dma_free(struct cs2_device_t *dp, struct cs_dmabuf_t *buf)
{
  struct pci_dev *pci_dev = dp->csdev.pci_dev;

  if (buf->base_ptr == NULL) return;

  dma_free_coherent(&pci_dev->dev, buf->size, buf->base_ptr, buf->phys_addr);

  buf->base_ptr = NULL;
}

/******************************************************************************
 *
 * cs2_dma_clear
 *
 *****************************************************************************/
static inline void cs2_dma_clear(struct cs_dmabuf_t *buf)
{
  memset(buf->base_ptr, 0, buf->size);
}

/******************************************************************************
 *
 * cs2_lock_init
 *
 *****************************************************************************/
static void cs2_lock_init(struct lock_queue_t *lock)
{
  lock->first = NULL;
  lock->last  = NULL;
  lock->free  = NULL;
  lock->mem   = NULL;
  spin_lock_init(&lock->lock);
  init_waitqueue_head(&lock->event);
}

/******************************************************************************
 *
 * cs2_lock_cleanup
 *
 *****************************************************************************/
static void cs2_lock_cleanup(struct lock_queue_t *queue)
{
  struct lock_queue_elem_t *elem;

  spin_lock_irq(&queue->lock);

  while ((elem = queue->first) != NULL)
  {
    queue->first = elem->next;
    elem->next = queue->free;
    queue->free = elem;
  }

  queue->last = NULL;

  spin_unlock_irq(&queue->lock);

  wake_up_interruptible(&queue->event);
}

/******************************************************************************
 *
 * cs2_lock_free
 *
 *****************************************************************************/
static void cs2_lock_free(struct lock_queue_t *queue)
{
  struct lock_queue_elem_t *elem;

  while ((elem = queue->mem) != NULL)
  {
    queue->mem = elem->next;
    kfree(elem);
  }
}

/******************************************************************************
 *
 * cs2_lock_wait
 *
 *****************************************************************************/
static int cs2_lock_wait(struct cs2_device_t *dp, struct lock_queue_t *queue)
{
  int i;
  pid_t pid = current->pid;
  struct lock_queue_elem_t *elem;

  spin_lock_irq(&queue->lock);

  for (elem = queue->first; elem != NULL; elem = elem->next)
  {
    if (elem->pid == pid) break;
  }

  spin_unlock_irq(&queue->lock);

  if (elem == NULL)
  {
    spin_lock_irq(&queue->lock);

    if (queue->free == NULL)
    {
      spin_unlock_irq(&queue->lock);

      // add new memory block to free list
      if ((elem = kmalloc(QTAB_STEP * sizeof(struct lock_queue_elem_t), GFP_KERNEL)) == NULL)
        return -ENOMEM;

      spin_lock_irq(&queue->lock);

      elem->next = queue->mem;
      queue->mem = elem++;

      for (i = QTAB_STEP-1; --i >= 0; )
      {
        elem->next = queue->free;
        queue->free = elem++;
      }
    }

    // get element from free list
    elem = queue->free;
    queue->free = elem->next;

    // queue in at the end
    elem->pid = pid;
    elem->next = NULL;

    if (queue->last == NULL)
      queue->first = elem;
    else
      queue->last->next = elem;

    queue->last = elem;

    spin_unlock_irq(&queue->lock);
  }

  if (queue->first == elem)
  {
    // no wait
    return 0;
  }
  else
  {
    int err = 0;
    wait_queue_entry_t waiter;

    init_waitqueue_entry(&waiter, current);
    add_wait_queue(&queue->event, &waiter);

    for (;;)
    {
      set_current_state(TASK_INTERRUPTIBLE);

      if (dp->tx_state >= ST_RESET)
      {
        log_info("reset while wait for lock'\n");

        switch (dp->tx_state)
        {
          case ST_TEMP:  err = -ECS2TEMPALARM; break;
          case ST_PANIC: err = -ECS2PANIC; break;
          case ST_DOWN:  err = -ECS2DOWN; break;
          default:       err = -ECS2RESET;
        }
        break;
      }

      if (queue->first == elem) break;

      if (signal_pending(current))
      {
        err = -ERESTARTSYS;
        break;
      }

      schedule();
    }

    set_current_state(TASK_RUNNING);
    remove_wait_queue(&queue->event, &waiter);

    return err;
  }
}

/******************************************************************************
 *
 * cs2_lock_unlock
 *
 *****************************************************************************/
static void cs2_lock_unlock(struct lock_queue_t *queue)
{
  pid_t pid = current->pid;
  struct lock_queue_elem_t *elem;
  struct lock_queue_elem_t *prev = NULL;

  spin_lock_irq(&queue->lock);

  // search current task in queue
  for (elem = queue->first; elem != NULL; elem = elem->next)
  {
    if (elem->pid == pid) break;

    prev = elem;
  }

  if (elem != NULL)
  {
    // remove element
    if (prev)
    {
      prev->next = elem->next;

      // last element
      if (prev->next == NULL)
        queue->last = prev;
    }
    else
    {
      // first element
      queue->first = elem->next;

      // queue is empty
      if (queue->first == NULL)
        queue->last = NULL;
    }

    // put element into free list
    elem->next = queue->free;
    queue->free = elem;
  }

  spin_unlock_irq(&queue->lock);

  wake_up_interruptible(&queue->event);
}

/******************************************************************************
 *
 * cs2_irq_enable
 *
 ******************************************************************************/
static void cs2_irq_enable(struct cs2_device_t *dp)
{
  writeb(BIT_IRQM_DMA_RX | BIT_IRQM_DMA_TX, dp->bar + REG_IRQM_DMA);
  writeb(BIT_IRQM_MBR_RX,                   dp->bar + REG_IRQM_MBR_RX);
}

/******************************************************************************
 *
 * cs2_irq_disable
 *
 ******************************************************************************/
static void cs2_irq_disable(struct cs2_device_t *dp)
{
  writeb(0, dp->bar + REG_IRQM_DMA);
  writeb(0, dp->bar + REG_IRQM_MBR_RX);
}

/******************************************************************************
 *
 * cs2_reset
 *
 ******************************************************************************/
static void cs2_reset(struct cs2_device_t *dp)
{
  dp->rx_reset = 1;
  dp->tx_reset = 1;

  wake_up_interruptible(&dp->rx_queue);
  wake_up_interruptible(&dp->tx_queue);

  writel(MBR_HALT_H, dp->bar + REG_MBR_TX_H);
  writel(MBR_HALT_L, dp->bar + REG_MBR_TX_L);
  schedule_timeout(HZ/50);

  writeb(BIT_SWRST, dp->bar + REG_SWRST);
  udelay(4);

  writeb(0, dp->bar + REG_SWRST);
  writeb(BIT_TRST, dp->bar + REG_TRST);
  udelay(10);

  readb(dp->bar + REG_TRST);
  udelay(10);

  writel(MBR_BL_GO, dp->bar + REG_MBR_TX_H);
  writel(MBR_BL_GO, dp->bar + REG_WRITE_ADDR1_H);

  cs2_irq_enable(dp);

  dp->tx_state = ST_RESET;
  dp->rx_state = ST_RESET;
}

/******************************************************************************
 *
 * cs2_mbr_write
 *
 ******************************************************************************/
static int cs2_mbr_write(struct cs2_device_t *dp, unsigned int hi, unsigned int lo)
{
  unsigned long start = jiffies;

  for (;;)
  {
    if ((readb(dp->bar + REG_MBR_TX_STAT) & BIT_MBR_TX_FULL) == 0)
    {
      writel(hi, dp->bar + REG_MBR_TX_H);
      writel(lo, dp->bar + REG_MBR_TX_L);
      return 0;
    }

    if (jiffies - start >= TIMEOUT_MBR_TX) break;

    schedule();
  }

  log_trace("write MBR timeout: %08x %08x\n", hi, lo);

  return 1;
}

/******************************************************************************
 *
 * cs2_abort_wr_dma
 *
 ******************************************************************************/
static void cs2_abort_wr_dma(struct cs2_device_t *dp)
{
  if (readb(dp->bar + REG_DMA_START) & BIT_DMA_TX_START)
    writeb(BIT_DMA_TX_CANCEL, dp->bar + REG_DMA_CANCEL);
}

/******************************************************************************
 *
 * cs2_abort_rd_dma
 *
 ******************************************************************************/
static void cs2_abort_rd_dma(struct cs2_device_t *dp)
{
  if (readb(dp->bar + REG_DMA_START) & BIT_DMA_RX_START)
    writeb(BIT_DMA_RX_CANCEL, dp->bar + REG_DMA_CANCEL);
}

/******************************************************************************
 *
 * cs2_sleep_on_event
 *
 ******************************************************************************/
static inline int cs2_sleep_on_event(
  wait_queue_head_t *wait_queue,
  volatile int      *state,
  int               event,
  long              timeout,
  int               *reset
)
{
  int err = 0;
  wait_queue_entry_t waiter;

  if (*state != event)
  {
    init_waitqueue_entry(&waiter, current);
    add_wait_queue(wait_queue, &waiter);

    for (;;)
    {
      set_current_state(TASK_INTERRUPTIBLE);

      if (*reset)
      {
        switch (*state)
        {
          case ST_TEMP:
            log_error("high temp. while 'sleep_on_event()'\n");
            err = -ECS2TEMPALARM;
            break;

          case ST_PANIC:
            log_error("panic event while 'sleep_on_event()'\n");
            err = -ECS2PANIC;
            break;

          default:
            log_error("reset while 'sleep_on_event()'\n");
            err = -ECS2RESET;
        }
        break;
      }

      if (*state == event) break;

      if (  (event == 0 || event >= ST_REQ_PENDING)
         && signal_pending(current)
         )
      {
        log_error("got signal\n");
        err = -ERESTARTSYS;
        break;
      }

      if ((timeout = schedule_timeout(timeout)) == 0)
      {
        err = -ECS2TIMEOUT;
        log_error("timeout while waiting on event: %d -> %d\n", *state, event);
        break;
      }
    }

    set_current_state(TASK_RUNNING);
    remove_wait_queue(wait_queue, &waiter);
  }

  return err;
}

/******************************************************************************
 *
 * cs2_legacy_isr
 *
 ******************************************************************************/
static irqreturn_t cs2_legacy_isr(int irq, void *data)
{
  irqreturn_t         ret = IRQ_NONE;
  u32                 regval;
  struct cs2_device_t  *dp = (struct cs2_device_t*)data;
  unsigned int        status;
  unsigned int        mbr_h;
  unsigned int        mbr_l;

  status = readl(dp->bar + REG_IRQS_DMA);

  if (status & BIT_IRQS_DMA_TX)
  {
    // DMA TX finish
    if (dp->tx_state != ST_WAIT_FOR_DMA)
    {
      log_error("unexpected TX DMA IRQ at state %d\n", dp->tx_state);
    }
    else
    {
      dp->tx_state = ST_DMA_FINISH;
    }

    writeb(BIT_IRQS_DMA_TX, dp->bar + REG_IRQS_DMA);

    if ((regval = readw(dp->bar + REG_DMA_ERR)) & BIT_DMA_ERR_TX)
    {
      log_error("DMA error on transmit: %x\n", regval);
      dp->tx_error = -(ECS2PCITX | ((regval >> 8) & 0x0f));
    }

    wake_up_interruptible(&dp->tx_queue);

    ret = IRQ_HANDLED;
  }

  if (status & BIT_IRQS_DMA_RX)
  {
    // DMA RX finish
    if (dp->rx_state != ST_WAIT_FOR_DMA)
    {
      log_error("unexpected RX DMA IRQ at state %d\n", dp->rx_state);
    }
    else
    {
      dp->rx_state = ST_DMA_FINISH;
    }

    writeb(BIT_IRQS_DMA_RX, dp->bar + REG_IRQS_DMA);

    if ((regval = readw(dp->bar + REG_DMA_ERR)) & BIT_DMA_ERR_RX)
    {
      log_error("DMA error on receive: %x\n", regval);
      dp->rx_error = -(ECS2PCIRX | ((regval >> 8) & 0x0f));
    }

    wake_up_interruptible(&dp->rx_queue);

    ret = IRQ_HANDLED;
  }

  if (readb(dp->bar + REG_IRQS_MBR_RX) & BIT_IRQS_MBR_RX)
  {
    // MBR RX full
    writeb(BIT_IRQS_MBR_RX, dp->bar + REG_IRQS_MBR_RX);

    ret = IRQ_HANDLED;

    if ((readb(dp->bar + REG_MBR_RX_STAT) & BIT_MBR_RX_FULL) == 0) goto isr_end;

    dp->mbr_h = mbr_h = readl(dp->bar + REG_MBR_RX_H);
    dp->mbr_l = mbr_l = readl(dp->bar + REG_MBR_RX_L);

    log_trace("MBR command received: %08x %08x\n", mbr_h, mbr_l);

    switch (mbr_h & MBR_TYP_MASK)
    {
      case (MBR_TX_ACK & MBR_TYP_MASK):
        if (dp->tx_state != ST_WAIT_FOR_ACK)
          break;

        if (mbr_h & MBR_ERR_MASK)
        {
          log_error("request rejected by CS2: %08x %08x\n", mbr_h, mbr_l);
          dp->tx_error = -ECS2NACK;
          dp->tx_state = ST_DMA_FINISH;
          wake_up_interruptible(&dp->tx_queue);
        }
        else
        {
          dp->tx_state = ST_WAIT_FOR_DMA;
          writeb(BIT_DMA_TX_START, dp->bar + REG_DMA_START);
        }
        goto isr_end;

      case (MBR_TX_EOT & MBR_TYP_MASK):
        if (dp->tx_state != ST_DMA_FINISH)
        {
          if (  dp->tx_state != ST_WAIT_FOR_DMA
             || (BIT_DMA_TX_START & readb(dp->bar + REG_DMA_START)) != 0
             )
            break;

          log_trace("reordering IRQs\n");
          writeb(BIT_IRQS_DMA_TX, dp->bar + REG_IRQS_DMA);
        }

        if (mbr_h & MBR_ERR_MASK)
        {
          log_error("tx error reported by CS2: %08x %08x\n", mbr_h, mbr_l);
          dp->tx_error = -ECS2TXCRC;
        }

        dp->tx_state = ST_GOT_EOT;
        wake_up_interruptible(&dp->tx_queue);

        goto isr_end;

      case (MBR_RX_REQ & MBR_TYP_MASK):
        if (dp->rx_state != ST_IDLE) break;

        dp->rx_state = ST_REQ_PENDING;
        dp->request = mbr_h;
        dp->req_len = (mbr_l & 0xffffff);
        wake_up_interruptible(&dp->rx_queue);
        goto isr_end;

      case 0:
        // workaround for boot loader 1.2.0.0 bug
        if (mbr_h != 0x207) break;
        // fallthrough

      case (MBR_RX_SPM & MBR_TYP_MASK):
        if (CS2_IS_PANIC(mbr_l))
        {
          dp->tx_state = ST_PANIC;
          dp->rx_state = ST_PANIC;
          dp->tx_reset = 1;
          dp->rx_reset = 1;
          wake_up_interruptible(&dp->rx_queue);
          wake_up_interruptible(&dp->tx_queue);
          log_error("panic message received, reason code %04x\n", mbr_l >> 16);
          goto isr_end;
        }

        if (CS2_BL_STATE(mbr_l) == CS2_ST_BL_TEMP)
        {
          dp->tx_state = ST_TEMP;
          dp->rx_state = ST_TEMP;
          dp->tx_reset = 1;
          dp->rx_reset = 1;
          wake_up_interruptible(&dp->rx_queue);
          wake_up_interruptible(&dp->tx_queue);
        }
        else
        {
          if (dp->tx_state >= ST_RESET)
          {
            dp->tx_state = ST_IDLE;
            dp->rx_state = ST_IDLE;
            dp->tx_reset = 0;
            dp->rx_reset = 0;
          }
        }

        dp->spm_state = 0;
        wake_up_interruptible(&dp->rst_queue);
        goto isr_end;

      case (MBR_ABORT & MBR_TYP_MASK):
        log_error("request aborted (mbr_l=%x)\n", mbr_l);
        log_error("state tx=%d, rx=%d\n", dp->tx_state, dp->rx_state);

        dp->rx_state = ST_IDLE;
        dp->tx_state = ST_IDLE;

        cs2_lock_unlock(&dp->lock_queue);
        goto isr_end;
    }

    log_trace("unexpected MBR command: %08x %08x\n", mbr_h, mbr_l);
    log_trace("state tx=%d, rx=%d\n", dp->tx_state, dp->rx_state);
  }

isr_end:
  return ret;
}

/******************************************************************************
 *
 * cs2_open
 *
 *****************************************************************************/
static int cs2_open(struct cs_session_t *session)
{
  return 0;
}

/******************************************************************************
 *
 * cs2_close
 *
 *****************************************************************************/
static int cs2_close(struct cs_session_t *session)
{
  struct cs2_device_t *dp = (struct cs2_device_t *)session->dp;
  cs2_lock_unlock(&dp->lock_queue);
  return 0;
}

/******************************************************************************
 *
 * cs2_info
 *
 *****************************************************************************/
static int cs2_info(struct cs2_device_t *dp, char *info, int max)
{
  char *p = info;
  int len = 0;
  int ct;

  if (dp == NULL) return -ENODEV;

  len += scnprintf(p+len, max-len, "slot  %-16.16s\n", dp->csdev.slot);

  if (readl(dp->bar + REG_READ_ADDR1_H) == 0x2086)
    len += scnprintf(p+len, max-len, "model 3\n");

  len += scnprintf(p+len, max-len, "tx    %s\n", StateTxt[dp->tx_state % 15]);
  len += scnprintf(p+len, max-len, "rx    %s\n", StateTxt[dp->rx_state % 15]);

  if (dp->rx_state == ST_TEMP)
    len += scnprintf(p+len, max-len, "temp  %d,%s C\n", dp->mbr_l>>20, (dp->mbr_l>>19)&1?"5":"0");

  len += scnprintf(p+len, max-len, "batt  ");

  if (  (dp->rx_state == ST_IDLE || dp->rx_state >= ST_RESET)
     && (dp->tx_state == ST_IDLE || dp->tx_state >= ST_RESET)
     )
  {
    u32 bstate = readb(dp->bar+REG_BATT_STATE) & (BIT_CAR_BATT | BIT_EXT_BATT);

    if (bstate == 0)
    {
      len += scnprintf(p+len, max-len, "ok\n");
    }
    else
    {
      if (bstate & BIT_CAR_BATT) len += scnprintf(p+len, max-len, "carrier battery low.\n");
      if (bstate & BIT_EXT_BATT) len += scnprintf(p+len, max-len, "external battery low.\n");
    }
  }
  else
  {
    len += scnprintf(p+len, max-len, "unknown\n");
  }

  len += scnprintf(p+len, max-len, "txrt  %8u  %8u\n", dp->tx_crc_rt, dp->tx_crc_err);
  len += scnprintf(p+len, max-len, "rxrt  %8u  %8u\n", dp->rx_crc_rt, dp->rx_crc_err);
  len += scnprintf(p+len, max-len, "count %8u  %8u\n", dp->tx_count, dp->rx_count);
  len += scnprintf(p+len, max-len, "mbr   %08x  %08x\n", dp->mbr_h, dp->mbr_l);

  for (ct = 0; ct < 256; ct += 8)
  {
    if (ct == 0x78)
    {
      len += scnprintf(p+len, max-len, "bar0  %08x  %04xxxxx  78\n", readl(dp->bar+ct+4), readw(dp->bar+ct+2));
      continue;
    }

    len += scnprintf(p+len, max-len, "bar0  %08x  %08x  %02x\n", readl(dp->bar+ct+4), readl(dp->bar+ct),ct);
  }

  p[len++] = 0;

  return len;
}

/******************************************************************************
 *
 * cs2_ioctl
 *
 *****************************************************************************/
static int cs2_ioctl(struct cs_session_t *session, unsigned int cmd, unsigned long arg)
{
  int err = 0;
  struct cs2_device_t *dp = (struct cs2_device_t *)session->dp;

  switch (cmd)
  {
    //-------------------------------------------------------------------------
    case CS2IOC_HRESET:
    //-------------------------------------------------------------------------
      cs2_reset(dp);
      cs2_lock_cleanup(&dp->lock_queue);
      break;

    //--------------------------------------------------------------------------------
    case CS2IOC_WAIT_RDY:
    //--------------------------------------------------------------------------------
    {
      struct cs2_waitrdy *p_wait = (struct cs2_waitrdy *)arg;
      long timeout;

      if (get_user(timeout, &p_wait->timeout)) return -EFAULT;

      if (timeout < 0) timeout = MAX_SCHEDULE_TIMEOUT;
      else             timeout = msecs_to_jiffies(timeout);

      if (dp->tx_state == ST_RESET)
      {
        long remain = wait_event_interruptible_timeout(dp->rst_queue, (dp->tx_state != ST_RESET), timeout);

        if (remain <= 0)
        {
          log_debug("wait_event_interruptible_timeout returned: %ld\n", remain);
          err = (remain < 0) ? remain : -ECS2TIMEOUT;
        }

        if (err == -ECS2TIMEOUT)
        {
          u32 mbr;

          err = -ECS2TIMEOUT;

          if ((mbr = readl(dp->bar + REG_MBR_RX_H)) < 16)
            err = -(ECS2NORST | (mbr & 0x0f));

          dp->tx_state = ST_IDLE;
          dp->rx_state = ST_IDLE;

          dp->tx_reset = 0;
          dp->rx_reset = 0;
        }
      }
      else
      {
        int dummy = 0;
        dp->spm_state = 1;
        err = cs2_sleep_on_event(&dp->rst_queue, &dp->spm_state, 0, timeout, &dummy);
      }

      if (err != 0) return err;

      if (put_user(dp->mbr_l, &p_wait->state) != 0) return -EFAULT;

      break;
    }

    //--------------------------------------------------------------------------------
    case CS2IOC_GETREQ:
    //--------------------------------------------------------------------------------
    {
      struct cs2_getreq *p_req = (struct cs2_getreq *)arg;
      long timeout;

      if (get_user(timeout, &p_req->timeout)) return -EFAULT;

      if (dp->rx_state == ST_IDLE)
      {
        if (timeout == 0) return -EAGAIN;

        if (timeout < 0) timeout = MAX_SCHEDULE_TIMEOUT;
        else             timeout = msecs_to_jiffies(timeout);

        if ((err = cs2_sleep_on_event(&dp->rx_queue, &dp->rx_state, ST_REQ_PENDING, timeout, &dp->rx_reset)) != 0)
          return err;
      }

      if (dp->rx_state != ST_REQ_PENDING) switch (dp->rx_state)
      {
        case ST_TEMP:  return -ECS2TEMPALARM;
        case ST_PANIC: return -ECS2PANIC;
        case ST_DOWN:  return -ECS2DOWN;
        default:       return -ECS2STATE;
      }

      if (  put_user(dp->req_len, &p_req->req_size) != 0
         || put_user(dp->request >> 24, &p_req->req_id) != 0
         )
        return -EFAULT;

      break;
    }

    //--------------------------------------------------------------------------------
    case CS2IOC_CANCEL:
    //--------------------------------------------------------------------------------
      if (dp->rx_state != ST_REQ_PENDING) return -ECS2NOREQ;

      if (cs2_mbr_write(dp, MBR_RX_NACK, dp->req_len)) return -ECS2TIMEOUT;

      dp->rx_state = ST_IDLE;
      return 0;

    //--------------------------------------------------------------------------------
    case CS2IOC_LOCK_WAIT:
    //--------------------------------------------------------------------------------
      return cs2_lock_wait(dp, &dp->lock_queue);

    //--------------------------------------------------------------------------------
    case CS2IOC_UNLOCK:
    //--------------------------------------------------------------------------------
      cs2_lock_unlock(&dp->lock_queue);
      return 0;

    //--------------------------------------------------------------------------------
    case CS2IOC_GETHWTYPE:
    //--------------------------------------------------------------------------------
    {
      int *p_int = (int *)arg;

      if (readl(dp->bar + REG_READ_ADDR1_H) == 0x2086)
      {
        if (put_user(MODEL_NQ3, p_int) != 0) return -EFAULT;
      }
      else
      {
        if (put_user(MODEL_CS2, p_int) != 0) return -EFAULT;
      }
      return 0;
    }

    //--------------------------------------------------------------------------------
    case CS2IOC_SETTMOUT:
    //--------------------------------------------------------------------------------
      if (arg > MAX_TIMEOUT) return -EINVAL;

      if (arg == 0) dp->timeout = MAX_SCHEDULE_TIMEOUT;
      else          dp->timeout = msecs_to_jiffies(arg);
      return 0;

    //--------------------------------------------------------------------------------
    case CS2IOC_GETTMOUT:
    //--------------------------------------------------------------------------------
    {
      long *p_long = (long *)arg;

      if (put_user(jiffies_to_msecs(dp->timeout), p_long)) return -EFAULT;
      return 0;
    }

    //--------------------------------------------------------------------------------
    case CS2IOC_GETINFO:
    //--------------------------------------------------------------------------------
    {
      struct cs2_info *p_info = (struct cs2_info *)arg;
      char            *buf = NULL;
      int             len;

      if ((buf = kmalloc(PAGE_SIZE, GFP_KERNEL)) == NULL) return -ENOMEM;

      if ((len = cs2_info(dp, buf, PAGE_SIZE)) < 0)
      {
        err = len;
      }
      else
      {
        if (  copy_to_user(p_info->buf, buf, len) != 0
           || put_user(len, &p_info->count) != 0
           )
          err = -EFAULT;
      }

      kfree(buf);
      break;
    }

    //-------------------------------------------------------------------------
    default:
    //-------------------------------------------------------------------------
    {
      log_error("unknown control code: 0x%08x", cmd);
      return -ENOTTY;
    }

  }

  return err;
}

/******************************************************************************
 *
 * cs2_read
 *
 ******************************************************************************/
static int cs2_read(struct cs_session_t *session, char __user *buff, size_t count)
{
  struct cs2_device_t *dp = (struct cs2_device_t *)session->dp;
  int           err;
  int           crc_flag;
  unsigned int  crc[2];
  unsigned int  *p_crc;
  unsigned long dlen;
  char          *buf;
  unsigned long merr = 0;
  int           retry_ct = 0;

retry:
  buf = buff;

  if (dp->rx_state == ST_IDLE)
  {
    if ((err = cs2_sleep_on_event(&dp->rx_queue, &dp->rx_state, ST_REQ_PENDING, TIMEOUT_TX_RX, &dp->rx_reset)) != 0)
      return err;
  }

  if (dp->rx_state != ST_REQ_PENDING)
  {
    if (dp->rx_state == ST_PANIC) return -ECS2PANIC;
    if (dp->rx_state == ST_TEMP) return -ECS2TEMPALARM;
    if (dp->rx_state == ST_DOWN) return -ECS2DOWN;
    return -ECS2STATE;
  }

  if ((dlen = dp->req_len) > count)
    return -ECS2BADLEN;

  crc_flag = dp->request & MBR_RX_CRC_FLG;

  if (cs2_mbr_write(dp, MBR_RX_ACK, dlen))
  {
    state_reset(dp->rx_state);
    return -ECS2TIMEOUT;
  }

  dp->rx_error = 0;
  crc[0] = crc[1] = 0;

  if (dlen <= CS2_DMA_SIZE)
  {
    writel(0,                       dp->bar+REG_WRITE_ADDR_H);
    writel(dp->rx_buf.phys_addr_lo, dp->bar+REG_WRITE_ADDR_L);

    writel(dlen, dp->bar+REG_WRITE_COUNT);

    dp->rx_state = ST_WAIT_FOR_DMA;
    writeb(BIT_DMA_RX_START, dp->bar+REG_DMA_START);

    err = cs2_sleep_on_event(&dp->rx_queue, &dp->rx_state, ST_DMA_FINISH, dp->timeout, &dp->rx_reset);

    state_reset(dp->rx_state);

    if (err != 0)
    {
      cs2_abort_rd_dma(dp);
      return err;
    }

    if (dp->rx_error != 0)
    {
      cs2_abort_rd_dma(dp);
      return dp->rx_error;
    }

    if (crc_flag && dlen > 8)
    {
      cs2_calc_crc(dp->rx_buf.base_ptr, dlen - 8, crc);

      p_crc = (unsigned int *)(dp->rx_buf.base_ptr + dlen - 8);

      if (p_crc[0] != crc[0] || p_crc[1] != crc[1])
      {
        log_error("CRC error on receive\n");

        if (retry_ct++ < CS2_MAX_RETRIES)
        {
          if (cs2_mbr_write(dp, MBR_RX_CRC, dlen) != 0)
            return -ECS2TIMEOUT;

          dp->rx_crc_rt++;
          goto retry;
        }

        dp->rx_crc_err++;
        cs2_mbr_write(dp, MBR_RX_EOT, dlen);
        return -ECS2RXCRC;
      }
    }

    if (cs2_mbr_write(dp, MBR_RX_EOT, dlen) != 0)
      return -ECS2TIMEOUT;

    if (copy_to_user(buf, dp->rx_buf.base_ptr, dlen) != 0)
      return -EFAULT;
  }
  else // dlen > CS2_DMA_SIZE
  {
    unsigned int  dma_addr = dp->rx_buf.phys_addr_lo;
    unsigned char *dma_buf = dp->rx_buf.base_ptr;
    unsigned char *last_buf;
    unsigned int  last_size;
    unsigned int  rest = dlen;
    unsigned int  size;
    int           buf_ct = 0;
    int           first_block = 1;

    if (!compat_access_ok(VERIFY_WRITE, buf, rest))
      return -EFAULT;

    do
    {
      if (rest <= CS2_DMA_SIZE/2) size = rest;
      else                        size = CS2_DMA_SIZE/2;

      last_size = size;
      rest -= size;

      writel(0,        dp->bar+REG_WRITE_ADDR_H);
      writel(dma_addr, dp->bar+REG_WRITE_ADDR_L);

      writel(size, dp->bar+REG_WRITE_COUNT);

      dp->rx_state = ST_WAIT_FOR_DMA;
      writeb(BIT_DMA_RX_START, dp->bar+REG_DMA_START);

      last_buf = dma_buf;

      dma_addr = dp->rx_buf.phys_addr_lo;
      dma_buf = dp->rx_buf.base_ptr;

      buf_ct ^= 1;
      if (buf_ct)
      {
        dma_buf += CS2_DMA_SIZE/2;
        dma_addr += CS2_DMA_SIZE/2;
      }

      if (!first_block)
      {
        if (crc_flag)
          cs2_calc_crc(dma_buf, CS2_DMA_SIZE/2, crc);

        merr += __copy_to_user(buf, dma_buf, CS2_DMA_SIZE/2);
        buf += CS2_DMA_SIZE/2;
      }
      first_block = 0;

      if ((err = cs2_sleep_on_event(&dp->rx_queue, &dp->rx_state, ST_DMA_FINISH, dp->timeout, &dp->rx_reset)) != 0)
      {
        state_reset(dp->rx_state);
        cs2_abort_rd_dma(dp);
        return err;
      }

      if (dp->rx_error != 0)
      {
        state_reset(dp->rx_state);
        cs2_abort_rd_dma(dp);
        return dp->rx_error;
      }
    }
    while (rest != 0);

    dp->rx_state = ST_IDLE;

    merr += __copy_to_user(buf, last_buf, last_size);

    if (crc_flag)
    {
      if ((last_size -= 8) != 0)
        cs2_calc_crc(last_buf, last_size, crc);

      p_crc = (unsigned int *)(last_buf + last_size);
      if (p_crc[0] != crc[0] || p_crc[1] != crc[1])
      {
        log_error(" CRC error on receive\n");

        if (retry_ct++ < CS2_MAX_RETRIES)
        {
          if (cs2_mbr_write(dp, MBR_RX_CRC, dlen) != 0)
            return -ECS2TIMEOUT;

          dp->rx_crc_rt++;
          goto retry;
        }

        dp->rx_crc_err++;
        cs2_mbr_write(dp, MBR_RX_EOT, dlen);
        return -ECS2RXCRC;
      }
    }

    if (cs2_mbr_write(dp, MBR_RX_EOT, dlen) != 0)
      return -ECS2TIMEOUT;

    if (merr)
      return -EFAULT;
  }

  dp->rx_count++;

  return dlen;
}

/******************************************************************************
 *
 * cs2_write
 *
 *****************************************************************************/
static int cs2_write(struct cs_session_t *session, const char __user *buff, size_t count)
{
  struct cs2_device_t *dp = (struct cs2_device_t *)session->dp;
  int           err;
  int           retry_ct = 0;
  unsigned int  fill;
  unsigned int  crc_len;
  unsigned int  rest;
  unsigned int  size;
  const    char *buf;
  unsigned int  crc[2];
  unsigned int  crc_extra_block;
  unsigned long cp_error = 0;

  if (dp->tx_state != ST_IDLE)
  {
    if (dp->rx_state == ST_PANIC) return -ECS2PANIC;
    if (dp->tx_state == ST_TEMP) return -ECS2TEMPALARM;
    if (dp->rx_state == ST_DOWN) return -ECS2DOWN;
    return -ECS2STATE;
  }

  fill = ((count ^ (CS2_DMA_BLOCK_SIZE - 1)) + 1) & (CS2_DMA_BLOCK_SIZE - 1);

retry:
  dp->tx_error = 0;
  crc[0] = crc[1] = 0;
  buf = buff;
  if ((dp->crc_flag & MBR_TX_CRC_FLG) == 0)
  {
    crc_len = 0;
    crc_extra_block = 0;
  }
  else
  {
    crc_len = sizeof(crc);
    crc_extra_block = 1;
  }

  if (count <= CS2_DMA_SIZE)
  {
    size = rest = count + fill;

    if (copy_from_user(dp->tx_buf.base_ptr, buf, count) != 0)
      return -EFAULT;

    if (fill)
      memset(dp->tx_buf.base_ptr + count, 0, fill);

    if (crc_len)
    {
      cs2_calc_crc(dp->tx_buf.base_ptr, size, crc);

      if (size < CS2_DMA_SIZE)
      {
        memcpy((unsigned char *)dp->tx_buf.base_ptr + size, crc, sizeof(crc));
        rest += sizeof(crc);
        crc_extra_block = 0;
      }
    }

    writel(0,                       dp->bar+REG_READ_ADDR_H);
    writel(dp->tx_buf.phys_addr_lo, dp->bar+REG_READ_ADDR_L);

    writel(rest, dp->bar+REG_READ_COUNT);

    dp->tx_state = ST_WAIT_FOR_ACK;

    if (cs2_mbr_write(dp, MBR_TX_REQ | dp->crc_flag,size + crc_len))
    {
      cs2_abort_wr_dma(dp);
      state_reset(dp->tx_state);
      return -ECS2TIMEOUT;
    }
  }
  else
  {
    unsigned char *dma_buf;
    int           buf_ct;

    size = count + fill;

    if (  compat_access_ok(VERIFY_READ, buf, count) == 0
       || __copy_from_user(dp->tx_buf.base_ptr, buf, CS2_DMA_SIZE/2) != 0
       )
      return -EFAULT;

    cs2_calc_crc(dp->tx_buf.base_ptr, CS2_DMA_SIZE/2, crc);

    writel(0,                       dp->bar+REG_READ_ADDR_H);
    writel(dp->tx_buf.phys_addr_lo, dp->bar+REG_READ_ADDR_L);

    writel(CS2_DMA_SIZE/2, dp->bar+REG_READ_COUNT);

    dp->tx_state = ST_WAIT_FOR_ACK;

    if (cs2_mbr_write(dp, MBR_TX_REQ | dp->crc_flag,size + crc_len))
    {
      state_reset(dp->tx_state);
      cs2_abort_wr_dma(dp);
      return -ECS2TIMEOUT;
    }

    rest = count - CS2_DMA_SIZE/2;
    buf_ct = 1;

    do
    {
      // dma_addr_t dma_addr;
      u32 dma_addr;

      buf += CS2_DMA_SIZE/2;

      dma_addr = dp->tx_buf.phys_addr_lo;
      dma_buf  = dp->tx_buf.base_ptr;

      if (buf_ct)
      {
        dma_buf  += CS2_DMA_SIZE/2;
        dma_addr += CS2_DMA_SIZE/2;
      }

      if (rest < CS2_DMA_SIZE/2)
      {
        cp_error |= __copy_from_user(dma_buf, buf, rest);

        if (fill != 0)
          memset(dma_buf + rest, 0, fill);

        size = rest + fill;
        rest = 0;

        if (crc_len)
        {
          cs2_calc_crc(dma_buf, size, crc);

          if (size < CS2_DMA_SIZE/2)
          {
            memcpy(dma_buf + size, crc, sizeof(crc));
            size += sizeof(crc);
            crc_extra_block = 0;
          }
        }
      }
      else
      {
        cp_error |= __copy_from_user(dma_buf, buf, CS2_DMA_SIZE/2);
        size = CS2_DMA_SIZE/2;
        rest -= CS2_DMA_SIZE/2;
        cs2_calc_crc(dma_buf, CS2_DMA_SIZE/2, crc);
      }

      if ((err = cs2_sleep_on_event(&dp->tx_queue, &dp->tx_state, ST_DMA_FINISH, dp->timeout, &dp->tx_reset)) != 0)
      {
        state_reset(dp->tx_state);
        cs2_abort_wr_dma(dp);
        return err;
      }

      if (dp->tx_error != 0)
      {
        log_error("tx_error = %d\n", dp->tx_error);
        state_reset(dp->tx_state);
        cs2_abort_wr_dma(dp);
        return dp->tx_error;
      }

      dp->tx_state = ST_WAIT_FOR_DMA;

      writel(0,        dp->bar+REG_READ_ADDR_H);
      writel(dma_addr, dp->bar+REG_READ_ADDR_L);

      writel(size, dp->bar+REG_READ_COUNT);

      writeb(BIT_DMA_TX_START, dp->bar+REG_DMA_START);

      buf_ct ^= 1;

    }
    while (rest > 0);
  }

  if (crc_extra_block)
  {
    if ((err = cs2_sleep_on_event(&dp->tx_queue, &dp->tx_state, ST_DMA_FINISH, dp->timeout, &dp->tx_reset)) != 0)
    {
      state_reset(dp->tx_state);
      cs2_abort_wr_dma(dp);
      return err;
    }

    if (dp->tx_error != 0)
    {
      log_error("tx_error = %d\n", dp->tx_error);
      state_reset(dp->tx_state);
      cs2_abort_wr_dma(dp);
      return dp->tx_error;
    }

    dp->tx_state = ST_WAIT_FOR_DMA;

    memcpy(dp->tx_buf.base_ptr, crc, sizeof(crc));

    writel(0,                       dp->bar+REG_READ_ADDR_H);
    writel(dp->tx_buf.phys_addr_lo, dp->bar+REG_READ_ADDR_L);

    writel(sizeof(crc), dp->bar+REG_READ_COUNT);

    writeb(BIT_DMA_TX_START, dp->bar+REG_DMA_START);
  }

  err = cs2_sleep_on_event(&dp->tx_queue, &dp->tx_state, ST_GOT_EOT, dp->timeout, &dp->tx_reset);

  dp->tx_state = ST_IDLE;

  if (err)
  {
    cs2_abort_wr_dma(dp);
    return err;
  }

  if (dp->tx_error)
  {
    if (dp->tx_error == -ECS2TXCRC)
    {
      log_error("CRC error on transmit\n");

      if (retry_ct++ < CS2_MAX_RETRIES)
      {
        dp->tx_crc_rt++;
        goto retry;
      }

      dp->tx_crc_err++;
    }

    return dp->tx_error;
  }

  if (cp_error)
    return -EFAULT;

  dp->tx_count++;

  return count;
}

/******************************************************************************
 *
 * cs2_proc_show
 *
 ******************************************************************************/
static int cs2_proc_show(struct seq_file *m, void *v)
{
  int                 err = 0;
  char                *buf = NULL;
  int                 len;
  struct cs2_device_t  *dp = (struct cs2_device_t*)m->private;

  if (dp == NULL) return -ENODEV;

  if ((buf = kmalloc(PAGE_SIZE, GFP_KERNEL)) == NULL) return -ENOMEM;

  if ((len = cs2_info(dp, buf, PAGE_SIZE)) < 0) CLEANUP(len);

  seq_printf(m, buf);

cleanup:
  kfree(buf);
  return err;
}

/******************************************************************************
 *
 * cs2_proc_write
 *
 ******************************************************************************/
static ssize_t cs2_proc_write(struct file *file, const char __user *buf, size_t count, loff_t *off)
{
  int  err = 0;
  struct cs2_device_t *dp = compat_file_to_PDE_DATA(file);
  char *p = NULL;

  if ((p = kmalloc(count+1, GFP_KERNEL)) == NULL) return -ENOMEM;

  if (copy_from_user(p, buf, count) != 0) CLEANUP(-EFAULT);
  p[count] = 0;

  if (strncasecmp(p, "RESET", 5) == 0)
  {
    // TODO
  }
  else
  {
    log_error("invalid argument: %s\n", p);
    CLEANUP(-EINVAL);
  }

  if (err < 0)
  {
    log_error("'%s' returned: %d\n", p, err);
    goto cleanup;
  }

  (void)dp;

cleanup:
  if (p != NULL) kfree(p);

  return (err != 0) ? err : count;
}

/******************************************************************************
 *
 * cs2_proc_open
 *
 ******************************************************************************/
static int cs2_proc_open(struct inode *inode, struct file *file)
{
  return single_open(file, cs2_proc_show, compat_PDE_DATA(inode));
}

//-----------------------------------------------------------------------------
static const proc_op_t cs2_proc_ops =
//-----------------------------------------------------------------------------
{
#ifdef HAVE_PROC_OPS
  .proc_open    = cs2_proc_open,
  .proc_read    = seq_read,
  .proc_write   = cs2_proc_write,
  .proc_lseek   = seq_lseek,
  .proc_release = single_release,
#else
  .owner   = THIS_MODULE,
  .open    = cs2_proc_open,
  .read    = seq_read,
  .write   = cs2_proc_write,
  .llseek  = seq_lseek,
  .release = single_release,
#endif
};

/******************************************************************************
 *
 * cs2_remove
 *
 *****************************************************************************/
static int cs2_remove(struct cs2_device_t *dp)
{
  struct pci_dev *pci_dev;

  if (dp == NULL) return -EINVAL;

  pci_dev = dp->csdev.pci_dev;

  // disable interrupts
  if (dp->bar != NULL) cs2_irq_disable(dp);

  // disconnect interrupt
  if (pci_dev != NULL)
  {
    free_irq(pci_dev->irq, dp);
    pci_disable_device(pci_dev);
  }

  cs2_dma_free(dp, &dp->rx_buf);
  cs2_dma_free(dp, &dp->tx_buf);

  cs2_io_unmap(dp);

  if (pci_dev != NULL) pci_release_regions(pci_dev);

  cs2_lock_free(&dp->lock_queue);

  memset(dp, 0, sizeof(struct cs2_device_t));
  kfree(dp);

  return 0;
}

/******************************************************************************
 *
 * cs2_probe
 *
 *****************************************************************************/
int cs2_probe(struct pci_dev *pci_dev, struct cs2_device_t **pp_dp)
{
  int err;
  struct cs2_device_t *dp;
  struct cs_device_t *csdev;

  // create device
  if ((dp = kmalloc(sizeof(struct cs2_device_t), GFP_KERNEL)) == NULL) return -ENOMEM;
  memset(dp, 0, sizeof(struct cs2_device_t));
  csdev = &dp->csdev;
  csdev->minor = -1;

  csdev->pci_dev = pci_dev;
  csdev->proc_ops = &cs2_proc_ops;

  csdev->p_remove = (int (*)(struct cs_device_t*))cs2_remove;

  csdev->p_open   = cs2_open;
  csdev->p_close  = cs2_close;
  csdev->p_ioctl  = cs2_ioctl;
  csdev->p_read   = cs2_read;
  csdev->p_write  = cs2_write;

  // map memory mapped register
  if ((err = pci_request_regions(pci_dev, "cs2")) != 0)
  {
    log_error("pci_request_regions returned: %d\n", err);
    CLEANUP(-ENODEV);
  }

  if ((err = cs2_io_map(dp)) != 0)
  {
    log_error("cs2_io_map returned: %d", err);
    goto cleanup;
  }

  // enable device
  if ((err = pci_enable_device(pci_dev)) != 0)
  {
    log_error("pci_enable_device returned: %d\n", err);
    goto cleanup;
  }

  // set master
  pci_set_master(pci_dev);

  // set DMA mask
  if ((err = dma_set_coherent_mask(&pci_dev->dev, DMA_BIT_MASK(32))) != 0)
  {
    log_error("dma_set_coherent_mask(32) returned: %d\n", err);
    goto cleanup;
  }

  // allocate DMA memory
  if ((err = cs2_dma_alloc(dp, CS2_DMA_SIZE, &dp->rx_buf)) != 0)
  {
    log_error("cs2_dma_alloc(rx_buf) returned: %d\n", err);
    goto cleanup;
  }

  if ((err = cs2_dma_alloc(dp, CS2_DMA_SIZE, &dp->tx_buf)) != 0)
  {
    log_error("cs2_dma_alloc(tx_buf) returned: %d\n", err);
    goto cleanup;
  }

  init_waitqueue_head(&dp->tx_queue);
  init_waitqueue_head(&dp->rx_queue);
  init_waitqueue_head(&dp->rst_queue);

  // connect interrupt
  if ((err = request_irq(pci_dev->irq, cs2_legacy_isr, IRQF_SHARED, "cs2", dp)) != 0)
  {
    log_error("request_irq returned: %d\n", err);
    goto cleanup;
  }

  log_trace("successfully initialized irq: %d\n", pci_dev->irq);

  // initialize device
  dp->tx_state = ST_IDLE;
  dp->rx_state = ST_IDLE;

  dp->timeout = TIMEOUT_TX_RX;

  cs2_lock_init(&dp->lock_queue);

  dp->tx_count = 0;
  dp->tx_crc_err = 0;
  dp->tx_crc_rt = 0;

  dp->rx_count = 0;
  dp->rx_crc_err = 0;
  dp->rx_crc_rt = 0;

  dp->spm_state = 0;
  dp->crc_flag = CrcMode ? MBR_TX_CRC_FLG | MBR_RX_CRC_FLG : 0;

  // enable interrupts
  cs2_irq_enable(dp);

  // kick start boot loader
  writel(MBR_BL_GO, dp->bar+REG_MBR_TX_H);
  writel(MBR_BL_GO, dp->bar+REG_WRITE_ADDR1_H);

  *pp_dp = dp;
  return 0;

cleanup:
  cs2_remove(dp);
  *pp_dp = NULL;
  return err;
}
