/* SPDX-License-Identifier: GPL-2.0-only
 *
 * Copyright 2024 Utimaco IS GmbH
 * All Rights Reserved.
 *
 */
#ifndef __CRYPTOSERVER_H__
#define __CRYPTOSERVER_H__

#include "compat.h"
#include "log.h"

/******************************************************************************
 *
 * Definitions
 *
 ******************************************************************************/
#define IRQ_MODE_LEGACY       0
#define IRQ_MODE_MSI          1
#define IRQ_MODE_MSI_MULTI    2
#define IRQ_MODE_MSIX         3

#if defined(__BYTE_ORDER__) && __BYTE_ORDER__ == __ORDER_BIG_ENDIAN__
  #define CS_BIG_ENDIAN
#else
  #define CS_LITTLE_ENDIAN
#endif

// #define USE_PCI_ERROR_REPORTING

/******************************************************************************
 *
 * Types
 *
 ******************************************************************************/

// IO memory
struct cs_iomem_t
{
  unsigned char __iomem  *base_ptr;
  union
  {
    struct
    {
#ifdef CS_BIG_ENDIAN
      u32 phys_addr_hi;
      u32 phys_addr_lo;
#else
      u32 phys_addr_lo;
      u32 phys_addr_hi;
#endif
    };

    phys_addr_t phys_addr;
  };

  unsigned long size;
};

// DMA buffer
struct cs_dmabuf_t
{
  void  *base_ptr;
  union
  {
    struct
    {
#ifdef CS_BIG_ENDIAN
      u32 phys_addr_hi;
      u32 phys_addr_lo;
#else
      u32 phys_addr_lo;
      u32 phys_addr_hi;
#endif
    };

    phys_addr_t phys_addr;
  };

  unsigned int  size;
};

struct cs_device_t;

// Session
struct cs_session_t
{
  struct cs_device_t  *dp;
  struct file         *file;
  u32                 messages[4];  // up to 128 messages
  spinlock_t          lock;
  unsigned int        has_sema;

  unsigned int        id;
  unsigned int        gen;
};

// Device
struct cs_device_t
{
  struct cs_device_t  *next;
  struct cs_device_t  *parent;

  short               minor;
  unsigned short      model;
  unsigned short      cardno;
  unsigned short      subfn;

  char                device_name[16];
  char                proc_name[24];
  char                slot[16];
  spinlock_t          lock;
  unsigned int        use_cnt;

  struct device           *device;
  struct device_attribute dev_attr_channel;
  struct device_attribute dev_attr_parent;

  struct pci_dev          *pci_dev;
  struct proc_dir_entry   *proc_entry;
  const proc_op_t         *proc_ops;

  int  (*p_remove)       (struct cs_device_t *dp);

  int  (*p_shutdown)     (struct cs_device_t *dp, int state);

  int  (*p_suspend)      (struct cs_device_t *dp);
  int  (*p_resume)       (struct cs_device_t *dp);

  void (*p_reset_prepare)(struct cs_device_t *dp);
  void (*p_reset_done)   (struct cs_device_t *dp);

  const char*(*p_state)  (struct cs_device_t *dp);

  int (*p_open)     (struct cs_session_t *session);
  int (*p_close)    (struct cs_session_t *session);
  int (*p_ioctl)    (struct cs_session_t *session, unsigned int cmd, unsigned long arg);
  int (*p_read)     (struct cs_session_t *session, char __user *buf, size_t max_len);
  int (*p_write)    (struct cs_session_t *session, const char __user *buf, size_t max_len);

  unsigned int (*p_poll)(struct cs_session_t *session, struct file *filp, struct poll_table_struct *wait);
};

/******************************************************************************
 *
 * Macros
 *
 *****************************************************************************/
#define CLEANUP(code)      { err = (code); goto cleanup; }

#define DIM(a)             ( sizeof((a)) / sizeof((a)[0]) )

#define UNUSED(x)           (void)((x))

#define MIN(a,b)            (a)<(b)?(a):(b)
#define MAX(a,b)            (a)>(b)?(a):(b)

#define WLEN(a)             (((a) + 3) / 4)

#define STR_HELPER(x)       #x
#define STR(x)              STR_HELPER(x)


//-----------------------------------------------------------------------------
// backlog
//-----------------------------------------------------------------------------
#define BACKLOG_ENTRY_SIZE  64

struct cs_backlog_entry_t
{
  const char *where;
  int err;
  char msg[BACKLOG_ENTRY_SIZE];
};

#define BACKLOG_SIZE    1024

struct cs_backlog_t
{
  struct cs_backlog_entry_t entries[BACKLOG_SIZE];
  unsigned int first;
  unsigned int next;
};

#ifdef BACKLOG
#define log_back(format,...)      cs_backlog_entry(&Backlog, (__func__), (err), format, ##__VA_ARGS__)
#define log_back_print()          cs_backlog_print(&Backlog)
#else
#define log_back(format,...)
#define log_back_print()
#endif

/******************************************************************************
 *
 * Globals
 *
 *****************************************************************************/
extern unsigned int MsiMode;
extern unsigned int ErrorReporting;

extern const char   DriverVersionString[];
extern const char   *LogLevelTxt[];
extern const char   *IrqModeTxt[];
extern const char   *ModelTxt[];

#ifdef BACKLOG
extern struct cs_backlog_t Backlog;
#endif

// util.c
extern void cs_backlog_entry(struct cs_backlog_t *bl, const char *where, int err, char *msg, ...);
extern void cs_backlog_print(struct cs_backlog_t *bl);

//extern void cs_xprint(const char *txt, int ofs, const void *data, int len);
extern int  cs_get_args(char *p_arg, const char *fmt, ...);

extern char *cs_time_to_ascii(const u32 epoch) __attribute__((weak));

// crc.c
extern void         crc16_init(void);
extern unsigned int crc16_calc(unsigned int crc, unsigned char *data, unsigned int len);



#endif
