// SPDX-License-Identifier: GPL-2.0-only
/*
 * Copyright 2024 Utimaco IS GmbH
 * All Rights Reserved.
 *
 */
#include <linux/string.h>

#include "cryptoserver.h"

/******************************************************************************
 *
 * Globals
 *
 *****************************************************************************/
static unsigned short Crc_tab1[256];
static unsigned short Crc_tab2[256];
static unsigned short Crc_tab3[256];
static unsigned short Crc_tab4[256];

/******************************************************************************
 *
 * crc16_calc_slow
 *
 ******************************************************************************/
static unsigned int crc16_calc_slow(unsigned int crc, unsigned char *data, unsigned int len)
{
  unsigned int dd;

  while (len-- != 0)
  {
    dd = (crc >> 8) & 0xff;
    crc = (crc << 8) | *data++;
    dd ^= dd >> 4;
    crc ^= (dd << 12) ^ (dd << 5) ^ dd;
  }

  return crc;
}

/******************************************************************************
 *
 * crc16_calc
 *
 ******************************************************************************/
unsigned int crc16_calc(unsigned int crc, unsigned char *data, unsigned int len)
{
  while (len >= 4)
  {
    len -= 4;
    crc = Crc_tab4[crc >> 8]
        ^ Crc_tab3[crc & 0xff]
        ^ Crc_tab2[data[0]]
        ^ Crc_tab1[data[1]]
        ^ ((data[2] << 8) | data[3]);
    data += 4;
  }

  if (len)
    crc = crc16_calc_slow(crc, data, len);

  return crc & 0xffff;
}

/******************************************************************************
 *
 * crc16_init
 *
 *****************************************************************************/
void crc16_init(void)
{
  unsigned char buf[6];
  int i;

  memset(buf, 0, 6);

  for (i = 0; i < 256; i++)
  {
    buf[0] = i;
    Crc_tab1[i] = crc16_calc_slow(0, buf, 3);
    Crc_tab2[i] = crc16_calc_slow(0, buf, 4);
    Crc_tab3[i] = crc16_calc_slow(0, buf, 5);
    Crc_tab4[i] = crc16_calc_slow(0, buf, 6);
  }
}
