/* SPDX-License-Identifier: GPL-2.0-only
 *
 * Copyright 2024 Utimaco IS GmbH
 * All Rights Reserved.
 *
 */
#ifndef __COMPAT_H__
#define __COMPAT_H__

#include <linux/version.h>
#include <linux/fs.h>
#include <linux/msi.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/compiler.h>


/******************************************************************************
 *
 * Redhat
 *
 *****************************************************************************/
#ifndef RHEL_RELEASE_VERSION
#define RHEL_RELEASE_VERSION(a, b) (((a) << 8) + (b))
#endif

#ifndef RHEL_RELEASE_CODE
#define RHEL_RELEASE_CODE   0
#endif

#ifndef RHEL_MAJOR
#define RHEL_MAJOR  0
#endif

/******************************************************************************
 *
 * Suse
 *
 *****************************************************************************/
#ifndef SLE_VERSION
#define SLE_VERSION(a,b,c) KERNEL_VERSION(a,b,c)
#endif

#ifdef CONFIG_SUSE_KERNEL
  #if ( LINUX_VERSION_CODE >= KERNEL_VERSION(4,12,14) )
    // Suse 15
    #define SLE_VERSION_CODE SLE_VERSION(15,0,0)
  #elif ( LINUX_VERSION_CODE >= KERNEL_VERSION(4,4,57) )
    // Suse 12SP3
    #define SLE_VERSION_CODE SLE_VERSION(12,3,0)
  #elif ( LINUX_VERSION_CODE >= KERNEL_VERSION(3,12,28) )
    // Suse 12
    #define SLE_VERSION_CODE SLE_VERSION(12,0,0)
  #endif
#endif

#ifndef SLE_VERSION_CODE
#define SLE_VERSION_CODE 0
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 6, 0)
#define HAVE_PROC_OPS
typedef struct proc_ops proc_op_t;
#else
typedef struct file_operations proc_op_t;
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 0, 0) || RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION(8,1)
# define compat_access_ok(type,addr,size) access_ok(addr,size)
#else
# define compat_access_ok(type,addr,size) access_ok(type,addr,size)
#endif


#if LINUX_VERSION_CODE < KERNEL_VERSION(6, 6, 0)
  #define USE_PCI_ERROR_REPORTING
#endif


/******************************************************************************
 *
 * compat_PDE_DATA
 *
 *****************************************************************************/
static inline void *compat_PDE_DATA(struct inode *inode)
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 10, 0)
  return PDE(inode)->data;
#else
  #if (  LINUX_VERSION_CODE >= KERNEL_VERSION(5, 17, 0) \
      || (  RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION(9,1) \
         && LINUX_VERSION_CODE >= KERNEL_VERSION(5, 14, 0) \
         ) \
      )
    return pde_data(inode);
  #else
    return PDE_DATA(inode);
  #endif
#endif
}

/******************************************************************************
 *
 * compat_file_to_PDE_DATA
 *
 *****************************************************************************/
static inline void *compat_file_to_PDE_DATA(struct file *file)
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 10, 0)
  struct inode *inode = ((struct seq_file *)file->private_data)->private;
#else
  struct inode *inode = file_inode(file);
#endif

  return compat_PDE_DATA(inode);
}

/******************************************************************************
 *
 * wait_queue_t has been renamed to wait_queue_entry_t
 *
 *****************************************************************************/
#if (  LINUX_VERSION_CODE < KERNEL_VERSION(4,13,0) \
    && (!(SLE_VERSION_CODE >= SLE_VERSION(15,0,0))) \
    )
  #define wait_queue_entry_t wait_queue_t
#endif

/******************************************************************************
 *
 * pci_enable_msi_range
 *
 *****************************************************************************/
#if (   LINUX_VERSION_CODE < KERNEL_VERSION(3, 14, 0) \
    && (!((RHEL_MAJOR == 6) && (RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION(6,8)))) \
    && (!((RHEL_MAJOR == 7) && (RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION(7,1) ))) \
    && (!(SLE_VERSION_CODE >= KERNEL_VERSION(12,0,0))) \
    )
static inline int pci_enable_msi_range(struct pci_dev *dev, int minvec, int maxvec)
{
	int nvec = maxvec;
	int rc;

	if (maxvec < minvec) return -ERANGE;

	do
  {
		rc = pci_enable_msi_block(dev, nvec);

		if (rc < 0)
    {
			return rc;
		}
    else if (rc > 0)
    {
			if (rc < minvec) return -ENOSPC;
			nvec = rc;
		}
	}
  while (rc);

	return nvec;
}
#endif

/******************************************************************************
 *
 * pci_enable_msix_range
 *
 *****************************************************************************/
#if (  LINUX_VERSION_CODE < KERNEL_VERSION(3, 13, 0) \
    && (!((RHEL_MAJOR == 6) && (RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION(6,8)))) \
    && (!((RHEL_MAJOR == 7) && (RHEL_RELEASE_CODE >= RHEL_RELEASE_VERSION(7,1) ))) \
    && (!(SLE_VERSION_CODE >= KERNEL_VERSION(12,0,0))) \
    )
static inline int pci_enable_msix_range(struct pci_dev *dev, struct msix_entry *entries, int minvec, int maxvec)
{
  int nvec = maxvec;
  int rc;

  if (maxvec < minvec) return -ERANGE;

  do
  {
    rc = pci_enable_msix(dev, entries, nvec);

    if (rc < 0)
    {
      return rc;
    }
    else if (rc > 0)
    {
      if (rc < minvec) return -ENOSPC;
      nvec = rc;
    }
  }
  while (rc);

  return nvec;
}
#endif

#if (LINUX_VERSION_CODE < KERNEL_VERSION(3, 19, 0))
  #define pci_load_saved_state(a,b)
#endif

#ifndef WRITE_ONCE
  #define WRITE_ONCE(x, val) ({ *(volatile typeof(x) *)&x = val; x; })
#endif

#ifdef CONFIG_X86_32
static inline __u64 readq(const volatile void __iomem *addr)
{
  const volatile u32 __iomem *p = addr;
  u32 low, high;

  low = readl(p);
  high = readl(p + 1);

  return low + ((u64)high << 32);
}

static inline void writeq(__u64 val, volatile void __iomem *addr)
{
  writel(val, addr);
  writel(val >> 32, addr+4);
}
#endif


#endif
