package com.utimaco.cs2.mdl;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.HashMap;

public abstract class SdkSequence {
	public static final byte PATTERN_OO = 2;
	public static final byte PATTERN_2S = 4;
	public static final byte PATTERN_SI = 6;
	public static final byte PATTERN_2I = 8;
	
    protected int maxSize = 0;
    protected static final int UNCOUNTED_SIZE = -1;
    private ArrayList<SdkBuffer> seq = new ArrayList<SdkBuffer>();
    private HashMap<String, SdkBuffer> map = new HashMap<String, SdkBuffer>();
    private ByteBuffer deserbuff = null;
    private ByteBuffer deserdata = null;
    private byte       deserpatt = PATTERN_OO;
    private int        desersqsz = 0; // Seq Size
    private int        desersqrl = 0; // (max) Record Length
    
    public SdkSequence () {
    	maxSize = UNCOUNTED_SIZE;
    }
    public SdkSequence (int maxSeqSize) {
    	this();
        maxSize = maxSeqSize;
    }
    
    public boolean containsKey(String n) {
    	return map.containsKey(n);
    }
    
    public SdkBuffer get(int n) {
    	return seq.get(n);
    }
    
    public SdkBuffer get(String n) {
    	return map.get(n);
    }
    
    public void setMaxSize() {
    	maxSize = seq.size();
    }
    public boolean add(String n, SdkBuffer obj) {
    	if (map.containsKey(n)) { return false; }
 
    	int chk = seq.size();
    	if ((maxSize != UNCOUNTED_SIZE) && (chk >= maxSize)) { return false; }
 
    	seq.add(obj);
    	map.put(n, obj);
    	
    	return true;
    }
    
    /**
     * Replaces the named object with a new object.  If unable, old object is retained.
     * 
     * @param n   - object name of existing object
     * @param obj - object to replace existing with
     * @return    - previously existing object if one, or new obj if not, or null if unable to add new obj.
     */
    public SdkBuffer replace(String n, SdkBuffer obj) {
    	SdkBuffer f = remove(n);
    	if (f == null) { // empty slot
    		return (add(n, obj) ? obj : null); // add succeeds, return obj, else null
    	} 
    	if (add(n, obj)) { return f; } // add succeeds, return f
    	
    	// Failed to add.  undo the remove.
    	add(n, f);
    	return null;
    }
    
    /**
     * Removes a named object
     * 
     * @param n - name of the object
     * @return - the object removed, null if none
     */
    public SdkBuffer remove(String n) {
    	if (map.containsKey(n)) {
    		SdkBuffer f = map.remove(n); 
    		seq.remove(f);
    		return f;
    	}
    	return null;
    }
    public void clear () {
    	map.clear();
    	seq.clear();
    }
    public int count() {
    	return seq.size();
    }
    
    protected boolean seqSet (byte [] bfr, int magic) {
    	deserbuff = ByteBuffer.wrap(bfr);
    	if (deserbuff == null) { return false; }
    	
    	// Magic:  4 byte Integer
    	if (deserbuff.getInt() != magic) { return false; }

    	// Pattern:  1 byte switchable
    	// Pattern contains information about how the number of records (sqsz - size) and
    	// how large each record is (sqrl - rec len) are *stored* (as bytes, shorts or ints)
    	// As a side effect, the value for each pattern equals the size of the storage, so
    	// PATTERN_OO (octet+octet) is set to (1+1=2) and 2S (short+short) is (2+2=4)
    	deserpatt = deserbuff.get();
    	switch (deserpatt) {
    	case PATTERN_OO:
    		desersqsz = (int)deserbuff.get();
    		desersqrl = (int)deserbuff.get();
    		break;
    	case PATTERN_2S:
    		desersqsz = (int)deserbuff.getShort();
    		desersqrl = (int)deserbuff.getShort();
    		break;
    	case PATTERN_2I:
    		desersqsz = (int)deserbuff.getInt();
    		desersqrl = (int)deserbuff.getInt();
    		break;
    	case PATTERN_SI:
    		desersqsz = (int)deserbuff.getShort();
    		desersqrl = (int)deserbuff.get();
    		break;
    	default:
    		return false;
    	}
    	deserdata = deserbuff.slice(); // deserdata points at .seqRec(0)
    	return true;
    }
    protected int seqSize () { return desersqsz; }
    protected int seqRecLen () { return desersqrl; }
    
    protected ByteBuffer seqPrep (int mgc) {
    	int len = 5; /* magic + pattern byte */;
    	int sz = seq.size();
    	int maxLen = 0;
    	for (int i = 0; i < sz; i++) {
    		if (maxLen < seq.get(i).length()) { maxLen = seq.get(i).length() + 40 /* uuid */; }
    	}
    	byte pttn = 0;
    	if ((sz < 256) && (maxLen < 256)) { pttn = PATTERN_OO; }
    	else if ((sz < 65536) && (maxLen < 65536)) { pttn = PATTERN_2S; }
    	else if (sz < 65536) { pttn = PATTERN_SI; }
    	else { pttn = PATTERN_2I; }

    	len += pttn;
    	len += (sz * maxLen);
    	
    	ByteBuffer bfr = ByteBuffer.allocate(len);
    	bfr.putInt(mgc);
    	bfr.put(pttn);
    	switch (pttn) {
    	case PATTERN_OO: bfr.put((byte)sz); bfr.put((byte)maxLen); break; 
		case PATTERN_2S: bfr.putShort((short)sz); bfr.putShort((short)maxLen); break; 
		case PATTERN_SI: bfr.putShort((short)sz); bfr.putInt(maxLen); break; 
		case PATTERN_2I: bfr.putInt(sz); bfr.putInt(maxLen); break; 
    	}
    	return bfr;
    }
    protected ByteBuffer seqRec (int n) {
    	if (deserdata == null) { return null; }
    	ByteBuffer tbf = deserdata.duplicate();
    	int i = (n * desersqrl);
    	if (i > tbf.capacity()) { return null; }
    	byte [] f = new byte[desersqrl];
    	tbf.position(i);
    	tbf.get(f, 0, desersqrl);
    	tbf = ByteBuffer.wrap(f);
    	int len = 0;
    	switch (deserpatt & 0x0F) {
    	case 0x1: len = (int)tbf.get(); break;
    	case 0x2: len = (int)tbf.getShort(); break;
    	case 0x4: len = (int)tbf.getInt(); break;
    	default: return null;
    	}
    	tbf = tbf.slice();
    	if (len >= tbf.limit()) { return null; }
    	tbf.limit(len);
    	return tbf;
    }
}
