package com.utimaco;

/* 
 * Advanced ML options
 * - external mu
 * - import/export keygen seed (keys created with ML module version >= 1.1.3)
 *
 * Arguments example
 * dev 3001@192.168.126.208 devbuild -p USR_0000,12345678 -group SLOT_0000
 *
 * Copyright included by reference, please see Utimaco_Demo_License.txt
 */

import java.io.File;
// import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Random;

import CryptoServerAPI.CryptoServerException;
import CryptoServerCXI.CryptoServerCXI;

import com.utimaco.aut.AutTest;
//import com.utimaco.aut.DilithiumTest;
import com.utimaco.aut.KeystoreTest;
//import com.utimaco.aut.KyberTest;
import com.utimaco.aut.MLDSATest;
import com.utimaco.aut.MLKEMTest;
import com.utimaco.aut.XmssTest;
import com.utimaco.cs2.mdl.*;
import com.utimaco.cs2.mdl.any.CxiKeyAttributes;
import com.utimaco.cs2.mdl.pqmi.*;

public class AdvancedOptions {
	static final Random r = new Random();
	static int module_id = 0xa6; //

	// if adding an enum to AUT ...
	static enum AUT { DILITHIUM, MLDSA, KYBER, MLKEM, LMS, HSS, XMSS, KEYSTORE };
	// ... add a false to the end of this list
	static final boolean [] TESTS = { false, false, false, false, false, false, false, false };

	static boolean dbg = true;

	static AdvancedOptions _singleton = new AdvancedOptions();
	static boolean test = false;

	/**
	 * END of TEST SPECIFIC
	 **********************************************************************/

	public static void main(String[] args) {
		SimpleArgs cla = new SimpleArgs(args);
		dbg = (cla.hasArg("-v") ? true : dbg);

		String device = getDevice(cla, "3001@127.0.0.1");
		//other options for default device = "PCI:0";
		//other options for default device = "288@192.168.4.183";
		// etc

		if (cla.hasArg("devbuild")) {
			module_id = 0x1A6;
		}
		
		int c_lgns = 0;
		ArrayList<Login> logins = new ArrayList<Login>();
		Login lgn = null;
		do {
			lgn = _singleton.new Login(args, c_lgns++);
			logins.add(lgn);
		} while (lgn.valid);

		new PqmiErrorList();

		CryptoServerCXI cxi = null;
		try {
			// create instance of CryptoServerCXI (opens connection to CryptoServer)
			// Note: does not support clusters.
			cxi = new CryptoServerCXI(device, 3000); // connection timeout is 3 seconds

			cxi.setTimeout(360000); // command timeout is 6 minute
			//cxi.setTimeout(60000); // command timeout is 1 minute

			cxi.setKeepSessionAlive(true); // in theory a no-op as the test code doesn't have any waitstates
			cxi.setEndSessionOnShutdown(true); // in theory a no-op as the session is manually closed below

			dbg("Device: " + cxi.getDevice());
			String zeros = "00000000";
			for (Login l : logins) {
				if (!l.valid) { continue; }
				dbg("User: " + l.user);
				l.login(cxi);
				String authstate = zeros + Integer.toHexString(cxi.getAuthState());
				dbg(" -> 0x" + authstate.substring(authstate.length()-8));
			}

			external_mu_test(cxi, cla);
			key_export_test(cxi, cla);

		} catch (IOException e) {
			System.out.println("IOException: " + e.getMessage());
		} catch (CryptoServerException cse) {
			System.out.println("CryptoServerException:  0x" + Integer.toHexString(cse.ErrorCode));
			if (cse.ErrorAnswer != null) {
				// do something
			}
		}

		// cleanup
		if (cxi != null) {
			// cxi.logoff();  		// ends authentication on session
			// cxi.endSession();  	// calls .logoff(), terminates session
			cxi.close(); 			// calls .endSession(), terminates connection
			cxi = null;
		}
	}


	private static void dbg (String d) {
		if (dbg) {
			System.out.println(d);
		}
	}

	private static String getDevice (SimpleArgs args, String def) {
		String cxi = System.getenv("CRYPTOSERVER");
		if (cxi == null) {
			cxi = def;
		}
		if (args.hasArg("dev")) {
			cxi = args.getArg("dev", cxi);
		}
		return cxi;
	}

	private static boolean external_mu_test(CryptoServerCXI cxi, SimpleArgs cla) {

		MLDSA_KeyGen tobj_ctor = null;

		int keytype = Pqmi.ML_DSA_KT_65 | Pqmi.ML_DSA_KT_FLAG;
		byte [] name = "ml_dsa_key".getBytes();
		byte [] group = cla.getArg("-group", "test").getBytes();

		int spec = 0;
		int flags = 0x0;
		int algo = CryptoServerCXI.KEY_ALGO_RAW;
		int usage = CryptoServerCXI.KEY_USAGE_SIGN | CryptoServerCXI.KEY_USAGE_VERIFY;
		int export = 0; // ALLOW_BACKUP only, no export wrapped
		byte [] seed = new byte[0];

		CxiKeyAttributes attributes = new CxiKeyAttributes();
		attributes.flags (flags);
		attributes.name (name);
		attributes.group (group);
		attributes.spec (spec);
		attributes.algo (algo);
		attributes.usage (usage);
		attributes.export (export);

		int outerflags = Pqmi.MODE_PSEUDO_RND | Pqmi.KEY_EXTERNAL;

		try {
			tobj_ctor = new MLDSA_KeyGen (cxi);
			tobj_ctor.flags (outerflags);
			tobj_ctor.type (keytype);
			tobj_ctor.attributes (attributes);
			tobj_ctor.seed (seed);
		} catch (SerializationError ser) {
			System.out.println("Serialization error on typed parameter in DilithiumGen.\n");
			return false;
		}
		try {
			boolean res = tobj_ctor.exec(cxi, module_id, Pqmi.SFC_DILITHIUM_KEYGEN);
			System.out.format("ML-DSA keygen %s\n", res ? "PASSED" : "FAILED");
			if (!res) return false;

		} catch (CryptoServerException e) {
			System.err.format("Error 0x%08X\n", e.ErrorCode);
			if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
				System.err.print(e.getLocalizedMessage());
			} else {
				System.err.println(ErrList.errtext(e.ErrorCode));
			}
			return false;
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}

		MLDSA_Sign tobj_sig = new MLDSA_Sign();
		MLDSA_Verify tobj_verify = new MLDSA_Verify();

		// MODE_EXTERNAL_MU is used to interpret msg parameter as external mu
		int s_flags = Pqmi.DILITHIUM_MODE_SIG_RAW | Pqmi.MODE_PSEUDO_RND | Pqmi.MODE_EXTERNAL_MU;

		// here we use a random value for mu
		// in a real case it would be calculated from the public key, context string and original message using SHAKE256 hashing
		byte [] mu = new byte[64];
		new Random().nextBytes(mu);

		tobj_sig.msg(mu);

		tobj_sig.key(tobj_ctor.resp);
		tobj_verify.key(tobj_ctor.resp);

		tobj_verify.msg(tobj_sig.msg());

		tobj_sig.type(tobj_ctor.type());  // just grabbing the keytype out of the Gen object.
		tobj_sig.flags(s_flags);

		tobj_verify.type(tobj_ctor.type());  // just grabbing the keytype out of the Gen object.
		tobj_verify.flags(s_flags);

		try {
			boolean res = tobj_sig.exec(cxi, module_id, Pqmi.SFC_DILITHIUM_SIGN);
			System.out.format("ML-DSA sign wih external mu %s\n", res ? "PASSED" : "FAILED");
			if (!res) return false;
			// The sv object's .resp is the signature.
		} catch (CryptoServerException e) {
			System.err.format("Error 0x%08X\n", e.ErrorCode);
			if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
				System.err.print(e.getLocalizedMessage());
			} else {
				System.err.println(ErrList.errtext(e.ErrorCode));
			}
			return false;
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}

		tobj_verify.sig(tobj_sig.resp); // this field (resp) contains the signature.

		try {
			boolean res = tobj_verify.exec(cxi, module_id, Pqmi.SFC_DILITHIUM_VERIFY);
			System.out.format("ML-DSA verify with external mu %s\n", res ? "PASSED" : "FAILED");
		} catch (CryptoServerException e) {
			System.err.format("Error 0x%08X\n", e.ErrorCode);
			if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
				System.err.print(e.getLocalizedMessage());
			} else {
				System.err.println(ErrList.errtext(e.ErrorCode));
			}
			return false;
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}

		return true;
	}

	private static boolean key_export_test(CryptoServerCXI cxi, SimpleArgs cla) {

		MLDSA_KeyGen tobj_ctor = null;

		int keytype = Pqmi.ML_DSA_KT_65 | Pqmi.ML_DSA_KT_FLAG;
		byte [] name = "ml_dsa_key".getBytes();
		byte [] group = cla.getArg("-group", "test").getBytes();
		byte [] originalKey, exportedKey, importedKey;

		int spec = 0;
		int flags = 0x0;
		int algo = CryptoServerCXI.KEY_ALGO_RAW;
		int usage = CryptoServerCXI.KEY_USAGE_SIGN | CryptoServerCXI.KEY_USAGE_VERIFY;
		int export = CryptoServerCXI.KEY_EXPORT_ALLOW; // allow export
		byte [] seed = new byte[0];

		CxiKeyAttributes attributes = new CxiKeyAttributes();
		attributes.flags (flags);
		attributes.name (name);
		attributes.group (group);
		attributes.spec (spec);
		attributes.algo (algo);
		attributes.usage (usage);
		attributes.export (export);

		int outerflags = Pqmi.MODE_PSEUDO_RND | Pqmi.KEY_OVERWRITE;

		try {
			tobj_ctor = new MLDSA_KeyGen (cxi);
			tobj_ctor.flags (outerflags);
			tobj_ctor.type (keytype);
			tobj_ctor.attributes (attributes);
			tobj_ctor.seed (seed);
		} catch (SerializationError ser) {
			System.out.println("Serialization error on typed parameter in DilithiumGen.\n");
			return false;
		}
		try {
			boolean res = tobj_ctor.exec(cxi, module_id, Pqmi.SFC_DILITHIUM_KEYGEN);
			System.out.format("ML-DSA keygen %s\n", res ? "PASSED" : "FAILED");
			if (!res) return false;

			originalKey = tobj_ctor.resp;
		} catch (CryptoServerException e) {
			System.err.format("Error 0x%08X\n", e.ErrorCode);
			if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
				System.err.print(e.getLocalizedMessage());
			} else {
				System.err.println(ErrList.errtext(e.ErrorCode));
			}
			return false;
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}

		// this can be skipped if the key is external
		PqmiKeyStore keyStore = null;
		try {
			keyStore = new PqmiKeyStore(Pqmi.KEY_EXPORT_FROM_STORE, keytype, attributes);
			boolean res = keyStore.exec(cxi, module_id, Pqmi.SFC_KEYSTORE);
			System.out.format("ML-DSA key export %s\n", res ? "PASSED" : "FAILED");
			if (!res) return false;
			exportedKey = keyStore.resp;
		} catch (CryptoServerException e) {
			System.err.format("Error 0x%08X\n", e.ErrorCode);
			if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
				System.err.print(e.getLocalizedMessage());
			} else {
				System.err.println(ErrList.errtext(e.ErrorCode));
			}
			return false;
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}

		CxiKeyAttributes attributes2 = new CxiKeyAttributes();
		attributes2.flags (flags);
		attributes2.name ("ml_dsa_key_imp".getBytes());
		attributes2.group (group);
		attributes2.spec (spec);
		attributes2.algo (algo);
		attributes2.usage (usage);
		attributes2.export (export);

		PqmiKeyImport keyImport = null;
		try {
			keyImport = new PqmiKeyImport(Pqmi.KEY_OVERWRITE, attributes2, exportedKey);
			boolean res = keyImport.exec(cxi, module_id, Pqmi.SFC_KEYSTORE_IMPORT);
			System.out.format("ML-DSA key import %s\n", res ? "PASSED" : "FAILED");
			if (!res) return false;
			importedKey = keyImport.resp;
		} catch (CryptoServerException e) {
			System.err.format("Error 0x%08X\n", e.ErrorCode);
			if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
			System.err.print(e.getLocalizedMessage());
			} else {
				System.err.println(ErrList.errtext(e.ErrorCode));
			}
			return false;
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}

		// sign with imported key, verify with original
		MLDSA_Sign tobj_sig = new MLDSA_Sign();
		int s_flags = Pqmi.DILITHIUM_MODE_SIG_RAW | Pqmi.MODE_PSEUDO_RND | MLDSA_Sign.FLAG_HAS_CTXT;

		tobj_sig.key(importedKey);
		tobj_sig.ctxt("ctxt");
		tobj_sig.msg("Test Message");
		tobj_sig.type(tobj_ctor.type());  // just grabbing the keytype out of the Gen object.
		tobj_sig.flags(s_flags);

		try {
			boolean res = tobj_sig.exec(cxi, module_id, Pqmi.SFC_DILITHIUM_SIGN);
			System.out.format("ML-DSA sign wih imported key %s\n", res ? "PASSED" : "FAILED");
			if (!res) return false;
			// The sv object's .resp is the signature.
		} catch (CryptoServerException e) {
			System.err.format("Error 0x%08X\n", e.ErrorCode);
			if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
				System.err.print(e.getLocalizedMessage());
			} else {
				System.err.println(ErrList.errtext(e.ErrorCode));
			}
			return false;
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}

		MLDSA_Verify tobj_verify = new MLDSA_Verify();
		tobj_verify.key(originalKey);
		tobj_verify.ctxt(tobj_sig.ctxt());
		tobj_verify.msg(tobj_sig.msg());
		tobj_verify.type(tobj_ctor.type());  // just grabbing the keytype out of the Gen object.
		tobj_verify.flags(s_flags);
		tobj_verify.sig(tobj_sig.resp); // this field (resp) contains the signature.

		try {
			boolean res = tobj_verify.exec(cxi, module_id, Pqmi.SFC_DILITHIUM_VERIFY);
			System.out.format("ML-DSA verify with original key %s\n", res ? "PASSED" : "FAILED");
		} catch (CryptoServerException e) {
			System.err.format("Error 0x%08X\n", e.ErrorCode);
			if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
				System.err.print(e.getLocalizedMessage());
			} else {
				System.err.println(ErrList.errtext(e.ErrorCode));
			}
			return false;
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}

		return true;
	}


	private class Login {
		String user = null;
		String spec = null;
		String pin = null;

		File file = null;
		boolean valid = false;
		// valid indicates that this /may/ represent a valid credential, not that it /is/ a valid login

		/*		
		user		Name of the user who wants to autenticate to the CryptoServer.
		keyFile		<-- mapped from -cred, or -s or if mistakenly -p with 3 parameters
			Key file user: Path to key file containing user's private key.
			Password user: null.
		password	<-- mapped from -cred, or -p
			Key file user: Password of the key file if using an encrypted file, null otherwise.
			Password user: Password of the user.
		 */
		Login (String [] args, int i) {
			int c = 0;
			boolean getNext = false;
			for (String a : args) {
				if (getNext) {
					getNext = false;
					String[] nkp = a.split(",");
					if (nkp.length == 2) {
						/* either 
						 * -s name,key (unencrypted)
						 * -s name,:cs2...
						 * -p name,pass
						 */
						user = nkp[0];
						spec = nkp[1];  
						pin = nkp[1]; // yes -- 1.  see later check

						file = new File(spec);
						valid = (spec.matches("^:cs2:.{3,4}:.*$")) || file.exists();

					} else if (nkp.length == 3) {
						// * -s name,key,pass (encrypted)
						user = nkp[0];
						spec = nkp[1];
						pin = nkp[2];
						file = new File(spec);
						valid = file.exists();
					}
					/* at this point, valid is true if the file exists, or if the spec is :cs2:...
					 * valid is false if the value isn't a file and the value isn't a PINPad identifier
					 * 
					 * In this case, we assume HMACpwd, if we've made it this far
					 */
					valid = true; // assume it's HMACpwd
					/*
					 * I may find some logic later for pre-determining if this is an hmac
					 * but I can't test it (using the HSM) because it would log everyone out if it was bad.
					 *
					 * For now, we rely on the cxi.logon(...) to bork if it is not a valid logon.
					 */

					if (!valid) { System.out.println("Auth param format invalid for " + a); }
					c++;
				}
				/* deprecated */
				if (a.compareTo("-s") == 0) {
					if (c != i) { c++; continue; }
					getNext = true;
				}
				/* deprecated */
				if (a.compareTo("-p") == 0) {
					if (c != i) { c++; continue; }
					getNext = true;
				}

				if (a.compareTo("-cred") == 0) {
					if (c != i) { c++; continue; }
					getNext = true;
				}
			}
		}

		boolean login (CryptoServerCXI cxi) {
			boolean ret = false;

			try {
				/* change 01g replaced logonpass/logonsign with the simple logon */
				byte [] pinbytes = new byte[0];
				if (pin != null) {
					pinbytes = pin.getBytes();
				}
				byte [] ovrt = new byte[pinbytes.length];
				cxi.logon(user, spec, pinbytes);
				System.arraycopy(ovrt,0,pinbytes,0,pinbytes.length);
				// also have pin to worry about.
				// pin = null;
				// but we don't know when the GC will reap it.
				return true;
			} catch (IOException e) {
				System.out.println(user + " login failed:  " + e.getMessage());
			} catch (CryptoServerException e) {
				System.out.println(user + " login failed:  0x" + Integer.toHexString(e.ErrorCode));
			}
			return ret;
		}
	}
}
