/*******************************************************************************
 *
 *  $Filename       : hbs.h           $
 *  $Version        : 1.2.0.0         $
 *  $Module         : HBS             $
 *  $Release Date   : 21.12.2023      $
 *
 *  Author          : Utimaco IS GmbH
 *
 *  History         :
 *  Date        |   Action                                  |   Author
 *  ------------|-------------------------------------------|-------------------
 *  20.03.2023  |   Creation                                |   DD
 *  24.07.2023  |   SHA-256/192 support                     |   DD
 *  21.12.2023  |   XMSS                                    |   DD
 *  ------------|-------------------------------------------|-------------------
 ******************************************************************************/

#ifndef __HBS_H_PUB_INCLUDED__
#define __HBS_H_PUB_INCLUDED__

#include <os_mdl.h>
#include <os_file.h>

#define HBS_MAX_HSS_LEVELS 8

#define HBS_RNG_TYPE_PSEUDO				0
#define HBS_RNG_TYPE_REAL				1

//-----------------------------------------------------------------------------
// module identification:
//-----------------------------------------------------------------------------
//#define HBS_MDL_ID              0x1A2
#define HBS_MDL_ID              0xA2
#define HBS_MDL_NAME            "HBS"
#ifdef CS2_SDK
    #define HBS_MDL_NAMEX       "Hash Based Signatures"        // Long name of the module
#else
    #ifdef DEBUG
        #define HBS_MDL_NAMEX   "Hash Based Signatures DEBUG"  // Long name of the module
    #else
        #define HBS_MDL_NAMEX   "Hash Based Signatures"        // Long name of the module
    #endif
#endif
#define HBS_MDL_VERSION         0x01040200                      // Version of the module  (d.x.d.d)


//-----------------------------------------------------------------------------
// public interface:
//-----------------------------------------------------------------------------
typedef struct
{
            void    *p_data;
            int     (*p_start)(T_OS_MDL_HDL, OS_FILE_HANDLE, void*);
            int     (*p_stop)(void);
            int     (*p_pre_replace)(void);
            int     (*p_pre_delete)(void);
            void    *dumy;
    const   void    *p_module_info;

	int (*p_hbs_lms_keygen)(
		int				rng_type,   // (I) RNG type
		int				levels,     // (I) number of LMS/OTS levels
		const int		lm_array[HBS_MAX_HSS_LEVELS],  // LMS parameters array
		const int		ots_array[HBS_MAX_HSS_LEVELS], // OTS parameters array
		unsigned long   *p_l_x,      // (O) length of private key
		unsigned char  **pp_x,      // (O) private key
		unsigned long   *p_l_y,      // (O) length of public key
		unsigned char  **pp_y,      // (O) public key
		unsigned long   *p_l_aux,    // (O) length of aux data
		unsigned char  **pp_aux     // (O) aux data
		);

	int(*p_hbs_lms_tokengen)(
		int				rng_type,   // (I) RNG type
		int				levels,     // (I) number of LMS/OTS levels
		const int		lm_array[HBS_MAX_HSS_LEVELS],  // LMS parameters array
		const int		ots_array[HBS_MAX_HSS_LEVELS], // OTS parameters array
		unsigned long   *p_l_x,      // (I/O) length of private key
		unsigned char  *p_x,        // (O) private key
		unsigned long   *p_l_y,      // (I/O) length of public key
		unsigned char  *p_y,        // (O) public key
		unsigned long   *p_l_aux,    // (I/O) length of aux data
		unsigned char  *p_aux       // (O) aux data
		);

	int(*p_hbs_lms_sign)(
		unsigned long   l_x,				// (I) length of private key
		unsigned char  *p_x,			// (I/O) private key
		unsigned long   l_aux,			// (I) length of aux data
		const unsigned char  *p_aux,	// (I) aux data
		unsigned long   l_msg,			// (I) length of message
		const unsigned char  *p_msg,	// (I) message
		unsigned long   *p_l_sig,		// (I/O) length of signature
		unsigned char  *p_sig			// (O) signature
		);

	int(*p_hbs_lms_verify)(
		unsigned long   l_y,				// (I) length of public key
		const unsigned char  *p_y,		// (I) public key
		unsigned long   l_msg,			// (I) length of message
		const unsigned char  *p_msg,	// (I) message
		unsigned long   l_sig,			// (I) length of signature
		const unsigned char  *p_sig		// (I) signature
		);

	int(*p_hbs_xmss_keygen)(
		int				rng_type,   // (I) RNG type
		int				mt,         // (I) ST / MT
		unsigned long    oid,		// (I) OID
		unsigned long   *p_l_x,      // (O) length of private key
		unsigned char  **pp_x,      // (O) private key
		unsigned long   *p_l_y,      // (O) length of public key
		unsigned char  **pp_y       // (O) public key
		);

	int(*p_hbs_xmss_keygen_seed)(
		int				mt,         // (I) ST / MT
		unsigned long    oid,		// (I) OID
		unsigned long    l_seed,     // (I) seed length
		unsigned char  *p_seed,     // (I) seed
		unsigned long   *p_l_x,      // (O) length of private key
		unsigned char  **pp_x,      // (O) private key
		unsigned long   *p_l_y,      // (O) length of public key
		unsigned char  **pp_y       // (O) public key
		);

	int(*p_hbs_xmss_tokengen)(
		int				rng_type,   // (I) RNG type
		int				mt,         // (I) ST / MT
		unsigned long    oid,		// (I) OID
		unsigned long   *p_l_x,      // (I/O) length of private key
		unsigned char  *p_x,        // (O) private key
		unsigned long   *p_l_y,      // (I/O) length of public key
		unsigned char  *p_y         // (O) public key
		);

	int(*p_hbs_xmss_sign)(
		int			   mt,              // (I) ST / MT
		unsigned long   oid,		        // (I) OID
		unsigned long   l_x,				// (I) length of private key
		unsigned char  *p_x,			// (I/O) private key
		unsigned long   l_msg,			// (I) length of message
		const unsigned char  *p_msg,	// (I) message
		unsigned long   *p_l_sig,		// (I/O) length of signature
		unsigned char  *p_sig			// (O) signature
		);

	int(*p_hbs_xmss_verify)(
		int			   mt,              // (I) ST / MT
		unsigned long   oid,		        // (I) OID
		unsigned long   l_y,				// (I) length of public key
		const unsigned char  *p_y,		// (I) public key
		unsigned long   l_msg,			// (I) length of message
		const unsigned char  *p_msg,	// (I) message
		unsigned long   l_sig,			// (I) length of signature
		const unsigned char  *p_sig		// (I) signature
		);

}
T_HBS_TABLE_PUB;

#ifdef _HBS_C_INT_
    //-----------------------------------------------------------------------------
    // function prototypes used by the module (internal)
    //-----------------------------------------------------------------------------
    int     hbs_start(T_OS_MDL_HDL,OS_FILE_HANDLE,void *);
    int     hbs_stop(void);
    int     hbs_pre_replace(void);
    int     hbs_pre_delete(void);
    int     hbs_sync(void);

	int hbs_lms_keygen(
		int				rng_type,   // (I) RNG type
		int				levels,     // (I) number of LMS/OTS levels
		const int		lm_array[HBS_MAX_HSS_LEVELS],  // LMS parameters array
		const int		ots_array[HBS_MAX_HSS_LEVELS], // OTS parameters array
		unsigned long   *p_l_x,      // (O) length of private key
		unsigned char  **pp_x,      // (O) private key
		unsigned long   *p_l_y,      // (O) length of public key
		unsigned char  **pp_y,      // (O) public key
		unsigned long   *p_l_aux,    // (O) length of aux data
		unsigned char  **pp_aux     // (O) aux data
		);

	int hbs_lms_tokengen(
		int				rng_type,   // (I) RNG type
		int				levels,     // (I) number of LMS/OTS levels
		const int		lm_array[HBS_MAX_HSS_LEVELS],  // LMS parameters array
		const int		ots_array[HBS_MAX_HSS_LEVELS], // OTS parameters array
		unsigned long   *p_l_x,      // (I/O) length of private key
		unsigned char  *p_x,        // (O) private key
		unsigned long   *p_l_y,      // (I/O) length of public key
		unsigned char  *p_y,        // (O) public key
		unsigned long   *p_l_aux,    // (I/O) length of aux data
		unsigned char  *p_aux       // (O) aux data
	);

	int hbs_lms_sign(
		unsigned long   l_x,				// (I) length of private key
		unsigned char  *p_x,			// (I/O) private key
		unsigned long   l_aux,			// (I) length of aux data
		const unsigned char  *p_aux,	// (I) aux data
		unsigned long   l_msg,			// (I) length of message
		const unsigned char  *p_msg,	// (I) message
		unsigned long   *p_l_sig,		// (I/O) length of signature
		unsigned char  *p_sig			// (O) signature
		);

	int hbs_lms_verify(
		unsigned long   l_y,				// (I) length of public key
		const unsigned char  *p_y,		// (I) public key
		unsigned long   l_msg,			// (I) length of message
		const unsigned char  *p_msg,	// (I) message
		unsigned long   l_sig,			// (I) length of signature
		const unsigned char  *p_sig		// (I) signature
		);

	int hbs_xmss_keygen(
		int				rng_type,   // (I) RNG type
		int				mt,         // (I) ST / MT
		unsigned long    oid,		// (I) OID
		unsigned long   *p_l_x,      // (O) length of private key
		unsigned char  **pp_x,      // (O) private key
		unsigned long   *p_l_y,      // (O) length of public key
		unsigned char  **pp_y       // (O) public key
	);

	int hbs_xmss_keygen_seed(
		int				mt,         // (I) ST / MT
		unsigned long    oid,		// (I) OID
		unsigned long    l_seed,     // (I) seed length
		unsigned char  *p_seed,     // (I) seed
		unsigned long   *p_l_x,      // (O) length of private key
		unsigned char  **pp_x,      // (O) private key
		unsigned long   *p_l_y,      // (O) length of public key
		unsigned char  **pp_y       // (O) public key
	);

	int hbs_xmss_tokengen(
		int				rng_type,   // (I) RNG type
		int				mt,         // (I) ST / MT
		unsigned long    oid,		// (I) OID
		unsigned long   *p_l_x,      // (I/O) length of private key
		unsigned char  *p_x,        // (O) private key
		unsigned long   *p_l_y,      // (I/O) length of public key
		unsigned char  *p_y         // (O) public key
	);

	int hbs_xmss_sign(
		int			   mt,              // (I) ST / MT
		unsigned long   oid,		        // (I) OID
		unsigned long   l_x,				// (I) length of private key
		unsigned char  *p_x,			// (I/O) private key
		unsigned long   l_msg,			// (I) length of message
		const unsigned char  *p_msg,	// (I) message
		unsigned long   *p_l_sig,		// (I/O) length of signature
		unsigned char  *p_sig			// (O) signature
	);

	int hbs_xmss_verify(
		int			   mt,              // (I) ST / MT
		unsigned long   oid,		        // (I) OID
		unsigned long   l_y,				// (I) length of public key
		const unsigned char  *p_y,		// (I) public key
		unsigned long   l_msg,			// (I) length of message
		const unsigned char  *p_msg,	// (I) message
		unsigned long   l_sig,			// (I) length of signature
		const unsigned char  *p_sig		// (I) signature
	);

#else
    //-----------------------------------------------------------------------------
    // external interface to be used by other modules
    //-----------------------------------------------------------------------------
    extern  MDL_GLOBAL      T_OS_MDL_HDL P_HBS;

    #define _P_HBS          ((T_HBS_TABLE_PUB *)P_HBS)         // shortcut

    #define P_hbs_data      (_P_HBS->_p_data);

    #define hbs_start           _P_HBS->p_start
    #define hbs_stop            _P_HBS->p_stop
    #define hbs_pre_replace     _P_HBS->p_pre_replace
    #define hbs_pre_delete      _P_HBS->p_pre_delete
    #define hbs_sync            _P_HBS->p_sync

	#define hbs_lms_keygen      _P_HBS->p_hbs_lms_keygen
	#define hbs_lms_tokengen    _P_HBS->p_hbs_lms_tokengen
	#define hbs_lms_sign        _P_HBS->p_hbs_lms_sign
	#define hbs_lms_verify      _P_HBS->p_hbs_lms_verify

	#define hbs_xmss_keygen			_P_HBS->p_hbs_xmss_keygen
	#define hbs_xmss_keygen_seed    _P_HBS->p_hbs_xmss_keygen_seed
	#define hbs_xmss_tokengen		_P_HBS->p_hbs_xmss_tokengen
	#define hbs_xmss_sign			_P_HBS->p_hbs_xmss_sign
	#define hbs_xmss_verify			_P_HBS->p_hbs_xmss_verify

#endif // _HBS_C_INT_


//-----------------------------------------------------------------------------
// Error Codes:  (0xB0000000 | (HBS_MDL_ID << 16) | (errno))
//-----------------------------------------------------------------------------

// --- BEGIN ERROR CODES ---

#define E_HBS_GENERAL                   0xB0A20000  // HBS general error
#define E_HBS_INTERNAL                  0xB0A20001  // internal error
#define E_HBS_MALLOC                    0xB0A20002  // memory allocation error
#define E_HBS_INVALID_MECH_PARAMS       0xB0A20003  // invalid mechanism parameters size 
#define E_HBS_INVALID_PARAM_VALUE       0xB0A20004  // invalid algorithm pararameters
#define E_HBS_GENKEY_FAILED             0xB0A20005  // key generation failed
#define E_HBS_INVALID_PRIV_KEY          0xB0A20006  // invalid key
#define E_HBS_SIGN_ERROR                0xB0A20007  // signature error
#define E_HBS_VERIFY_ERROR              0xB0A20008  // verification error
#define E_HBS_SIGNATURE_REQUIRED        0xB0A20009  // signature required at init
#define E_HBS_DB_INSERT_FAILED          0xB0A2000A  // unable to save data to HBS datatbase
#define E_HBS_BUFFER_TOO_SMALL          0xB0A2000B  // buffer too small

#define E_HBS_INVALID_TAG_LEN			0xB0A2000C  // invalid TLV tag length   
#define E_HBS_INVALID_TAG				0xB0A2000D  // invalid TLV tag
#define E_HBS_OUT_OF_TAG_SPACE          0xB0A2000E  // not enough memory for TLV
#define E_HBS_BLOB_FORMAT               0xB0A2000F  // invalid format of BLOB
#define E_HBS_BLOB_CV                   0xB0A20010  // control value of BLOB failed
#define E_HBS_INVALID_KEY               0xB0A20011  // invalid key

#define E_HBS_NO_KEYS_LEFT				0xB0A20100  // all key indexes used
#define E_HBS_INVALID_INTERVALS			0xB0A20101  // invalid intervals database
#define E_HBS_INTERVALS_BUFFER_LEN      0xB0A20102  // out of memory for intervals
#define E_HBS_KEYS_NOT_AVAILABLE        0xB0A20103  // no key index available
#define E_HBS_INTERVALS_OVERLAP         0xB0A20104  // error joining two intervals
#define E_HBS_EXCEEDED_INTERVALS_LIMIT  0xB0A20105  // to many intervals
#define E_HBS_PERMISSION_DENIED			0xB0A20106  // permission denied
#define E_HBS_VERSION_NOT_SUPPORTED		0xB0A20107  // version not supported
#define E_HBS_IMPORT_NOT_ALLOWED		0xB0A20108  // import not allowed

#define E_HBS_INIT_SEM_FAILED			0xB0A20109  // initialization of database locking failed
#define E_HBS_REQ_SEM_FAILED			0xB0A2010A  // database locking failed

#define E_HBS_INVALID_HSM_ID				0xB0A20200  // invalid HSM ID
#define E_HBS_CORRUPTED_HSM_ID				0xB0A20201  // corrupted HSM ID
#define E_HBS_UNSUPPORTED_HSM_ID			0xB0A20202  // HSM ID not supported
#define E_HBS_HSM_ID_EXISTS					0xB0A20203  // HSM ID already exists
#define E_HBS_HSM_ID_NOT_FOUND				0xB0A20204  // HSM ID not found

#define E_HBS_INVALID_TRUSTED_HSM_ID		0xB0A2020A  // invalid trusted HSM ID
#define E_HBS_CORRUPTED_TRUSTED_HSM_ID		0xB0A2020B  // corrupted trusted HSM ID
#define E_HBS_UNSUPPORTED_TRUSTED_HSM_ID	0xB0A2020C  // trusted HSM ID not supported
#define E_HBS_TRUSTED_HSM_ID_EXISTS			0xB0A2020D  // trusted HSM ID already exists
#define E_HBS_TRUSTED_HSM_ID_NOT_FOUND		0xB0A2020E  // trusted HSM ID not found

#define E_HBS_INVALID_COUNTER_RECORD		0xB0A2020F  // invalid counter record
#define E_HBS_CORRUPTED_COUNTER_RECORD		0xB0A20210  // corrupted counter record
#define E_HBS_UNSUPPORTED_COUNTER_RECORD	0xB0A20211  // counter record not supported
#define E_HBS_COUNTER_RECORD_EXISTS			0xB0A20212  // counter record already exists

#define E_HBS_INVALID_DB_RECORD				0xB0A20213  // invalid database record
#define E_HBS_CORRUPTED_DB_RECORD			0xB0A20214  // corrupted database record
#define E_HBS_UNSUPPORTED_DB_RECORD			0xB0A20215  // database record not supported

#define E_HBS_COUNTER_OVERFLOW				0xB0A20216  // counter overflow

#define E_HBS_INVALID_MESSAGE				0xB0A20217  // invalid message
#define E_HBS_CORRUPTED_MESSAGE				0xB0A20218  // corrupted message
#define E_HBS_UNSUPPORTED_MESSAGE			0xB0A20219  // message not supported
#define E_HBS_INVALID_RECEIVER_HSM_ID		0xB0A2021A  // invalid receiver HSM ID
#define E_HBS_INVALID_KEY_REFERENCE			0xB0A2021B  // invalid key reference
#define E_HBS_MESSAGE_EXPIRED    			0xB0A2021C  // message expired 
#define E_HBS_INVALID_COUNTERS_SIZE			0xB0A2021D  // invalid counter size

#define E_HBS_INVALID_KEYTYPE  				0xB0A2021E  // invalid key type

// --- END ERROR CODES ---

#endif // __HBS_H_PUB_INCLUDED__
