/**************************************************************************************************
 *
 * Filename   : csxapi.h
 *
 * Author     : Dipl. Ing. Sven Kaltschmidt
 *              Utimaco IS GmbH
 *
 * Description: CryptoServer High Level Communication API
 *
 *************************************************************************************************/
#ifndef __CSXAPI_H
#define __CSXAPI_H

#ifndef CSXAPI
  #ifdef WIN32
    #define CSXAPI
    #define CSXAPI_LOCAL
  #else
    #if __GNUC__ >= 4
      #define CSXAPI __attribute__ ((visibility ("default")))
      #define CSXAPI_LOCAL  __attribute__ ((visibility ("hidden")))
    #else
      #define CSXAPI
      #define CSXAPI_LOCAL
    #endif
  #endif
#else
  #define CSXAPI_LOCAL
#endif


#include "cs_core.h"

#include <stddef.h>

/******************************************************************************
 *
 * Definitions
 *
 ******************************************************************************/
#define CSA_AUTH_MECH_RSA_SIGN      0
#define CSA_AUTH_MECH_CLR_PWD       1
#define CSA_AUTH_MECH_SHA1_PWD      2
#define CSA_AUTH_MECH_RSA_SC        3
#define CSA_AUTH_MECH_HMAC_PWD      4
#define CSA_AUTH_MECH_ECDSA_SIGN    5

#define CSA_AUTH_MECH_AUTO          0xFF


#define DEFAULT_CONNECT_TIMEOUT     10000
#define DEFAULT_READ_TIMEOUT        30000
#define DEFAULT_KEEP_ALIVE          (5 * 60)

/******************************************************************************
 *
 * Errorcodes
 *
 ******************************************************************************/

#define SYS_ERR(type,err) (((type) << 12) | ((err) & 0xFFF))

// --- BEGIN ERROR CODES ---

#define E_CSXAPI                        0xB90C        // CSXAPI
#define E_CSXAPI_ALLOC                  0xB90C0001    // memory allocation failed
#define E_CSXAPI_DATA_LEN               0xB90C0002    // invalid data length
#define E_CSXAPI_RESP_LEN               0xB90C0003    // invalid response length
#define E_CSXAPI_BUF_SIZE               0xB90C0004    // buffer size too small
#define E_CSXAPI_PARAM                  0xB90C0005    // invalid parameter value
#define E_CSXAPI_MAX_PIN                0xB90C0006    // maximum number of cached pins reached
#define E_CSXAPI_SESSION_INVALID        0xB90C0007    // invalid session
#define E_CSXAPI_SESSION_ERROR          0xB90C0008    // connection or session error, session must be closed

#define E_CSXAPI_AUTH_KEY_TYPE_UNSUPPORTED   0xB90C0020  // HSM auth key type not supported
#define E_CSXAPI_AUTH_KEY_MODE_UNSUPPORTED   0xB90C0021  // HSM auth key mode not supported

#define E_CSXAPI_CLUSTER                0xB90C01      // Cluster API
#define E_CSXAPI_CLUSTER_OPEN           0xB90C0101    // unable to open any device
#define E_CSXAPI_CLUSTER_LOGON          0xB90C0102    // unable to logon to any device
#define E_CSXAPI_CLUSTER_EXEC           0xB90C0103    // unable to execute command on any device
#define E_CSXAPI_CLUSTER_NAME           0xB90C0104    // unknown device name
#define E_CSXAPI_CLUSTER_CON_PARAM      0xB90C0105    // invalid connection parameter
#define E_CSXAPI_CLUSTER_HDL_IN_USE     0xB90C0106    // handle still in use

#define E_CSXAPI_WIN                    0xB90C1       // windows system error
#define E_CSXAPI_UNIX                   0xB90C2       // unix system error

// --- END ERROR CODES ---


/******************************************************************************
 *
 * Function Prototypes
 *
 ******************************************************************************/

#ifdef __cplusplus
  extern "C" {
#endif


/*
 * Opens a connection to a CryptoServer (either PCI or TCP)
 */
CSXAPI int cs_open_connection
(
  const char    *device,          // I: device specifier (e.g. PCI:0 / 192.168.1.1)
  unsigned int  connect_timeout,  // I: connection timeout [ms]
  unsigned int  read_timeout,     // I: read (command) timeout [ms]
  int           *p_h_cs           // O: connection handle
);

/*
 * Closes a connection to a CryptoServer
 */
CSXAPI int cs_close_connection
(
  int           h_cs              // I: connection handle
);

/*
 * Sets read (command) timeout
 */
CSXAPI int cs_set_timeout
(
  int           h_cs,             // I: connection handle
  int           timeout           // I: read (command) timeout to be set [ms]
);

/*
 * Returns the current read (command) timeout value
 */
CSXAPI int cs_get_timeout
(
  int           h_cs,             // I: connection handle
  int           *p_timeout        // O: current read (command) timeout [ms]
);

/*
 * Prepares authentication but doesn't create secure messaging.
 * Multiple authentications may be chained.
 */
CSXAPI int cs_prepare_auth
(
  int                 h_cs,          // I: connection handle
  unsigned int        mech,          // I: CSA_AUTH_MECH_XXX
  const unsigned char *p_user,       // I: user name
  unsigned int        l_user,        // I: length of user name (e.g. strlen(p_user) )
  const char          *p_key,        // I: smartcard specifier or key file (zero term. string)
  const unsigned char *p_pwd,        // I: pin (smartcard) or password (key file)
  unsigned int        l_pwd,         // I: length of password (e.g. strlen(p_pwd) )
  void                *p_sess_ctx,   // I: previous context or NULL
  void                **pp_sess_ctx  // O: new context
);

/*
 * Ends authentication and frees all resources.
 */
CSXAPI void cs_end_authentication
(
  void          *p_sess_ctx       // I: pointer to session context
);

/*
 * Creates a secure messaging (encrypted) session.
 */
CSXAPI int cs_get_sessionkey_ex
(
  int           h_cs,             // I: connection handle
  void          *p_sess_ctx       // I: pointer to session context
);

enum cert_chain_result {
    CERT_CHAIN_OK,
    CERT_CHAIN_INVALID,
    CERT_CHAIN_MISSING
};

enum cert_id {
    id_cak_dak,
    id_dak_vend,
    id_dak_oper,
    id_cak_cust
};

struct cert_dump_node {
    struct cert_dump_node *next;
    enum cert_id cert_id;
    char *cert;
    size_t l_cert;
};

struct cert_dump {
    struct cert_dump_node *dump;
};

struct cert_chain_info {
    const char *root;
    size_t l_root;
    enum cert_chain_result res;
};

struct handshake_cert_info {
    struct cert_chain_info vend;    /**< Vendor's root certificate/chain verification result. */
    struct cert_chain_info oper;    /**< Device operator's root certificate/chain verification result. */
    struct cert_chain_info cust;    /**< Customer's root certificate/chain verification result. */
};

/**
 * @brief Free memory allocated by \p cs_get_sessionkey_cert.
 */
void cs_free_cert_dump(struct cert_dump *cert_dump);

/**
 * @brief Open a new Secure Messaging session.
 * @param[in] h_cs Handle to an open CryptoServer connection as returned by \p cs_open_connection.
 * @param[in, out] p_sess_ctx Session context as returned by \p cs_prepare_auth or a previous call to
 *      \p cs_get_sessionkey_cert. If the function returns 0, this will be a valid session context.
 *      In this case, it can be used in \p cs_exec_command to execute remote commands over a secure
 *      channel and must eventually be freed by a call to \p cs_end_session.
 * @param[in, out] cert_info Set the root certificates for the chains you wish to verify. If the
 *      field l_root of a certificate is > 0, the res field will be set to the result of the
 *      chain verification. All chains will be checked, regardless of the results of earlier chains.
 *      Set to NULL if you don't care about certificate chains.
 * @param[out] cert_dump If this is not NULL, a singly linked list of all certificate chains received
 *      from the server will be dumped here. Must be freed using \p cs_free_cert_dump.
 * @return 0 on success, error code otherwise.
 */
int cs_get_sessionkey_cert
(
  int h_cs,
  void *p_sess_ctx,
  struct handshake_cert_info *cert_info,
  struct cert_dump *cert_dump
);

/*
 * Enable or disable an additional thread that keeps a session alive and prevents it from expiring after 15 minutes idle time.
 */
CSXAPI int cs_keep_session_alive_ex
(
  int   h_cs,                     // I:
  void  *p_sess_ctx,              // I:
  int   time_interval             // I: time interval in seconds
);

/*
 * Ends a session and frees all resources.
 * Don't use session context pointer later.
 */
CSXAPI void cs_end_session
(
  int           h_cs,             // I: connection handle
  void          *p_sess_ctx       // I: pointer to session context
);

/*
 * Retrieves resulting authentication state
 */
CSXAPI int cs_get_auth_state
(
  int           h_cs,             // I: connection handle
  void          *p_sess_ctx,      // I: pointer to session context
  int           *p_state          // O: resulting authentication state
);

/// Sends Restart command to Cryptoserver
#define cs_restart_device(handle) cs_restart_device_ex(handle, NULL)
CSXAPI int cs_restart_device_ex
(
  int           h_cs,             // I: connection handle
  void          *p_sess_ctx       // I: pointer to session context
);

/// Query information about model of Cryptoserver used: 1: CS2000,CS10/CS50  2: CS Se
CSXAPI int cs_get_model_nr(int cs_handle, int* model);

CSXAPI int cs_get_driver_info(int handle, CSA_INFOX* drv_info);

CSXAPI int csx_set_msg_handler(int handle, void (*p_msg)(unsigned char*,int));
/*
 * Executes a command in a thread-safe way.
 */
CSXAPI int cs_exec_command
(
  int           h_cs,             // I: connection handle
  void          *p_sess_ctx,      // I: pointer to session context
  unsigned int  fc,               // I: function code (module id)
  unsigned int  sfc,              // I: sub function code
  const unsigned char *p_cmd,     // I: pointer to command data
  unsigned int  l_cmd,            // I: length of command data
  unsigned char **pp_answ,        // O: pointer to answer data (memory allocation in error-free case)
  unsigned int  *p_l_answ         // O: pointer to answer length
);

/*
 * Frees answer buffer.
 */
CSXAPI void cs_free_answ(unsigned char *p_answ);

/*
 * Converts error code into error message text (if possible). Do not use in threadsafe environment.
 */
CSXAPI char *cs_get_error_msg(int err);

/*
 * Converts error code into error message text (if possible) and writes it into a caller supplied buffer.
 */
CSXAPI char *cs_get_error_msg_buf(int err, char* psz, unsigned int nsz);

/*
 * @deprecated Use cs_prepare_auth and cs_get_sessionkey_ex instead
 */
CSXAPI int cs_logon
(
  int           h_cs,             // I: connection handle
  unsigned int  mech,             // I: CSA_AUTH_MECH_XXX
  unsigned char *p_user,          // I: user name
  unsigned int  l_user,           // I: length of user name (e.g. strlen(p_user) )
  char          *p_key,           // I: smartcard specifier or key file (zero term. string)
  unsigned char *p_pwd,           // I: pin (smartcard) or password (key file)
  unsigned int  l_pwd,            // I: length of password (e.g. strlen(p_pwd) )
  void          *p_sess_ctx,      // I: previous context or NULL
  void          **pp_sess_ctx     // O: new context
);

/*
 * @deprecated Use cs_end_session instead
 */
CSXAPI void cs_logoff
(
  int           h_cs,             // I: connection handle
  void          *p_sess_ctx       // I: pointer to session context
);

/******************************************************************************
 *
 * CryptoServer Cluster API
 *
 ******************************************************************************/

typedef struct
{
  void (*state_changed)(char *device, int state);
  void (*error_occurred)(char *device, int err, char *func_name);
}
CS_EVENT_HANDLER;

typedef struct
{
  const char        **device_tab;       // I: array of device specifier
  unsigned int      device_cnt;         // I: number of devices
  unsigned int      connect_timeout;    // I: connection timeout [ms]
  unsigned int      read_timeout;       // I: read (command) timeout [ms]
  unsigned int      fallback_interval;  // I: fallbackinterval
  CS_EVENT_HANDLER  *eventHandler;      // I: eventHandler
}
CON_PARAM;

CSXAPI void cs_cluster_set_event_handler
(
  CS_EVENT_HANDLER  *p_event_handler  // I: event handler
);

CSXAPI int cs_cluster_open
(
  const char    **devices,        // I: array of device specifier
  unsigned int  ndevs,            // I: number of devices
  unsigned int  connect_timeout,  // I: connection timeout [ms]
  unsigned int  read_timeout,     // I: read (command) timeout [ms]
  int           *p_hdl            // O: cluster handle
);

CSXAPI int cs_cluster_open_ex
(
  CON_PARAM         *p_param,     // I: connection parameters
  unsigned int      l_param,      // I: sizeof p_param
  int               *p_hdl        // O: cluster handle
);

CSXAPI int cs_cluster_close
(
  int hdl                         // I : cluster handle
);

CSXAPI int cs_cluster_get_hdl
(
  int hdl,                        // I: cluster handle
  int *p_h_cs                     // O: current device handle
);

//extract a session context from cluster handle
CSXAPI void* cs_cluster_get_session_ctx
(
  int hdl                         // I: cluster handle
);

CSXAPI int cs_cluster_set_msg_handler
(
  int hdl,                        // I: cluster handle
  void (*p_msg_func)(unsigned char *p_msg, int l_msg)   // I: message handler callback
);

CSXAPI int cs_cluster_logon
(
  int                   hdl,      // I: cluster handle
  const unsigned char   *p_user,  // I: user name
  unsigned int          l_user,   // I: length of user name (e.g. strlen(p_user) )
  const char            *p_key,   // I: smartcard specifier or key file (zero term. string)
  const unsigned char   *p_pwd,   // I: password
  unsigned int          l_pwd     // I: length of password (e.g. strlen(p_pwd) )
);

/**
 * @brief Create a new authenticated Secure Messaging session.
 *      Call this multiple times on the same cluster handle to obtain a session authenticated
 *      by multiple users.
 * @param[in, out] hdl Handle to a CryptoServer cluster as returned by \p cs_cluster_open_ex.
 *      The session info is stored as part of the referenced cluster.
 * @param[in] user Long name of user authenticating the new session.
 * @param[in] l_user Length of \p user.
 * @param[in] key Smartcard specifier or path to a key file. Must be a zero-terminated string
 *      or NULL (if the user's authentication token is simply a password).
 * @param[in] pwd User password, password of key file or PIN of smartcard. May be NULL for public
 *      key users. In this case, you will be prompted for password entry if required.
 * @param[in] l_pwd Length of \p pwd.
 * @param[in, out] cert_info Set the root certificates for the chains you wish to verify. If the
 *      field l_root of a certificate is > 0, the res field will be set to the result of the
 *      chain verification. All chains will be checked, regardless of the results of earlier chains.
 *      Set to NULL if you don't care about certificate chains.
 * @return 0 on success, error code otherwise.
 */
CSXAPI int cs_cluster_logon_cert(
    int hdl,
    const unsigned char *user,
    unsigned int l_user,
    const char *key,
    const unsigned char *pwd,
    unsigned int l_pwd,
    struct handshake_cert_info *cert_info
    );

CSXAPI int cs_cluster_logoff
(
  int h_cst                       // I: cluster handle
);

CSXAPI int cs_cluster_exec
(
  int           hdl,              // I: cluster handle
  unsigned int  fc,               // I: function code (module id)
  unsigned int  sfc,              // I: sub function code
  const unsigned char *p_cmd,     // I: pointer to command data
  unsigned int  l_cmd,            // I: length of command data
  unsigned char **pp_answ,        // O: pointer to answer data (memory allocation in error-free case)
  unsigned int  *p_l_answ         // O: pointer to answer length
);

typedef struct
{
  unsigned int mask;
  unsigned int code;
}
CLUSTER_ERROR;

CSXAPI int cs_cluster_exec_ex
(
  int           hdl,              // I: cluster handle
  unsigned int  fc,               // I: function code (module id)
  unsigned int  sfc,              // I: sub function code
  const CLUSTER_ERROR *errlist,   // I: list of error codes the cluster should not reply to the API but switch to another device
  unsigned int  err_ct,           // I: dimension of error list
  const unsigned char *p_cmd,     // I: pointer to command data
  unsigned int  l_cmd,            // I: length of command data
  unsigned char **pp_answ,        // O: pointer to answer data (memory allocation in error-free case)
  unsigned int  *p_l_answ         // O: pointer to answer length
);

CSXAPI int cs_cluster_set_timeout
(
  int           hdl,              // I: cluster handle
  unsigned int  read_timeout      // I: read (command) timeout [ms]
);

CSXAPI int cs_cluster_keep_session_alive
(
  int           hdl               // I: cluster handle
);

CSXAPI int cs_cluster_get_auth_state
(
  int           hdl,              // I  cluster handle
  int           *p_state          // O  state mask (xxxxxxxx)
);

CSXAPI int cs_cluster_set_fallback_interval
(
  int           hdl,              // I: cluster handle
  unsigned int  interval          // I: fallback interval [sec]
);

CSXAPI const char * cs_cluster_get_current_device
(
  int           hdl               // I: cluster handle
);

// utility function for SDK customers
CSXAPI void cs_xtrace
(
  char          *txt,
  const void    *data, 
  int           len
);

#ifdef __cplusplus
  }
#endif

#endif

