package com.utimaco.aut;

import java.io.IOException;
import java.nio.ByteBuffer;

import com.utimaco.SimpleArgs;
import com.utimaco.cs2.mdl.ErrList;
import com.utimaco.cs2.mdl.SerializationError;
import com.utimaco.cs2.mdl.Utils;
import com.utimaco.cs2.mdl.any.CxiKeyAttributes;
import com.utimaco.cs2.mdl.pqmi.MLDSA_KeyGen;
import com.utimaco.cs2.mdl.pqmi.MLDSA_Sign;
import com.utimaco.cs2.mdl.pqmi.MLDSA_Verify;
import com.utimaco.cs2.mdl.pqmi.PqmiKeyStore;
import com.utimaco.cs2.mdl.pqmi.Pqmi;

import CryptoServerAPI.CryptoServerException;
import CryptoServerCXI.CryptoServerCXI;

public class MLDSATest implements AutTest {
	// All external access to the QuantumProtect libraries are via the "Pqmi" module
	static int module_id = 0xa6; // Pqmi

	// Each module will have zero or more sub-functions that can be called, identified by
	// their "sub-function code" (SFC, an integer).  This class calls a key generation SFC, a 
	// data signing SFC and a signature verifying SFC.
	//
	// Both the Round 3 (lattice) module and the FIPS 203/204 (ml) modules use the same
	// SFC entry points.  Selection of which module depends on the key type information
	static final int sfc_id_gen_dil = Pqmi.SFC_DILITHIUM_KEYGEN; 
	static final int sfc_id_sign_dil = Pqmi.SFC_DILITHIUM_SIGN; 
	static final int sfc_id_verf_dil = Pqmi.SFC_DILITHIUM_VERIFY; 

	// It also makes use of the keystore SFC, for accessing keys stored on the HSM.
	static final int sfc_id_keystore = Pqmi.SFC_KEYSTORE; 

	CryptoServerCXI cxi;
	SimpleArgs args;
	boolean ctors;
	int rnd_touse; // Random-number generator to use (0 - Real, 1 - Pseudo)
	int sto_touse; // Key-store to use (0 - internal, 1 - external)
	// Internal keystore can also benefit from the "overwrite" flag see below.
	boolean asynch; // RFU to be implemented later
	boolean genkeyonly; // only generate an internal key so other things might be tested

	public MLDSATest(CryptoServerCXI cxi, SimpleArgs cla) {
		this(cxi, cla, false, false);
	}
	public MLDSATest(CryptoServerCXI cxi, SimpleArgs cla, boolean asynch) {
		this(cxi, cla, asynch, false);
	}
	public MLDSATest(CryptoServerCXI cxi, SimpleArgs cla, boolean asynch, boolean genkey) {
		this.cxi = cxi;
		this.args = cla;
		this.ctors = false;
		// command line: use -drbg or -trng to select the Real RNG, without we use the Pseudo-RNG.
		this.rnd_touse = cla.hasArg("drbg:trng") ? Pqmi.MODE_REAL_RND : Pqmi.MODE_PSEUDO_RND;
		// command line: use -ksint or -intks or -internal to use an internal key-store, else external.
		this.sto_touse = cla.hasArg("ksint:intks:internal") ? 0 : Pqmi.KEY_EXTERNAL;
		// command line: If -ksint, optional -overwrite to overwrite any stored key
		this.sto_touse |= cla.hasArg("overwrite") ? Pqmi.KEY_OVERWRITE : 0;
		this.asynch = asynch;
		this.genkeyonly = genkey;
		if (this.genkeyonly) {
			this.sto_touse = Pqmi.KEY_OVERWRITE;
		}

		module_id = AutTest.getModuleId(cxi, "PQMI");

	}

	public boolean go() {
		boolean ret = true;
		boolean ret2 = false;

		// command line: If -suite, will run all options and all key sizes, otherwise only the
		// key size supplied on the command line, as well as the command-line provided 
		// storage and RNG flags
		if (!args.hasArg("suite")) {
			// command line: Use -keytype [2|3|5] to select for 4x4, 6x5 or 8x7 key sizes.
			int keytype = args.getArg("keytype", 2);
			int kt = AutTest.getMLDSAKeytype(keytype);
			announceMLDSA(rnd_touse, sto_touse, kt);

			ret = gensignverify(false, rnd_touse, sto_touse, kt, 0);
			AutTest.announce(ret ? "Complete\n" : "Failed\n");
		}
		else {
			int [] keytypes = {2, 3, 5};
			for (int kt = 0; kt < keytypes.length; kt++) {
				int keytype = keytypes[kt] | Pqmi.ML_DSA_KT_FLAG;

				AutTest.announce(String.format("MLDSA-PSEU-EXT-%d-%d", keytype, kt));

				ret2 = gensignverify(false, Pqmi.MODE_PSEUDO_RND, Pqmi.KEY_EXTERNAL, keytype, 0);
				AutTest.announce(ret2 ? "Complete\n" : "Failed\n");
				if (!ret2) { ret = false; continue; }

				if (args.hasArg("full")) {
					AutTest.announce(String.format("MLDSA-DRBG-EXT-%d-%d", keytype, kt));
					ret2 = gensignverify(false, Pqmi.MODE_REAL_RND, Pqmi.KEY_EXTERNAL, keytype, 0);
					AutTest.announce(ret2 ? "Complete\n" : "Failed\n");
					if (!ret2) ret = false;

					AutTest.announce(String.format("MLDSA-PSEU-INT-%d-%d", keytype, kt));
					ret2 = gensignverify(false, Pqmi.MODE_PSEUDO_RND, Pqmi.KEY_OVERWRITE, keytype, 0);
					AutTest.announce(ret2 ? "Complete\n" : "Failed\n");
					if (!ret2) ret = false;

					AutTest.announce(String.format("MLDSA-DRBG-INT-%d-%d", keytype, kt));
					ret2 = gensignverify(false, Pqmi.MODE_REAL_RND, Pqmi.KEY_OVERWRITE, keytype, 0);
					AutTest.announce(ret2 ? "Complete\n" : "Failed\n");
					if (!ret2) ret = false;
				}
			}
		}
		return true;
	}

	public static void announceMLDSA(int rng, int keystore, int keytype) {
		int kt = AutTest.getMLDSAKeytype(keytype);
		String store = AutTest.storageType(keystore);
		AutTest.announce(String.format("MLDSA-%s-EXT-%d", store, kt));
	}

	/**
	 * 
	 * @param args
	 * @param cxi
	 * @param ctors
	 * @return
	 */
	private boolean gensignverify (boolean ctors, int rnd, int sto, int keytype, int spec) { 
		boolean ret = true;

		// This is the "HSM-aware POJO" Typed Interface used to generate a key.
		MLDSA_KeyGen tobj_ctor = null;

		// There are two constructors below, one uses the POJO's full constructor (true) or just calls setters on the empty
		// object above.
		boolean use_ctor = ctors;

		// One of the fields of the generator object is a "packed" CXI Key Attributes template.

		// This is the template, it must be serialized for use, however the DilithiumGen POJO setter
		// can take the object, and will call the passed-in object's serializer to get the serialized
		// buffer.  
		// NOTE: This happens on the call, subsequent changes to the attributes template
		// have NO affect on anything that serialized it previously.
		CxiKeyAttributes attributes = new CxiKeyAttributes(); 

		// CxiKeyAttributes - typed field attributes
		int flags = 0x0;
		// CXI stores its keys, indexed by the MD5 of the value {NAME||GROUP||SPEC} (|| here means 
		// 'concatenated with', so NAME concatenated with GROUP etc).
		byte [] name = "dil".getBytes();
		byte [] group = "SLOT_0000".getBytes();

		int algo = CryptoServerCXI.KEY_ALGO_RAW;
		int usage = CryptoServerCXI.KEY_USAGE_SIGN | CryptoServerCXI.KEY_USAGE_VERIFY;
		int export = 0; // ALLOW_BACKUP only, no export wrapped

		if (use_ctor) {
			// Using the constructor with parameters
			attributes = new CxiKeyAttributes (
					flags,
					name,
					group,
					spec,
					algo,
					usage,
					export, 
					new byte[0]
					);
		} else {
			// Using setters to set values after the call
			attributes = new CxiKeyAttributes ();
			attributes.flags (flags);
			attributes.name (name);
			attributes.group (group);
			attributes.spec (spec);
			attributes.algo (algo);
			attributes.usage (usage);
			attributes.export (export);
		}

		// In most versions of Dilithium, seed is ignored.  In some, seed is used as a seed into 
		// the HSM's random number generator. In CAVP automated tests, it's used as the actual seed
		// into the hashing algorithm used to elaborate the key -- ie it results in a deterministic
		// public/private key pair.
		// 
		// This version of Dilithium (ML) ignores this value.
		byte [] seed = new byte[0];

		// rnd -- whether use the DRBG TRNG or use the PRNG (only)
		//     -- Pqmi.MODE_REAL_RND or Pqmi.MODE_PSEUDO_RND
		// sto -- whether to store the key internal on the HSM or return a key blob
		//     -- if there is already an internal key with this {NAME||GROUP||Spec}, KEY_OVERWRITE
		//     -- will cause it to be overwritten.
		int outerflags = rnd | sto | (sto == Pqmi.KEY_EXTERNAL ? 0 : Pqmi.KEY_OVERWRITE);

		System.out.format("Keytype is %d (%x), flags is 0x%08x\n", keytype, keytype, outerflags);

		try {
			if (use_ctor) {
				tobj_ctor = new MLDSA_KeyGen (
						outerflags,     // the rnd/sto flags above
						keytype,		// One of the MLDSA keytypes, ie Pqmi.ML_DSA_KT_44 65 or 87
						attributes,     // The attributes object from above
						seed            // ignored
						);
			} else {
				tobj_ctor = new MLDSA_KeyGen (cxi);
				tobj_ctor.flags (outerflags);
				tobj_ctor.type (keytype);
				tobj_ctor.attributes (attributes);
				tobj_ctor.seed (seed);
			}
		} catch (SerializationError ser) {
			System.out.println("Serialization error on typed parameter in DilithiumGen.\n");
			return false;
		}

		/* ************************************* */
		if (ret) { 
			System.out.print("GEN STAGE:     ");

			ret = false;
			try {
				// Each of the "HSM-Aware POJO" classes can be "executed".  When one of the class 'exec' methods
				// are called, the class instance will serialize itself, and then call the HSM using the cxi 
				// instance in the exec call (other exec calls use a stored cxi).  In it's simplest form,
				// with predefined module_id and sub-function code ID, and a stored CXI, the class supports
				// foo.exec(), but we use a passed-in cxi, mid and sfc id here for clarity.
				//
				// To use foo.exec() does require modifying the underlying static values stored in the class.

				boolean res = tobj_ctor.exec(cxi, module_id, sfc_id_gen_dil);

				System.out.println(res ? "PASSED" : "FAILED"); 

				// When the call succeeds, any result byte-buffer is placed into the object's .resp field.
				// A suitable "HSM-Aware POJO" class can deserialize these responses into a class 
				// instance, when the system is set up that way.  

				// If you selected an external key, the .resp is the key blob (KB).  For an internal key,
				// it's a key handle (KH)
				ret = true;
			} catch (CryptoServerException e) {
				System.err.format("Error 0x%08X\n", e.ErrorCode);
				if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
					System.err.print(e.getLocalizedMessage());
				} else {
					System.err.println(ErrList.errtext(e.ErrorCode));
				}
				return false;
			} catch (Exception e) {
				e.printStackTrace();
				return false;
			}
		}
		if (!ret) return false;

		if (genkeyonly) {
			return true;
		}

		// Demonstration of how to request the public key part, given a Key Handle (KH)
		PqmiKeyStore key = null;
		if (sto != Pqmi.KEY_EXTERNAL) {
			// test getkey\
			if (ret) { 
				System.out.print("FIND STAGE:    ");

				try {
					// If you are familiar with PKCS11, this is like C_FindObjectsInit/FindObjects/FindObjectsFinal
					// Note we use the CxiKeyAttributes object from above as the key locator.
					key = new PqmiKeyStore(Pqmi.KEY_LOAD_FROM_STORE, keytype, attributes);
					boolean res = key.exec(cxi, module_id, sfc_id_keystore);
					System.out.println(res ? "PASSED" : "FAILED"); 
					ret = true;
				} catch (CryptoServerException e) {
					System.err.format("Error 0x%08X\n", e.ErrorCode);
					if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
						System.err.print(e.getLocalizedMessage());
					} else {
						System.err.println(ErrList.errtext(e.ErrorCode));
					}
					return false;
				} catch (Exception e) {
					e.printStackTrace();
					return false;
				}
			}
		} 
		if (!ret) return false;

		// we're doing both here but in reality you'd have these in different 
		// code paths
		MLDSA_Sign tobj_sig = new MLDSA_Sign();
		MLDSA_Verify tobj_verify = new MLDSA_Verify();

		int s_flags = Pqmi.DILITHIUM_MODE_SIG_RAW | Pqmi.MODE_PSEUDO_RND;
		if (args.hasArg("-ph")) {
			// pre-hash is not supported in this release.
			// s_flags |= MLDSA_Sign.FLAG_PRE_HASH;
			System.out.println("MLDSA Prehash is not supported in this release.\n");
		}
		if (sto == Pqmi.KEY_EXTERNAL) { 	
			// if it was an external key blob, the signing key is contained in the Gen object's .resp field.
			// Both the KB and the KH representations are of both the public and private keys, so use the
			// same value for the verify step.
			tobj_sig.key(tobj_ctor.resp);
			tobj_verify.key(tobj_ctor.resp);
			// You can also use the PqmiKeyStore, with the Pqmi.KEY_GET_PUBLIC_KEY flag,
			// to retrieve just the public key part for use in the verify object.
		} else {
			// it's an internal key, so we retrieved it above.
			// Both the KB and the KH representations are of both the public and private keys, so use the
			// same value for the verify step.
			tobj_sig.key(key.resp);
			tobj_verify.key(tobj_ctor.resp);
			// You can also use the PqmiKeyStore, with the Pqmi.KEY_GET_PUBLIC_KEY flag,
			// to retrieve just the public key part for use in the verify object.
		}

		tobj_sig.msg("Test Message");
		if (args.hasArg("-msg")) {
			// command line, use a fixed message found by -msg <message>
			tobj_sig.msg(args.getArg("-msg", tobj_sig.msg()));
		}
		if (args.hasArg("-ctxt")) {
			// If you are supplying a context, use -ctxt <context> (limited to 255 bytes) 
			tobj_sig.ctxt(args.getArg("-ctxt", ""));
			tobj_verify.ctxt(args.getArg("-ctxt", ""));
			s_flags |= MLDSA_Sign.FLAG_HAS_CTXT;
		}
		if (args.hasArg("-obj")) {
			// alternately, you can provide an -obj <filename> instead of a -msg
			String fname = args.getArg("-obj", (String)null);
			if (fname != null) {
				byte[] data = null;
				try {
					data = Utils.readFileAsBytes(fname);
					if (data.length > 250*1024) {
						System.err.format("No can do: Limit is one 256Kb (with overhead) size with this release.\n");
						return false;
					}
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
				if (data == null) { data = "Test Message".getBytes(); }
				tobj_sig.msg(data);
			}
		}
		tobj_verify.msg(tobj_sig.msg());

		tobj_sig.type(tobj_ctor.type());  // just grabbing the keytype out of the Gen object.
		tobj_sig.flags(s_flags);

		tobj_verify.type(tobj_ctor.type());  // just grabbing the keytype out of the Gen object.
		tobj_verify.flags(s_flags);

		if (ret) {
			System.out.print("SIGN STAGE:    ");

			try {
				boolean res = tobj_sig.exec(cxi, module_id, sfc_id_sign_dil);
				System.out.println(res ? "PASSED" : "FAILED"); 
				// The sv object's .resp is the signature.
				ret = true;
			} catch (CryptoServerException e) {
				System.err.format("Error 0x%08X\n", e.ErrorCode);
				if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
					System.err.print(e.getLocalizedMessage());
				} else {
					System.err.println(ErrList.errtext(e.ErrorCode));
				}
				return false;
			} catch (Exception e) {
				e.printStackTrace();
				return false;
			}
		}
		tobj_verify.sig(tobj_sig.resp); // this field (resp) contains the signature.

		if (ret) {
			System.out.print("VERIFY STAGE:  ");

			ret = false;
			try {
				boolean res = tobj_verify.exec(cxi, module_id, sfc_id_verf_dil);
				System.out.println(res ? "PASSED" : "FAILED"); 
				ret = true;
			} catch (CryptoServerException e) {
				System.err.format("Error 0x%08X\n", e.ErrorCode);
				if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
					System.err.print(e.getLocalizedMessage());
				} else {
					System.err.println(ErrList.errtext(e.ErrorCode));
				}
				return false;
			} catch (Exception e) {
				e.printStackTrace();
				return false;
			}
		}


		return ret;
	}

	private boolean bulk_gensignverify (boolean ctors, int rnd, int sto, int keytype, int spec) { 
		boolean ret = true;

		// for notes and commentary, see gensignverify, above.
		MLDSA_KeyGen tobj_ctor = null;
		boolean use_ctor = ctors;
		CxiKeyAttributes attributes = new CxiKeyAttributes(); 

		// CxiKeyAttributes - typed field attributes
		int flags = 0x0;
		byte [] name = "dil_bulk".getBytes();
		byte [] group = "testseries".getBytes();

		int algo = CryptoServerCXI.KEY_ALGO_RAW;
		int usage = CryptoServerCXI.KEY_USAGE_SIGN | CryptoServerCXI.KEY_USAGE_VERIFY;
		int export = 0; // ALLOW_BACKUP only, no export wrapped

		if (use_ctor) {
			// Using the constructor with parameters
			attributes = new CxiKeyAttributes (flags, name, group, spec, algo, usage, export, new byte[0]);
		} else {
			// Using setters to set values after the call
			attributes = new CxiKeyAttributes ();
			attributes.flags (flags);
			attributes.name (name);
			attributes.group (group);
			attributes.spec (spec);
			attributes.algo (algo);
			attributes.usage (usage);
			attributes.export (export);
		}

		byte [] seed = new byte[0];
		int outerflags = rnd | sto | (sto == Pqmi.KEY_EXTERNAL ? 0 : Pqmi.KEY_OVERWRITE);

		System.out.format("Keytype is %d (%x), flags is 0x%08x\n", keytype, keytype, outerflags);

		try {
			if (use_ctor) {
				tobj_ctor = new MLDSA_KeyGen (outerflags, keytype, attributes, seed);
			} else {
				tobj_ctor = new MLDSA_KeyGen (cxi);
				tobj_ctor.flags (outerflags);
				tobj_ctor.type (keytype);
				tobj_ctor.attributes (attributes);
				tobj_ctor.seed (seed);
			}
		} catch (SerializationError ser) {
			System.out.println("Serialization error on typed parameter in DilithiumGen.\n");
			return false;
		}

		/* ************************************* */
		if (ret) { 
			System.out.print("GEN STAGE:     \n");

			ret = false;
			try {
				tobj_ctor.exec(cxi, module_id, sfc_id_gen_dil); // passes or throws
				ret = true;
			} catch (CryptoServerException e) {
				System.err.format("Error 0x%08X\n", e.ErrorCode);
				if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
					System.err.print(e.getLocalizedMessage());
				} else {
					System.err.println(ErrList.errtext(e.ErrorCode));
				}
				return false;
			} catch (Exception e) {
				e.printStackTrace();
				return false;
			}
		}
		if (!ret) return false;

		PqmiKeyStore key = null;
		if (sto != Pqmi.KEY_EXTERNAL) {
			if (ret) { 
				System.out.print("FIND STAGE:    \n");

				try {
					key = new PqmiKeyStore(Pqmi.KEY_LOAD_FROM_STORE, keytype, attributes);
					key.exec(cxi, module_id, sfc_id_keystore);
					ret = true;
				} catch (CryptoServerException e) {
					System.err.format("Error 0x%08X\n", e.ErrorCode);
					if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
						System.err.print(e.getLocalizedMessage());
					} else {
						System.err.println(ErrList.errtext(e.ErrorCode));
					}
					return false;
				} catch (Exception e) {
					e.printStackTrace();
					return false;
				}
			}
		} 
		if (!ret) return false;

		MLDSA_Sign tobj_sig = new MLDSA_Sign();
		MLDSA_Verify tobj_verify = new MLDSA_Verify();

		int s_flags = Pqmi.DILITHIUM_MODE_SIG_RAW | Pqmi.MODE_PSEUDO_RND;
		int s_flags_init = s_flags | CryptoServerCXI.FLAG_CRYPT_INIT;
		int s_flags_final = s_flags | CryptoServerCXI.FLAG_CRYPT_FINAL;
		if (args.hasArg("-ph")) {
			// pre-hash is not supported in this release.
			// s_flags |= MLDSA_Sign.FLAG_PRE_HASH;
			System.out.println("MLDSA Prehash is not supported in this release.\n");
		}
		if (sto == Pqmi.KEY_EXTERNAL) { 	
			tobj_sig.key(tobj_ctor.resp);
			tobj_verify.key(tobj_ctor.resp);
		} else {
			tobj_sig.key(key.resp);
			tobj_verify.key(tobj_ctor.resp);
		}

		tobj_sig.msg("Test Message");
		if (args.hasArg("-msg")) {
			// command line, use a fixed message found by -msg <message>
			tobj_sig.msg(args.getArg("-msg", tobj_sig.msg()));
		}
		if (args.hasArg("-ctxt")) {
			// If you are supplying a context, use -ctxt <context> (limited to 255 bytes) 
			tobj_sig.ctxt(args.getArg("-ctxt", ""));
			s_flags_init |= MLDSA_Sign.FLAG_HAS_CTXT;
		}
		byte [] data = null;
		if (args.hasArg("-obj")) {
			// alternately, you can provide an -obj <filename> instead of a -msg
			String fname = args.getArg("-obj", (String)null);
			if (fname != null) {
				try {
					data = Utils.readFileAsBytes(fname);
					if (data.length < 250*1024) {
						System.err.format("No can do: For this test, file should be > 250Kb.\n");
						return false;
					}
				} catch (IOException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
				if (data == null) {
					System.err.format("No can do: For this test, -obj <file> should be > 250Kb.\n");
					return false;
				}
			}
		}

		tobj_sig.type(tobj_ctor.type());  // just grabbing the keytype out of the Gen object.
		tobj_sig.flags(s_flags_init);

		tobj_verify.type(tobj_ctor.type());  // just grabbing the keytype out of the Gen object.
		tobj_verify.flags(s_flags_init);

		if (ret) {
			System.out.print("SIGN STAGE:    (chunking)\n");
			ByteBuffer bb = ByteBuffer.wrap(data);
			int max_offset = 1024*254; // leave some room for overhead
			do {
				int chunk_sz = Math.min(max_offset, bb.remaining());
				if (chunk_sz < max_offset) {
					tobj_sig.flags(s_flags_final);
				}
				byte [] msg = new byte[chunk_sz];
				bb.get(msg);
				tobj_sig.msg(msg);
				try {
					tobj_sig.exec(cxi, module_id, sfc_id_sign_dil);
				} catch (CryptoServerException e) {
					System.err.format("Error 0x%08X\n", e.ErrorCode);
					if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
						System.err.print(e.getLocalizedMessage());
					} else {
						System.err.println(ErrList.errtext(e.ErrorCode));
					}
					return false;
				} catch (Exception e) {
					e.printStackTrace();
					return false;
				}
				// tobj_sig.state() holds the intermediate state returned by the call
				if (bb.remaining() > 0) {
					tobj_sig.state(tobj_sig.resp);
					tobj_sig.flags(s_flags);
					tobj_sig.ctxt((byte[])null);  
				}
			} while (bb.remaining() > 0);
		}

		// The signature is in tobj_sig.resp
		tobj_verify.sig(tobj_sig.resp); // this field (resp) contains the signature.

		if (ret) {
			System.out.print("VERIFY STAGE:    (chunking)\n");
			ByteBuffer bb = ByteBuffer.wrap(data);
			int max_offset = 1024*254; // leave some room for overhead
			do {
				int chunk_sz = Math.min(max_offset, bb.remaining());
				if (chunk_sz < max_offset) {
					tobj_verify.flags(s_flags_final);
				}
				byte [] msg = new byte[chunk_sz];
				bb.get(msg);
				tobj_verify.msg(msg);
				try {
					tobj_verify.exec(cxi, module_id, sfc_id_verf_dil);
				} catch (CryptoServerException e) {
					System.err.format("Error 0x%08X\n", e.ErrorCode);
					if ((e.ErrorCode & (int)0xBFFF0000) < 0xB1000000) {
						System.err.print(e.getLocalizedMessage());
					} else {
						System.err.println(ErrList.errtext(e.ErrorCode));
					}
					return false;
				} catch (Exception e) {
					e.printStackTrace();
					return false;
				}
				// tobj_sig.state() holds the intermediate state returned by the call
				if (bb.remaining() > 0) {
					tobj_verify.state(tobj_verify.resp);
					tobj_verify.flags(s_flags);
					tobj_verify.ctxt((byte[])null);  
				}
			} while (bb.remaining() > 0);
		}


		return ret;
	}

}
