#include "qptool2.h"
#include "hbs_defs.h"

static CK_BYTE bTrue = 1;
static CK_BYTE bFalse = 0;

#define HBS_MAX_HSS_LEVELS				8
#define HBS_RNG_TYPE_PSEUDO				0
#define HBS_RNG_TYPE_REAL				1

unsigned int hbs_mechanism_len_by_levels(unsigned int levels) {
	unsigned int i = 1; // RNG type
	i += 1; // where levels will be written
	i += (levels * 2); // how many levels, 2 bytes per
	i += 2; // short for the aux size
	return i;
}

#ifndef min
#define min(a, b) ((a) < (b) ? (a) : (b))
#endif

// =============================================================================================
// Page 16 - 4.1.1 HBS_MECH_LMS_GENKEY 

// EXAMPLE

int hbs_int_gen_key(CK_FUNCTION_LIST_PTR pFunctions, CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE_PTR p_keyHandle, unsigned int asToken)
{
	int					err = 0;

	CK_UTF8CHAR_PTR 	keyLabel = (CK_UTF8CHAR_PTR)"lms_key";
	CK_ULONG            l_keyLabel = (CK_ULONG)strlen((char *)keyLabel);
	CK_BBOOL 			token = (asToken ? 1 : 0);

	CK_OBJECT_CLASS 	keyClass = CKO_SECRET_KEY;
	CK_KEY_TYPE 		keyType = CKK_GENERIC_SECRET;
	CK_ULONG 			byteLength = 32; // not important, required by C_Generate VDM

	CK_MECHANISM 		mechanism;
	CK_BYTE				mechParams[20]; // max mechanism parameter length for 8 levels 1+1+(2*8)+2 = 20 

	unsigned short auxsize = 10916; // per the doc

	CK_ATTRIBUTE 		keyTemplate[] =
	{
			{ CKA_CLASS, 		&keyClass, 		sizeof(keyClass) },
			{ CKA_KEY_TYPE, 	&keyType, 		sizeof(keyType) },
			{ CKA_TOKEN, 		&token, 		sizeof(token) },
			{ CKA_LABEL, 		keyLabel, 		l_keyLabel },
			{ CKA_DERIVE, 		&bTrue, 		sizeof(bTrue) },
			{ CKA_VALUE_LEN, 	&byteLength, 	sizeof(byteLength) },
	};

	// Prepare mechanism parameters
	unsigned char *pmp = mechParams;
	memcpy(pmp, "->", 2); pmp += 2;
	// 1 byte - type of random number generator
	*pmp = HBS_RNG_TYPE_PSEUDO;
	++pmp;
	// 1 byte - LMS/OTS levels
	*pmp = 1;
	++pmp;
	//		1 byte - LMS type
	*pmp = LMS_SHA256_N32_H10;
	++pmp; 
	//		1 byte - OTS type
	*pmp = LMOTS_SHA256_N32_W4;
	++pmp;
	//2 bytes - optional, size of auxilary data (1 - 16k)
	*pmp = ((auxsize & 0xFF00) >> 8);
	++pmp;
	*pmp =  (auxsize & 0x00FF);
	++pmp;
	/*
	store_int1((unsigned char)HBS_RNG_TYPE_PSEUDO, pmp); pmp += 1;
	store_int1(1, pmp); pmp += 1;
	store_int1((unsigned char)LMS_SHA256_N32_H10, pmp); pmp += 1;
	store_int1((unsigned char)LMOTS_SHA256_N32_W4, pmp); pmp += 1;
	store_int2(auxsize, pmp); pmp += 2; // aux size
	*/
	memcpy(pmp, "<-", 2);

	mechanism.mechanism = HBS_MECH_LMS_GENKEY;
	mechanism.pParameter = mechParams + 2;
	mechanism.ulParameterLen = hbs_mechanism_len_by_levels(1);
	
	entry("hbs_int_gen_key:C_GenerateKey");
	err = pFunctions->C_GenerateKey(hSession, 
		&mechanism,
		keyTemplate,
		sizeof(keyTemplate) / sizeof(CK_ATTRIBUTE),
		p_keyHandle);
	exuent("hbs_int_gen_key:C_GenerateKey", err);
	if (err != 0x0) {
		printf("[genkey]: C_GenerateKey returned 0x%08x\n", err);
		goto cleanup;
	}

cleanup:
	return err;
}


// =============================================================================================
// Page 17 - 4.1.2 HBS_MECH_LMS_GET_PUBKEY 

// EXAMPLE

int hbs_int_get_pubkey(
	CK_FUNCTION_LIST_PTR pFunctions, 
	CK_SESSION_HANDLE hSession, 
	CK_OBJECT_HANDLE baseKey, 
	CK_OBJECT_HANDLE_PTR p_pubHandle)
{
	int 				err = 0;
	CK_OBJECT_CLASS 	keyClass = CKO_SECRET_KEY;
	CK_OBJECT_CLASS     pubclass = CKO_PUBLIC_KEY;
	CK_ULONG            keytype = CKO_VENDOR_DEFINED;

	// any symetric key type, only temporary
	CK_KEY_TYPE 		keyType = CKK_AES;
	CK_ULONG 			byteLength = 256 / 8;
	CK_UTF8CHAR_PTR 	keyLabel = (CK_UTF8CHAR_PTR)"tmp_lms_key";
	CK_ULONG            l_keyLabel = (CK_ULONG)strlen((char *)keyLabel);
	CK_ATTRIBUTE 		keyTemplate[] =
	{
			{ CKA_CLASS, 		&keyClass, 		sizeof(keyClass) },
			{ CKA_KEY_TYPE, 	&keyType, 		sizeof(keyType) },
			{ CKA_TOKEN, 		&bFalse, 		sizeof(bFalse) },
			{ CKA_LABEL, 		keyLabel, 		l_keyLabel },
			{ CKA_VALUE_LEN, 	&byteLength, 	sizeof(byteLength) }
	};

	CK_ULONG 			l_keyTemplate = sizeof(keyTemplate) / sizeof(CK_ATTRIBUTE);
	CK_MECHANISM 		mechanism;

	// There are two concepts on "retrieve the public key".
	// First, "we are getting the public key so that we can verify the thing"
	// Second, "we are getting the public key, so that we can give it to a trading partner to verify the thing"
	
	// Get the public data, which is the information needed to validate the signature internally
	CK_BYTE 			p_pubData[21504]; // Maximum size of data 
	CK_ULONG 			l_pubData = 21504;
	CK_ATTRIBUTE 		pubDataFetch[] =
	{
		{ CKA_UTI_CUSTOM_DATA, p_pubData, l_pubData }
	};

	CK_ULONG 			l_pubDataFetch = sizeof(pubDataFetch) / sizeof(CK_ATTRIBUTE);

	CK_ATTRIBUTE 		pubkeyTemplate[] =
	{
		{ CKA_CLASS,		&pubclass, sizeof(pubclass) },
		{ CKA_KEY_TYPE,		&keytype, sizeof(keytype) },
		{ CKA_TOKEN,		&bFalse, sizeof(bFalse) },
		{ CKA_VALUE,		p_pubData, 0},
		{ CKA_VERIFY,		&bTrue, sizeof(bTrue) },
	};
	CK_ULONG 			l_pubkeyTemplate = sizeof(pubkeyTemplate) / sizeof(CK_ATTRIBUTE);
	CK_ULONG ckavalue = 3; // offset into pubkeyTemplate

	mechanism.mechanism = HBS_MECH_LMS_GET_PUBKEY;
	mechanism.pParameter = NULL;
	mechanism.ulParameterLen = 0;

	// Create new key, value is random, but pubdata has public key
	entry("hbs_int_get_pubkey: C_DeriveKey");
	err = pFunctions->C_DeriveKey(hSession, &mechanism,
		baseKey,
		keyTemplate,
		l_keyTemplate,
		p_pubHandle);
	exuent("hbs_int_get_pubkey: C_DeriveKey", err);
	if (err != 0x0) {
		printf("[get_pub_key]: C_DeriveKey returned 0x%08x\n", err);
		goto cleanup;
	}

	// get pubData = public key
#ifdef INCLUDED_ONLY_TO_SHOW_HOW
	entry("hbs_int_get_pubkey: C_GetAttributeValue");
	err = pFunctions->C_GetAttributeValue(hSession,
		p_pubHandle,
		pubDataFetch,
		l_pubDataFetch);
	exuent("hbs_int_get_pubkey: C_GetAttributeValue", err);
	if (err) {
		printf("[get_pub_key]: C_GetAttributeValue returned 0x%08x\n", err);
		goto cleanup;
	}

	l_pubData = pubDataFetch[0].ulValueLen;
	
	if (l_pubData == CK_UNAVAILABLE_INFORMATION)
	{
		err = CKR_ATTRIBUTE_VALUE_INVALID;
		printf("[get_pub_key]: Couldn't extract attribute.\n");
		goto cleanup;
	}

	cs_xprint("Verifier information", pubDataFetch[0].pValue, 32);
	printf("\n");
#endif

cleanup:
	// nothing to clean up here
	return err;
}


// =============================================================================================
// Page 19 - 4.1.3 HBS_MECH_LMS_SIGN 


// ----------------------------------------------------
// EXAMPLE Single Data Block

int sign_all(
	CK_FUNCTION_LIST_PTR pFunctions,
	CK_SESSION_HANDLE hSession,
	CK_OBJECT_HANDLE 		hKey,
	CK_BYTE_PTR 			p_msg,
	CK_ULONG				l_msg,
	CK_BYTE_PTR				p_sig,
	CK_ULONG_PTR			ll_sig)
{
	int 					err = 0;
	unsigned char 			params[] = { 0 };
	CK_ULONG 				sig_len = (CK_ULONG)4096;

	if (*ll_sig < sig_len) {
		if (*ll_sig == 0) {
			*ll_sig = sig_len;
			return 0;
		}
		return CKR_BUFFER_TOO_SMALL;
	}
	if (l_msg > 250 * 1024) {
		return 0x111L; // ERROR_BUFFER_OVERFLOW;
	}

	printf("\nLength of message:  %d\n", l_msg);
	cs_xprint("S: Message (one-shot) (truncated)", p_msg, (l_msg > 16) ? 16 : l_msg);
	printf("\n");

	CK_MECHANISM 			mechanism;
	mechanism.mechanism = HBS_MECH_LMS_SIGN;

	mechanism.pParameter = params;
	mechanism.ulParameterLen = 0;

	entry("sign_all:C_SignInit");
	err = pFunctions->C_SignInit(hSession, &mechanism, hKey);
	exuent("sign_all:C_SignInit", err);
	if (err != CKR_OK) {
		printf("[sign_all]: C_SignInit returned 0x%08x\n", err);
		goto cleanup;
	}

	entry("sign_all:C_Sign");
	err = pFunctions->C_Sign(hSession, p_msg, l_msg, p_sig, ll_sig); 
	exuent("sign_all:C_Sign", err);
	if (err != CKR_OK) {
		printf("[sign_all]: C_Sign returned 0x%08x\n", err);
		goto cleanup;
	}
	
	cs_xprint("S: Signature (one-shot) (truncated)", p_sig, (*ll_sig > 64) ? 64 : *ll_sig);
	printf("\n");

cleanup:
	// nothing to clean up here
	return err;
}

int verify_all(
	CK_FUNCTION_LIST_PTR pFunctions,
		CK_SESSION_HANDLE hSession,
		CK_OBJECT_HANDLE 		hKey,
		CK_BYTE_PTR 			p_msg,
		CK_ULONG				l_msg,
		CK_BYTE_PTR				p_sig,
		CK_ULONG				l_sig)
{
	int 					err = 0;
	unsigned char 			params[] = { 0 };

	CK_MECHANISM 			mechanism;
	mechanism.mechanism = HBS_MECH_LMS_VERIFY;
	mechanism.pParameter = params;
	mechanism.ulParameterLen = 0;

	if (l_msg > 250 * 1024) {
		return 0x111L; // ERROR_BUFFER_OVERFLOW;
	}

	cs_xprint("V: Signature (one-shot) (truncated)", p_sig, (l_sig > 64) ? 64 : l_sig);
	printf("Length of message:  %d\n", l_msg);
	cs_xprint("V: Message (one-shot) (truncated)", p_msg, (l_msg > 16) ? 16 : l_msg);
	printf("\n");

	entry("verify_all:C_VerifyInit");
	err = pFunctions->C_VerifyInit(hSession, &mechanism, hKey);
	exuent("verify_all:C_VerifyInit", err);
	if (err != CKR_OK) {
		printf("[sign_all]: C_VerifyInit returned 0x%08x\n", err);
		goto cleanup;
	}

	entry("verify_all:C_Verify");
	err = pFunctions->C_Verify(hSession, p_msg, l_msg, p_sig, l_sig);
	exuent("verify_all:C_Verify", err);
	if (err != CKR_OK) {
		printf("[sign_all]: C_Verify returned 0x%08x\n", err);
		goto cleanup;
	}

	printf("Signature verified.\n");
	printf("\n");

cleanup:
	return err;
}

// ----------------------------------------------------------------------------
// EXAMPLE Multi Data Block (4.1.3 HBS_MECH_LMS_SIGN)

int sign_chunked(
	CK_FUNCTION_LIST_PTR pFunctions,
	CK_SESSION_HANDLE hSession,
	CK_OBJECT_HANDLE 		hKey,
	CK_BYTE_PTR 			p_msg,
	CK_ULONG				l_msg,
	CK_BYTE_PTR				p_sig,
	CK_ULONG_PTR			ll_sig)
{
	int 					err = 0;
	unsigned char 			params[] = { 0 };
	unsigned int            sig_len = 4096; 
	CK_ULONG sent = 0, chunk_size = 4096;

	CK_MECHANISM 			mechanism;

	if (*ll_sig < sig_len) {
		if (*ll_sig == 0) {
			*ll_sig = sig_len;
			return 0;
		}
		return CKR_BUFFER_TOO_SMALL;
	}

	printf("\nLength of message:  %d\n", l_msg);
	cs_xprint("V: Message (multipass) (truncated)", p_msg, (l_msg > 16) ? 16 : l_msg);
	printf("\n");

	mechanism.mechanism = HBS_MECH_LMS_SIGN;
	mechanism.pParameter = params;
	mechanism.ulParameterLen = 0;

	entry("sign_chunked:C_SignInit");
	err = pFunctions->C_SignInit(hSession, &mechanism, hKey);
	exuent("sign_chunked:C_SignInit", err);
	if (err != CKR_OK) {
		printf("[sign_chunked]: C_SignInit returned 0x%08x\n", err);
		goto cleanup;
	}

	while (sent < l_msg) {
		CK_ULONG len = min(chunk_size, l_msg - sent);
		entry("sign_chunked:C_SignUpdate");
		err = pFunctions->C_SignUpdate(hSession, p_msg + sent, len);
		exuent("sign_chunked:C_SignUpdate", err);
		if (err != CKR_OK) {
			printf("[sign_chunked]: C_SignUpdate returned 0x%08x\n", err);
			goto cleanup;
		}
		sent += len;
	}

	entry("sign_chunked:C_SignFinal");
	err = pFunctions->C_SignFinal(hSession, p_sig, ll_sig);
	exuent("sign_chunked:C_SignFinal", err);
	if (err != CKR_OK) {
		printf("[sign_chunked]: C_SignFinal returned 0x%08x\n", err);
		goto cleanup;
	}

	cs_xprint("S: Signature (multipass) (truncated)", p_sig, (*ll_sig > 64) ? 64 : *ll_sig);
	printf("\n");

cleanup:
	return err;
}

int verify_chunked(
	CK_FUNCTION_LIST_PTR pFunctions,
	CK_SESSION_HANDLE hSession,
	CK_OBJECT_HANDLE 		hKey,
	CK_BYTE_PTR 			p_msg,
	CK_ULONG				l_msg,
	CK_BYTE_PTR				p_sig,
	CK_ULONG				l_sig)
{
	int 					err = 0;
	CK_ULONG sent = 0, chunk_size = 4096;

	// SIGNATURE VERIFICATION
	CK_MECHANISM 			mechanism;

	mechanism.mechanism = HBS_MECH_LMS_VERIFY;
	mechanism.pParameter = NULL; // or p_sig? oasis doc has NULL/0, HBS doc has p_sig/l_sig
	mechanism.ulParameterLen = 0; // or l_sig?

	cs_xprint("V: Signature (multipass) (truncated)", p_sig, (l_sig > 64) ? 64 : l_sig);
	printf("Length of message:  %d\n", l_msg);
	cs_xprint("V: Message (multipass) (truncated)", p_msg, (l_msg > 16) ? 16 : l_sig);
	printf("\n");

	entry("verify_chunked:C_VerifyInit");
	err = pFunctions->C_VerifyInit(hSession, &mechanism, hKey);
	exuent("verify_chunked:C_VerifyInit", err);
	if (err != CKR_OK) {
		printf("[sign_chunked]: C_VerifyInit returned 0x%08x\n", err);
		goto cleanup;
	}

	while (sent < l_msg) {
		CK_ULONG len = min(chunk_size, l_msg - sent);
		entry("verify_chunked:C_VerifyUpdate");
		err = pFunctions->C_VerifyUpdate(hSession, p_msg + sent, len);
		exuent("verify_chunked:C_VerifyUpdate", err);
		if (err != CKR_OK) {
			printf("[sign_chunked]: C_VerifyUpdate returned 0x%08x\n", err);
			goto cleanup;
		}
		sent += len;
	}

	entry("verify_chunked:C_VerifyFinal");
	err = pFunctions->C_VerifyFinal(hSession, p_sig, l_sig);
	exuent("verify_chunked:C_VerifyFinal", err);
	if (err != CKR_OK) {
		printf("[sign_chunked]: C_VerifyFinal returned 0x%08x\n", err);
		goto cleanup;
	}

	printf("Signature verified.\n");
	printf("\n");

cleanup:
	return err;
}

/*

// =============================================================================================
// Page 21 -4.1.5 HBS_MECH_XMSS_GENKEY 
// EXAMPLE

int xmss_genkey(CK_OBJECT_HANDLE_PTR p_keyHandle)
{
	int 					err = 0;
	CK_UTF8CHAR_PTR 		keyLabel = "xmss_key";
	CK_BBOOL token = 0; 	// available only in this session
	CK_OBJECT_CLASS 		keyClass = CKO_SECRET_KEY;

	CK_KEY_TYPE 			keyType = CKK_GENERIC_SECRET;
	CK_ULONG 				byteLength = 32; // not important, required by interface

	CK_ATTRIBUTE keyTemplate[] =
	{
			{ CKA_CLASS, 		&keyClass, 		sizeof(keyClass) },
			{ CKA_KEY_TYPE, 	&keyType, 		sizeof(keyType) },
			{ CKA_TOKEN, 		&token, 		sizeof(token) },
			{ CKA_LABEL, 		keyLabel, 		(CK_ULONG)strlen(keyLabel) },
			{ CKA_DERIVE, 		&boolTrue, 		sizeof(boolTrue) },
			{ CKA_VALUE_LEN, 	&byteLength, 	sizeof(byteLength) },
	};
	CK_ULONG 		l_keyTemplate = 6;

	CK_MECHANISM 	mechanism;
	CK_BYTE 		mechParams[128];

	mechParams[0] = (unsigned char)HBS_RNG_TYPE_PSEUDO; // RNG type
	mechParams[1] = (unsigned char)0; // mt -> single tree
	mechParams[2] = (unsigned char)1; // oid -> XMSS-SHA2_10_256

	mechanism.mechanism = HBS_MECH_XMSS_GENKEY;
	mechanism.pParameter = mechParams;
	mechanism.ulParameterLen = 3;

	if ((err = pfunc_list->C_GenerateKey(session_id,
		&mechanism,
		keyTemplate,
		l_keyTemplate,
		p_keyHandle
	)) != CKR_OK)
	{
		printf("[xmss_genkey]: C_GenerateKey returned 0x%08x\n", err);
		goto cleanup;
	}

cleanup:
	return err;
}


////=============================================================================================
// Page 22 - 4.1.6 HBS_MECH_XMSS_GET_PUBKEY 
// EXAMPLE

int xmss_get_pubkey(CK_OBJECT_HANDLE baseKey)
{
	int 			err = 0;

	CK_OBJECT_CLASS 		keyClass = CKO_SECRET_KEY;
	// any symetric 		key type, only temporary
	CK_KEY_TYPE 			keyType = CKK_AES;
	CK_ULONG 				byteLength = 256 / 8;
	CK_UTF8CHAR_PTR 		keyLabel = "tmp_xmss_key";
	CK_ATTRIBUTE 			keyTemplate[] =
	{
			{ CKA_CLASS, 		&keyClass, 		sizeof(keyClass) },
			{ CKA_KEY_TYPE, 	&keyType, 		sizeof(keyType) },
			{ CKA_TOKEN, 		&boolFalse, 	sizeof(boolFalse) },
			{ CKA_LABEL, 		keyLabel, 		(CK_ULONG)strlen(keyLabel) },
			{ CKA_VALUE_LEN, 	&byteLength, 	sizeof(byteLength) }
	};

	CK_ULONG 			l_keyTemplate = 5;
	CK_MECHANISM 		mechanism;
	CK_OBJECT_HANDLE 	derivedKey;
	// extract 			pubData
	CK_BYTE 			p_pubData[128];
	CK_ULONG 			l_pubData = 128;
	CK_ATTRIBUTE 		pubDataFetch[] =
	{
			{ CKA_UTI_CUSTOM_DATA, p_pubData, l_pubData }
	};
	CK_ULONG l_pubDataFetch = 1;

	mechanism.mechanism = HBS_MECH_XMSS_GET_PUBKEY;
	mechanism.pParameter = NULL;
	mechanism.ulParameterLen = 0;

	// Create new key, value is random, but pubdata has public key
	if ((err = pfunc_list->C_DeriveKey(session_id,
		&mechanism,
		baseKey,
		keyTemplate,
		l_keyTemplate,
		&derivedKey)) != CKR_OK)
	{
		printf("[xmss_get_pubkey]: C_DeriveKey returned 0x%08x\n", err);
		goto cleanup;
	}

	// get pubData = public key

	if ((err = pfunc_list->C_GetAttributeValue(session_id,
		derivedKey,
		pubDataFetch,
		l_pubDataFetch
	)) != CKR_OK)
		goto cleanup;

	l_pubData = pubDataFetch[0].ulValueLen;

	if (l_pubData == CK_UNAVAILABLE_INFORMATION)
	{
		err = CKR_ATTRIBUTE_VALUE_INVALID;
		printf("[xmss_get_pubkey]: Couldn't extract attribute.\n");
		goto cleanup;
	}

	if (prnout) {
		printf("Public key:\n");
		bin2hex(p_pubData, l_pubData);
		printf("\n");
	}

cleanup:
	// nothing to clean up here
	return err;
}




//-------------------------------------------------------------
// EXAMPLE Single Data Block

int xmss_sign_all(
	CK_OBJECT_HANDLE 		hKey,
	CK_BYTE_PTR 			p_msg,
	CK_ULONG 				l_msg)
{
	int 					err = 0;
	unsigned char 			params[] = { 0 };
	unsigned char 			sig[4096];
	CK_ULONG 				sig_len = (CK_ULONG) sizeof(sig);

	CK_MECHANISM 			mechanism;
	mechanism.mechanism = HBS_MECH_XMSS_SIGN;

	mechanism.pParameter = params;
	mechanism.ulParameterLen = 0;

	if ((err = pfunc_list->C_SignInit(session_id, &mechanism, hKey)) != CKR_OK) {
		printf("[xmss_sign_all]: C_SignInit returned 0x%08x\n", err);
		goto cleanup;
	}

	if ((err = pfunc_list->C_Sign(session_id, p_msg, l_msg, sig, &sig_len)) != CKR_OK) {
		printf("[xmss_sign_all]: C_Sign returned 0x%08x\n", err);
		goto cleanup;
	}

	if (prnout) {
		printf("Signature:\n");
		bin2hex(sig, sig_len);
		printf("\n");
	}

	// SIGNATURE VERIFICATION

	mechanism.mechanism = HBS_MECH_XMSS_VERIFY;

	if ((err = pfunc_list->C_VerifyInit(session_id, &mechanism, hKey)) != CKR_OK) {
		printf("[xmss_sign_all]: C_VerifyInit returned 0x%08x\n", err);
		goto cleanup;
	}

	if ((err = pfunc_list->C_Verify(session_id, p_msg, l_msg, sig, sig_len)) != CKR_OK) {
		printf("[xmss_sign_all]: C_Verify returned 0x%08x\n", err);
		goto cleanup;
	}

	printf("XMSS Signature verified.\n");
	printf("\n");

cleanup:
	return err;
}





//-------------------------------------------------------------
// EXAMPLE multi Data Block (4.1.7 HBS_MECH_XMSS_SIGN)

int xmss_sign_chunked(
	CK_OBJECT_HANDLE 		hKey,
	CK_BYTE_PTR 			p_msg,
	CK_ULONG 				l_msg)
{

	int 					err = 0;
	unsigned char 			params[] = { 0 };
	unsigned char 			sig[4096];
	CK_ULONG 				sig_len = (CK_ULONG) sizeof(sig);

	CK_MECHANISM 			mechanism;
	mechanism.mechanism = HBS_MECH_XMSS_SIGN;

	mechanism.pParameter = params;
	mechanism.ulParameterLen = 0;

	if ((err = pfunc_list->C_SignInit(session_id, &mechanism, hKey)) != CKR_OK) {
		printf("[xmss_sign_chunked]: C_SignInit returned 0x%08x\n", err);
		goto cleanup;
	}

	CK_ULONG sent = 0, chunk_size = 6;
	while (sent < l_msg) {
		CK_ULONG len = min(chunk_size, l_msg - sent);
		if ((err = pfunc_list->C_SignUpdate(session_id, p_msg + sent, len)) != CKR_OK) {
			printf("[xmss_sign_chunked]: C_SignUpdate returned 0x%08x\n", err);
			goto cleanup;
		}
		sent += len;
	}

	if ((err = pfunc_list->C_SignFinal(session_id, sig, &sig_len)) != CKR_OK) {
		printf("[xmss_sign_chunked]: C_SignFinal returned 0x%08x\n", err);
		goto cleanup;
	}

	if (prnout) {
		printf("Signature:\n");
		bin2hex(sig, sig_len);
		printf("\n");
	}

	// SIGNATURE VERIFICATION

	mechanism.mechanism = HBS_MECH_XMSS_VERIFY;
	mechanism.pParameter = sig;
	mechanism.ulParameterLen = sig_len;

	if ((err = pfunc_list->C_VerifyInit(session_id, &mechanism, hKey)) != CKR_OK) {
		printf("[xmss_sign_chunked]: C_VerifyInit returned 0x%08x\n", err);
		goto cleanup;
	}

	sent = 0;

	while (sent < l_msg) {
		CK_ULONG len = min(chunk_size, l_msg - sent);
		if ((err = pfunc_list->C_VerifyUpdate(session_id, p_msg + sent, len)) != CKR_OK) {
			printf("[xmss_sign_chunked]: C_VerifyUpdate returned 0x%08x\n", err);
			goto cleanup;
		}
		sent += len;
	}
	if ((err = pfunc_list->C_VerifyFinal(session_id, sig, sig_len)) != CKR_OK) {
		printf("[xmss_sign_chunked]: C_VerifyFinal returned 0x%08x\n", err);
		goto cleanup;
	}

	printf("XMSS Signature verified.\n");
	printf("\n");

cleanup:
	return err;
}


*/
