package com.utimaco.cs2.mdl;

import java.io.BufferedReader;
/* Modification History
 * ====================
 * i1b,20240401,riw     inf loop bug in convert to hex, probably never seen because nobody uses this but me
 * 01a,2021xxxx,riw/uti archeologic artifact
 */
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Base64;
import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import CryptoServerAPI.CryptoServerException;
import CryptoServerCXI.CryptoServerCXI;

public class Utils {
	/**
	 * The CryptoServer comms path is limited to 256Kb chunks.  If automatic
	 * chunking is not provided by the lower level interface, then you can
	 * use this method to generate a hash of a file for example, and then
	 * send that instead, if the process supports this option.
	 * 
	 * @param cxi - if !null, uses Java_CXI to generate hash, otherwise host system
	 * @param pathname - String, path to file to read
	 * @param hashname - String, which hash to use (SHA-256, SHA-512, SHA2-256, etc)
	 * @return byte [] - the hash bytes
	 * @throws IOException - if an I/O error occurs reading from the stream
	 */
	public static byte [] hash(CryptoServerCXI cxi, String pathname, String hashname) throws NoSuchAlgorithmException, IOException, CryptoServerException {
		byte[] array = readFileAsBytes(pathname);
		if (cxi != null) {
			int mech = CryptoServerCXI.MECH_MODE_HASH;
			switch (hashname) {
			case "SHA_384": mech |= CryptoServerCXI.MECH_HASH_ALGO_SHA384; break;
			case "SHA-256": mech |= CryptoServerCXI.MECH_HASH_ALGO_SHA256; break;
			case "SHA3-256": mech |= CryptoServerCXI.MECH_HASH_ALGO_SHA3_256 ; break;
			case "SHA3_384": mech |= CryptoServerCXI.MECH_HASH_ALGO_SHA3_384; break;
			case "SHA3-512": mech |= CryptoServerCXI.MECH_HASH_ALGO_SHA3_512; break;
			default:
				mech |= CryptoServerCXI.MECH_HASH_ALGO_SHA512; break;
			}
			byte [] hash = cxi.computeHash(0, mech, array, null, null);
			return hash;
		}
		MessageDigest md = MessageDigest.getInstance(hashname);
		md.update(array);
		return md.digest();
	}
	/**
	 * Reads file pointed to by argument into a new byte [].
	 * 
	 * @param pathname - String, path to file to read
	 * @return byte [] - contents of the file
	 * @throws IOException - if an I/O error occurs reading from the stream
	 */
	public static byte [] readFileAsBytes(String pathname) throws IOException {
		byte[] array = Files.readAllBytes(new File(pathname).toPath());
		return array;
	}

	/** 
	 * Writes String data to the file pointed to by pathname.  Warning: swallows all exceptions
	 * 
	 * @param pathname - String, path to file 
	 * @param data - String, data to write
	 * @return - true if file written, false if exception happened on write
	 */
	public static boolean writeFile(String pathname, String data) {
		try {
			File f = new File(pathname);
			f.createNewFile();
			Path p = Files.write(f.toPath(), data.getBytes());
			return (p != null);
		} catch (IOException e) {
			System.out.println("IOE: " + e.getLocalizedMessage());
		} catch (Exception e) {
			System.out.println("E: " + e.getLocalizedMessage());
		}
		return false;
	}

	/** 
	 * Writes String data to the file pointed to by pathname.  Warning: swallows all exceptions
	 * If the cdirs param is true, will create intermediate directory structure if possible.
	 * @param pathname - String, path to file 
	 * @param data - String, data to write
	 * @param cdirs - boolean, create directories if needed
	 * @return - true if file written, false if exception happened on write
	 */
	public static boolean writeFile(String pathname, String data, boolean cdirs) {
		if (cdirs) {
			try {
				File f = new File(pathname);
				if (f.getParent() != null) {
					File p = new File(f.getParent());
					p.mkdirs();
				}
			} catch (Exception e) {
				System.out.println("E: " + e.getMessage());
			}
		}
		return writeFile(pathname, data);
	}

	/**
	 * Writes byte data to the file pointed to by pathname.  Warning: Swallows all exceptions.
	 *
	 * @param pathname - String, path to file 
	 * @param data - binary data to write
	 * @param cdirs - boolean, create directories if needed
	 * @return - true if file written, false if exception happened on write
	 */
	public static boolean writeFile(String pathname, byte [] data, boolean cdirs) {
		if (cdirs) {
			try {
				File f = new File(pathname);
				if (f.getParent() != null) {
					File p = new File(f.getParent());
					p.mkdirs();
				}
			} catch (Exception e) {
				System.out.println("E: " + e.getMessage());
			}
		}
		FileOutputStream fos;
		try {
			fos = new FileOutputStream(pathname);
			fos.write(data);
			fos.close();
			return true;
		} catch (IOException e) {
			e.printStackTrace();
		}
		return false;
	}

	// BEWARE ENCODING ISSUES  Some of this may result in ISO_8859_1 instead of ascii
	
	/**
	 * Base64URL encode a byte [] -> byte []
	 * 
	 * @param in - byte [] of data
	 * @return - resulting base64URL encoded byte array
	 */
	public static byte [] b64URLencode(byte [] in) {
		return Base64.getUrlEncoder().encode(in);
	}
	
	/**
	 * Base64URL encode a byte [] -> ISO_8859_1 String
	 * @param in - byte [] of data
	 * @return - resulting base64URL encoded text string
	 */
	public static String b64URLencodeToString(byte[] in) {
		return Base64.getUrlEncoder().encodeToString(in);
	}
	
	/**
	 * Base64URL decode a byte [] -> byte []
	 * 
	 * @param in - Base64URL encoded String
	 * @return - resulting byte []
	 */
	public static byte [] b64URLdecode(String in) {
		return Base64.getUrlDecoder().decode(in);
	}
	/**
	 * Base64URL decode a ISO_8859_1 String -> 
	 * 
	 * @param in - Base64URL encoded String
	 * @return - resulting new String(byte [])
	 */
	public static String b64URLdecodeToString(String in) {
		return new String(b64URLdecode(in));
	}
	
	
	/**
	 * Base64 encode a byte [] -> byte []
	 * 
	 * @param in - byte [] of data
	 * @return - resulting base64 encoded byte array
	 */
	public static byte [] b64encode(byte [] in) {
		return Base64.getEncoder().encode(in);
	}

	/**
	 * Base64 encode a byte [] -> ISO_8859_1 String
	 * @param in - byte [] of data
	 * @return - resulting base64 encoded text string
	 */
	public static String b64encodeToString(byte [] in) {
		return Base64.getEncoder().encodeToString(in);
	}

	/**
	 * Base64 decode a byte [] -> byte []
	 * 
	 * @param in - Base64 encoded String
	 * @return - resulting byte []
	 */
	public static byte [] b64decode(String in) {
		return Base64.getDecoder().decode(in);
	}
	/**
	 * Base64 decode a ISO_8859_1 String -> 
	 * @param in
	 * @return
	 */
	public static String b64decodeToString(String in) {
		return new String(b64decode(in));
	}

    /**
     * Convert ascii hex like FF112244 to a byte array
     *
     * @param s
     * @return byte[]
     */
    public static byte[] convertToByteArray(String s) {
        int len = s.length();
        byte[] data = new byte[len / 2];
        for (int i = 0; i < len; i += 2) {
            data[i / 2] = (byte) ((Character.digit(s.charAt(i), 16) << 4)
                    + Character.digit(s.charAt(i+1), 16));
        }
        return data;
    }

    /**
     * Convert String to bytes, then hexify result, return as hex ascii string
     *
     * @param data - String
     * @return String (hex ascii)
     */
    public static String convertToHex(String data) { return convertToHex(data.getBytes(), true); }

    /**
     * Return data (byte array) as hex ascii string
     *
     * @param data - String
     * @return String (hex ascii)
     */
    public static String convertToHex(byte[] data) { return convertToHex(data, true); }

    // TODO: Rationalize this with that
    /**
     * Convert String to bytes, then hexify result, return as string (using upper case A-F if requested).
     *
     * @param data - String
     * @param uc - boolean, uppercase A-F if true
     * @return String (hex ascii)
     */
    public static String convertToHex(String data, boolean uc) {
        return convertToHex(data.getBytes(), uc);
    }

    /**
     * Return array as hex ascii string (using upper case A-F if requested).
     *
     * @param data - String
     * @param uc - boolean, uppercase A-F if true
     * @return String (hex ascii)
     */
    public static String convertToHex(byte[] data, boolean uc) {
        StringBuffer buf = new StringBuffer();
        for (int i = 0; i < data.length; i++) {
            String ab = Integer.toHexString(data[i] & 0xFF);
            buf.append(ab);
        }
    	String ret = buf.toString();
    	if (uc) {
    		ret = ret.replaceAll("a", "A")
    				.replaceAll("b", "B")
    				.replaceAll("c", "C")
    				.replaceAll("d", "D")
    				.replaceAll("e", "E")
    				.replaceAll("f", "F");
        }
    	return ret;
    }

    /** 
     * Test input string to see if it is ^[0-9a-fA-F]*$
     * 
     * @param in
     * @return boolean
     */
    public static boolean isHex(String in) {
    	Pattern p = Pattern.compile("^[0-9a-fA-F\n]*$");
    	Matcher m = p.matcher(in);
    	return m.matches();
    }

    /**
     * Loads a text file from com/utimaco, passes it through a rudimentary template 
     * engine and dumps it to stdout.
     * 
     * Template: if not null, the template parameter is expected to be a hash map of
     * key-value pairs.
     * 
     * The text file may include things like #THIS# or #THAT#, and the engine
     * will look for the keys 'THIS' and 'THAT' in the template map and replace
     * the full marker with the value found.  Does not support circular references.
     * If not found, replaces marker with 'undefined-template-reference'.
     * 
     * Given "THIS" -> this, and
     * 
     * #THIS# works, #THAT# doesn't
     * 
     * The result is 
     * 
     * this works, undefined-template-reference doesn't.
     * 
     * Markers must match #[_0-9A-Z]+#
     * 
     * @param filename
     */
	public static void printHelp(String filename, HashMap<String, String> template) {
		InputStream inputStream = null;
		InputStreamReader streamReader = null;
		Pattern p = Pattern.compile(".*#[_0-9A-Z]+#.*");
		
		try {
			ClassLoader.getSystemClassLoader();
			inputStream = ClassLoader.
					getSystemResourceAsStream("com/utimaco/" + filename);
			streamReader =  new InputStreamReader(inputStream, "UTF-8");
			BufferedReader in = new BufferedReader(streamReader);

			for (String line; (line = in.readLine()) != null;) {
				Matcher m = p.matcher(line);
				if (m.matches()) {
					String t = line;
					for (String k : template.keySet()) {
						t = t.replaceAll("#"+k.toUpperCase()+"#", template.get(k));
					}
					System.out.println(t);
				} else {
					System.out.println(line);
				}
			}

			inputStream.close();
			streamReader.close();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} finally {
		}

		System.exit(0);
	}
	
}
